# -*- coding:utf-8 -*-

import utils.common.log as logger
from plugins.CSDR_CSHA_VHA.common import CommonUtil
from plugins.CSDR_CSHA_VHA.common.AutoRetry import retry
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_all_server_nodes
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.ServerProcessor import ServerProcessor
from plugins.CSDR_CSHA_VHA.common.Validater import judge_rollback_step_result
from utils.business.param_util import ParamUtil
from utils.common.exception import FCUException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

logger.init("eReplication")


class RollbackBackendService(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        super(RollbackBackendService, self).__init__(
            project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region = regionid_list[0]
        self.service_type = get_service_type(self.project_id)

        params = ParamUtil()
        server_params = CommonUtil.get_server_params(
            self.project_id, self.region, self.service_type)
        self.server_ip1 = server_params["server_ip1"]
        self.server_business_user = server_params["server_business_user"]
        self.server_business_user_pwd = server_params[
            "server_business_user_pwd"]
        self.server_root_pwd = server_params["server_root_pwd"]
        self.encrypt_pwd = params.get_value_from_cloud_param(
            self.project_id, self.service_type,
            "eReplication_data_encrypt_pwd", self.region)

    def pre_check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    @retry(3, 20, 20, (FCUException, Exception),
           validate=judge_rollback_step_result)
    def execute(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行安装&配置

        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """

        try:
            logger.info("Start restore config data.")
            if not self.server_ip1:
                logger.error("Get server ip failed.")
                raise FCUException("665008")
            all_ips = get_all_server_nodes(
                self.server_ip1, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            if len(all_ips) not in [2, 4] or self.server_ip1 not in all_ips:
                logger.error(
                    f"Query eReplication ip from {self.server_ip1} failed.")
                raise FCUException("665006")

            for ip in all_ips:
                processor = ServerProcessor(
                    ip, self.server_business_user,
                    self.server_business_user_pwd, self.server_root_pwd)
                processor.rollback_data(self.encrypt_pwd, project_id)
                processor.rollback_patch(project_id)
            logger.info("Restore config data successfully.")
        except FCUException as fe:
            logger.error(f"Restore data failed, err_msg={str(fe)}.")
            return Message(500, fe)
        except Exception as e:
            logger.error(f"Restore data failed, err_msg={str(e)}.")
            return Message(500, FCUException('665002', str(e)))
        return Message(200)

    def rollback(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行回滚

        :return: Message对象
        """

        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """标准调用接口：重试

        :return:
        """

        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行注册，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)
