# -*- coding:utf-8 -*-
import time

import utils.common.log as logger
from plugins.CSDR_CSHA_VHA.common.AutoRetry import retry
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_all_server_nodes
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_server_params
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.ServerProcessor import ExcThread
from plugins.CSDR_CSHA_VHA.common.ServerProcessor import ServerProcessor
from plugins.CSDR_CSHA_VHA.common.ServerProcessor import UpgradeProcess
from plugins.CSDR_CSHA_VHA.common.Validater import judge_upgrade_step_result
from utils.common.exception import FCUException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

logger.init("eReplication")


class InstallPatch(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        super(InstallPatch, self).__init__(
            project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region = regionid_list[0]
        self.service_type = get_service_type(self.project_id)

        server_params = get_server_params(
            self.project_id, self.region, self.service_type)
        self.server_ip1 = server_params["server_ip1"]
        self.server_business_user = server_params["server_business_user"]
        self.server_business_user_pwd = server_params[
            "server_business_user_pwd"]
        self.server_root_pwd = server_params["server_root_pwd"]

    def pre_check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    @retry(3, 20, 20, (FCUException, Exception),
           validate=judge_upgrade_step_result)
    def execute(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行安装&配置

        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """

        try:
            logger.info("Install Server patch process.")
            if not self.server_ip1:
                logger.error("Get server ip failed.")
                raise FCUException("665008")
            all_ips = get_all_server_nodes(
                self.server_ip1, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            if len(all_ips) not in [2, 4] or self.server_ip1 not in all_ips:
                logger.error(
                    f"Query eReplication ip from {self.server_ip1} failed.")
                raise FCUException("665006")
            # 第一步：安装补丁需要执行SQL，需要先启动服务初始化数据库表
            u_process = UpgradeProcess(
                self.pod_id, self.project_id, self.region, self.service_type)
            u_process.start_service(all_ips)
            time.sleep(60)
            # 第二步：启动后先执行上传补丁包操作，预留充足的时间做数据同步，防止备节点数据库表还未初始化就停止服务
            self._upload_patch(all_ips)
            # 第三步：停止服务
            self._stop_service(all_ips)
            # 第四步：安装补丁
            self._install_patch(all_ips)
            logger.info("Install Server patch finish.")
            return Message(200)
        except FCUException as fe:
            logger.error(f"Install Server patch failed: {fe}.")
            return Message(500, fe)
        except Exception as e:
            logger.error(f"Install Server patch failed: {e}.")
            return Message(500, FCUException('665002', str(e)))

    def rollback(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行回滚

        :return: Message对象
        """

        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """标准调用接口：重试

        :return:
        """

        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行注册，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    def _upload_patch(self, all_ips):
        """
        上传补丁包到Server节点
        :return:
        """
        logger.info("Start upload Server patch.")
        upload_threads = list()
        for host in all_ips:
            processor = ServerProcessor(
                host, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            upload_thread = ExcThread(processor.upload_patch)
            upload_threads.append(upload_thread)
        for thread in upload_threads:
            thread.start()
        for thread in upload_threads:
            thread.join()
            if thread.exit_code != 0:
                logger.error(
                    f"Upload patch thread return {thread.exc_traceback}")
                raise Exception(thread.exception)

    def _install_patch(self, all_ips):
        """
        登录Server节点执行安装补丁包操作
        :return:
        """
        logger.info("Start install Server patch.")
        install_threads = list()
        for host in all_ips:
            processor = ServerProcessor(
                host, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            install_thread = ExcThread(processor.install_patch)
            install_threads.append(install_thread)
        for thread in install_threads:
            thread.start()
        for thread in install_threads:
            thread.join()
            if thread.exit_code != 0:
                logger.error(
                    f"Install patch thread return {thread.exc_traceback}")
                raise Exception(thread.exception)

    def _stop_service(self, all_ips):
        """
        停止服务
        :return:
        """
        logger.info("Stop BCM service.")
        stop_threads = list()
        for host in all_ips:
            processor = ServerProcessor(
                host, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            stop_thread = ExcThread(processor.stop_service)
            stop_threads.append(stop_thread)
        # 停止所有节点的服务
        for thread in stop_threads:
            thread.start()
        for thread in stop_threads:
            thread.join()
            if thread.exit_code != 0:
                logger.error(
                    f"Stop service thread return {thread.exc_traceback}")
                raise Exception(thread.exception)
