# -*- coding:utf-8 -*-
import utils.common.log as logger
from plugins.CSDR_CSHA_VHA.common.AutoRetry import retry
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_all_server_nodes
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_ha_all_server_nodes
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_server_node_region_id
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_server_params
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.ServerProcessor import UpgradeProcess
from plugins.CSDR_CSHA_VHA.common.Validater import judge_upgrade_step_result
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.common.exception import FCUException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

logger.init("eReplication")


class UpgradeBackService(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list):
        super(UpgradeBackService, self).__init__(
            project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region = regionid_list[0]
        self.service_type = get_service_type(self.project_id)

        server_params = get_server_params(
            self.project_id, self.region, self.service_type)
        self.server_ip1 = server_params["server_ip1"]
        self.server_business_user = server_params["server_business_user"]
        self.server_business_user_pwd = server_params[
            "server_business_user_pwd"]
        self.server_root_pwd = server_params["server_root_pwd"]
        self.mo_cmdb_ins = ManageOneCmdbUtil(self.project_id)

    def pre_check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    @retry(3, 20, 20, (FCUException, Exception),
           validate=judge_upgrade_step_result)
    def execute(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行安装&配置

        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """

        try:
            logger.info("Upgrade eReplication service process.")
            if not self.server_ip1:
                logger.error("Get server ip failed.")
                raise FCUException("665008")
            primary_region_id = ""
            standby_region_id = ""
            # 获取一主三备场景下所有的server ip
            all_ips = get_all_server_nodes(
                self.server_ip1, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            if len(all_ips) not in [2, 4] or self.server_ip1 not in all_ips:
                logger.error(
                    f"Query eReplication ip from {self.server_ip1} failed.")
                raise FCUException("665006")
            # 获取当前region所有的server ip
            current_region_ips = get_ha_all_server_nodes(
                self.server_ip1, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            if len(current_region_ips) < 1 or len(current_region_ips) > 2 \
                    or self.server_ip1 not in current_region_ips:
                logger.error(
                    f"Query eReplication ip from {self.server_ip1} failed.")
                raise FCUException("665006")

            # 区分主region ip和备region ip
            process = UpgradeProcess(
                self.pod_id, self.project_id, self.region, self.service_type)
            if set(all_ips) == set(current_region_ips):
                process.upgrade_service(all_ips, project_id, self.region)
            else:
                primary_ips = current_region_ips
                for primary_ip in primary_ips:
                    all_ips.remove(primary_ip)
                standby_ips = all_ips
                # 获取region id
                for primary_ip in primary_ips:
                    primary_region_id = get_server_node_region_id(
                        primary_ip, self.server_business_user,
                        self.server_business_user_pwd, self.server_root_pwd)
                    if primary_region_id:
                        logger.info(
                            f"Get primary region result: "
                            f"{primary_region_id}.")
                        break
                for standby_ip in standby_ips:
                    standby_region_id = get_server_node_region_id(
                        standby_ip, self.server_business_user,
                        self.server_business_user_pwd, self.server_root_pwd)
                    if standby_region_id:
                        logger.info(
                            f"Get standby region result: "
                            f"{standby_region_id}.")
                        break
                process.upgrade_service(
                    primary_ips, project_id, primary_region_id)
                process.upgrade_service(
                    standby_ips, project_id, standby_region_id)

                # 重新配置一主三备关系
                primary_ips.extend(standby_ips)
                all_ips = primary_ips
                region_ids = f"{primary_region_id},{standby_region_id}"
                process.config_one_a_three_s(
                    all_ips, project_id=self.project_id,
                    region_ids=region_ids)
                logger.info(
                    "Upgrade eReplication successfully in all nodes.")
            return Message(200)
        except FCUException as fe:
            logger.error(
                f"Upgrade eReplication failed, err_msg={str(fe)}.")
            return Message(500, fe)
        except Exception as e:
            logger.error(
                f"Upgrade eReplication failed, err_msg={str(e)}.")
            return Message(500, FCUException('665002', str(e)))

    def rollback(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行回滚

        :return: Message对象
        """

        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """标准调用接口：重试

        :return:
        """

        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行注册，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)
