#!/usr/bin/csbs_python
# pylint: disable=missing-docstring
# - encoding:utf-8 -
import sys
import netifaces
import configparser

CONF_FILE_PATH = "/opt/huawei/dj/cfg/sys.ini"
DB_CA_FILE = "/opt/huawei/dj/DJSecurity/server-ca/ca-cert.pem"
NOT_SYSTEM = ["hostname", "tenant_ip", "internal_ip", "manage_ip"]
FEATURE = ["cms_info_flag", "cluster_first_node", "public_zookeeper_nodes",
           "public_zookeeper_password"]
parser = configparser.RawConfigParser()
parser.read(CONF_FILE_PATH)


def get_local_ip():
    try:
        ip_version = system_config("ip_version")
        net = netifaces.AF_INET6 if ip_version == "ipv6" else netifaces.AF_INET
        interface = netifaces.gateways()['default'][net][1]
        return netifaces.ifaddresses(interface)[net][0]['addr']
    except Exception:
        return None


def get_default_gateway(*kargs):
    try:
        ip_version = system_config("ip_version")
        net = netifaces.AF_INET6 if ip_version == "ipv6" else netifaces.AF_INET
        return netifaces.gateways()['default'][net][0]
    except Exception:
        return None


def system_config(item):
    system_opt = parser.options("SYSTEM")
    if item not in system_opt:
        return None
    return parser.get("SYSTEM", item)


def feature_config(item):
    feature_opt = parser.options("FEATURE")
    if item not in feature_opt:
        return None
    return parser.get("FEATURE", item)


def get_local_info(key):
    manage_ip = get_local_ip()
    if key == "manage_ip":
        return manage_ip
    manage_opt = parser.options(manage_ip)
    return parser.get(manage_ip, key) if key in manage_opt else None


def get_all_local_info(key):
    s = get_node(True)
    if key == "manage_ip":
        return s
    r = [parser.get(m, key) for m in s.split(',') if key in parser.options(m)]
    return ",".join(r)


def other_config(key, show_all=False):
    if not show_all:
        return get_local_info(key)
    return get_all_local_info(key)


def get_node(show_all=False):
    section = parser.sections()
    manage_ip = section
    manage_ip.remove("SYSTEM")
    manage_ip.remove("FEATURE")
    if system_config("profile") == "private_recover":
        manage_ip.remove(system_config("gaussdb_float_ip"))
    return ",".join(manage_ip) if show_all else len(manage_ip)


def get_node_index(*kargs):
    ip_list = get_node(True).split(',')
    local_manage_ip = get_local_ip()
    if local_manage_ip:
        return ip_list.index(local_manage_ip) + 1
    return None


def get_zookeeper_nodes(*kargs):
    pub_zk_nodes = feature_config("public_zookeeper_nodes")
    if pub_zk_nodes and pub_zk_nodes != 'None':
        return pub_zk_nodes
    manage_ip = get_node(True).split(',')
    return ",".join(manage_ip)


def get_gaussdb_nodes(*kargs):
    nodes = feature_config("gaussdb_nodes")
    return nodes if nodes else system_config("gaussdb_nodes")


def get_ommha_nodes(*kargs):
    nodes = feature_config("omm-ha_nodes")
    return nodes if nodes else system_config("omm-ha_nodes")


def get_ha_peer_node(*kargs):
    ha_nodes = get_ommha_nodes().split(',')
    local_ip = get_local_ip()
    for node in ha_nodes:
        if node != local_ip and local_ip in ha_nodes:
            return node
    return None


def get_db_peer_node(*kargs):
    db_nodes = get_gaussdb_nodes().split(',')
    local_ip = get_local_ip()
    for node in db_nodes:
        if node != local_ip and local_ip in db_nodes:
            return node
    return None


def cluster_first_node(*kargs):
    node = feature_config("cluster_first_node")
    if not node or node in get_node(True).split(','):
        return get_node(True).split(',')[0]
    return node


def get_db_ca_file(*kwargs):
    db_ca_file = system_config("db_ca_file")
    if db_ca_file and db_ca_file != 'None':
        return db_ca_file
    else:
        return DB_CA_FILE


function_map = {
    "node": get_node,
    "node_index": get_node_index,
    "zookeeper_nodes": get_zookeeper_nodes,
    "gaussdb_nodes": get_gaussdb_nodes,
    "omm-ha_nodes": get_ommha_nodes,
    "ha_peer_node": get_ha_peer_node,
    "db_peer_node": get_db_peer_node,
    "default_gateway": get_default_gateway,
    "cluster_first_node": cluster_first_node,
    "db_ca_file": get_db_ca_file,
}


def get_parameters_value(item, show_all=False):
    if item in function_map.keys():
        return function_map[item](show_all)
    if item in FEATURE:
        return feature_config(item)
    if item not in NOT_SYSTEM:
        return system_config(item)
    return other_config(item, show_all)


def main(argv):
    if (len(argv) != 2 and len(argv) != 1) \
            or (len(argv) == 2 and argv[1] != "list"):
        print("Parameter inputed error!")
        return 1
    param1 = argv[0].lstrip("-")
    print(get_parameters_value(param1.lower(), len(argv) == 2))
    return 0


if __name__ == "__main__":
    code = main(sys.argv[1:])
    exit(code)
