#!/bin/bash
cur_dir=$(dirname $(readlink -f $0))
LOG_FILE="/var/log/huawei/dj/install.log"
source $(dirname ${cur_dir})//utils.sh

if [  -e /etc/SuSE-release ];then
    OS_TYPE="suse"
elif [  -e /etc/euleros-release ];then
    OS_TYPE="euler"
fi

function check_firewall_status()
{
    if [ "$OS_TYPE" == "suse" ];then
        rcSuSEfirewall2 status
        if [ $? -eq 0 ];then
            log_error "firewall is opened,should be close the firewall before install CSBS"
            return 1
        fi
    fi
    if [ "$OS_TYPE" == "euler" ];then
        systemctl status firewalld | grep -w active
        if [ $? -eq 0 ];then
            log_error "firewall is opened,should be close the firewall before install CSBS"
            return 1
        fi
    fi
    log_info "Check firewall status OK."
    return 0
}

function check_os_version()
{
    if [ "$OS_TYPE" == "suse" ];then
        hwPlatform=`uname -i`
        osVersion=`cat /etc/SuSE-release | grep VERSION | awk '{print $3}'`
        osPatch=`cat /etc/SuSE-release | grep PATCHLEVEL | awk '{print $3}'`
        if [[ "${hwPlatform}" != "x86_64" || "${osVersion}" != "12" || "${osPatch}" != "2" ]];then
            log_error "Not supported operation system. Only supported SUSE Linux Enterprise Server 12 (x86_64) SP2."
            return 1
        fi
    fi
    if [ "$OS_TYPE" == "euler" ];then
        hwPlatform=`uname -i`
        osVersion=`cat /etc/euleros-release | grep EulerOS | awk '{print $3}'`
        osPatch=`cat /etc/euleros-release | grep EulerOS | awk '{print $4}'`
        if [[ "${hwPlatform}" == "x86_64" && "${osVersion}" == "2.0" ]] || [[ "${hwPlatform}" == "aarch64" && "${osVersion}" == "2.0" ]];then
            log_info "Check required os version OK."
        else
            log_error "Not supported operation system. Only supported Euler 2.0 (x86_64)."
            return 1
        fi
    fi
    log_info "Check required os version OK."
    return 0
}

function check_docker_status()
{
    daemon_num=`ps -ef|grep /usr/bin/docker|grep daemon|grep -v grep`
    if [ ${daemon_num} -le 0 ];then
        log_error "Docker engine is not running."
        return 1
    fi
    log_info "Check required docker version OK."
    return 0
}

function check_dir_exist()
{
    REQ_PHY_DIR="/ /opt"
    for dirItem in ${REQ_PHY_DIR}
    do
        dirExt=`df -h |awk '{print $6}' |grep "${dirItem}$"`
        if [ ! $dirExt ];then
            log_error "($dirItem) dist don't exist..."
            return 1
        fi
    done
    return 0
}

function check_free_disk_space()
{
    REQ_PHY_DIR_IN_KB="/=5242880 /opt=62914560"
    for chkPair in ${REQ_PHY_DIR_IN_KB}
    do
        chkDir=`echo "${chkPair}" | cut -d '=' -f 1`
        requiredSpace=`echo "${chkPair}" | cut -d '=' -f 2`
        freeSpace=`df -k ${chkDir} | tail -n -1 | awk '{print $4}'`
        ret=`df | grep "${chkDir}$"`
        if [ $? -eq 0 ];then
            if [ "${freeSpace}" -lt "${requiredSpace}" ];then
                log_warn "Current free disk space (${freeSpace}KB) for \"${chkDir}\" is insufficient. It should not be less than ${requiredSpace}KB."
            fi
        else
            log_error "($chkDir) dist don't exist..."
            return 1
        fi
    done
   log_info "Check required free disk space OK."
   return 0
}

function check_cpu_count()
{
    REQ_CPU_CORE_NUM=4 #4U
    cpuCnt=`cat /proc/cpuinfo | grep processor | wc -l`
    if [ "${cpuCnt}" -lt "${REQ_CPU_CORE_NUM}" ];then
        log_error "The number of processors should not less than ${REQ_CPU_CORE_NUM}."
        return 1
    fi
    log_info "Check required cpu count OK."
    return 0
}

function check_memory()
{
    REQ_PHY_MEM_IN_KB=7340032 #8G
    totalMem=`cat /proc/meminfo | grep MemTotal | awk '{print $2}'`
    if [ "${totalMem}" -lt "${REQ_PHY_MEM_IN_KB}" ];then
        log_error "System total memory (${totalMem}KB) should not less than ${REQ_PHY_MEM_IN_KB}KB."
        return 1
    fi
    log_info "Check required system memory OK."
    return 0
}

function check_share_memory()
{
    REQ_SHR_MEM_SEG_IN_KB=2194304 #2G
    maxSharedMemSeg=`cat /proc/sys/kernel/shmmax`
    maxSharedMemSeg=`echo ${maxSharedMemSeg}/1024 | bc`
    if [ "${maxSharedMemSeg}" -lt "${REQ_SHR_MEM_SEG_IN_KB}" ];then
        log_error "Max size of a shared memory segment (${maxSharedMemSeg}KB) should not be less than ${REQ_SHR_MEM_SEG_IN_KB}KB."
        return 1
    fi
    log_info "Check required share memory OK."
    return 0
}


function check_umask()
{
    ret_var=0
    umask_val=`umask`
    if [ $umask_val -gt '0027' ];then
        log_error "umask val($umask_val) not meet the requirement:0027."
        return 1
    fi
    log_info "Successfully checked umask."
    return 0
}

check_firewall_status ; CHECK_RESULT $? "Check firewall status ！"
check_os_version ; CHECK_RESULT $? "Check os version ！"
check_docker_status ; CHECK_RESULT $? "Check docker status failed！"
check_dir_exist ; CHECK_RESULT $? "Check dir exist failed！"
check_free_disk_space ; CHECK_RESULT $? "Check free disk space failed！"
check_cpu_count ; CHECK_RESULT $? "Check cpu count  failed！"
check_memory ; CHECK_RESULT $? "Check memory  failed！"
check_share_memory ; CHECK_RESULT $? "Check share memory failed！"
check_umask ; CHECK_RESULT $? "Check umask failed！"

exit 0







