import oc
import fcntl
import cms_util
import functools


class Task(object):
    def execute(self, cms_config, data):
        pass


class OC23Task(Task):
    def execute(self, cms_config, data):
        oc_token = get_token(cms_config)
        try:
            oc.send_alarm_to_23(cms_config.endpoint, data, oc_token)
        except Exception:
            cms_util.expire_token(oc_token)
            raise


class OC30Task(Task):
    def execute(self, cms_config, data):
        oc_token = get_token(cms_config)
        try:
            oc.send_alarm_to_30(cms_config.endpoint, data, oc_token)
        except Exception:
            cms_util.expire_token(oc_token)
            raise


class OC60Task(Task):
    def execute(self, cms_config, data):
        oc_token = get_token(cms_config, True)
        try:
            oc.send_alarm_to_60(cms_config.endpoint,
                                data,
                                oc_token,
                                cms_util.get_manage_ip())
        except Exception:
            cms_util.expire_token(oc_token)
            raise


def flock(file_path, lock_stat):
    """文件锁，可用于进程间互斥"""
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            with open(file_path) as file:
                fcntl.flock(file.fileno(), lock_stat)
                ret = func(*args, **kwargs)
                fcntl.flock(file.fileno(), fcntl.LOCK_UN)
                return ret
        return wrapper
    return decorator


@flock("/opt/omm/oma/tools/lock/get_token.lock", fcntl.LOCK_EX)
def get_token(cms_config, use_new_token_interface=False):
    oc_token = cms_util.get_oc_token_from_cms()
    if oc_token is None:
        if use_new_token_interface:
            oc_token = oc.get_token_from_60(cms_config.endpoint,
                                            cms_config.username,
                                            cms_config.password,
                                            cms_util.get_manage_ip())
        else:
            oc_token = oc.get_token(cms_config.endpoint,
                                    cms_config.username,
                                    cms_config.password,
                                    cms_util.get_manage_ip())
        cms_util.put_oc_token_to_cms(oc_token)
    return oc_token


def dispatch_task(alarm_type):
    task = None
    if alarm_type == "oc2.3":
        task = OC23Task()
    elif alarm_type == "oc3.0":
        task = OC30Task()
    elif alarm_type == "oc6.0":
        task = OC60Task()
    else:
        pass

    if task is None:
        raise Exception("alarm_type %d not support" % alarm_type)
    return task
