#!/bin/bash
if [ ${UID} -ne 0 ];then
  sudo $(dirname $(readlink -f $0))/$(basename $(readlink -f $0)) $@;exit $?
fi
source /opt/huawei/dj/inst/bin/utils.sh
SSH_KEY_OUT_FILE="/opt/huawei/dj/DJSecurity/ssh/kmc_update_master_key"
SSH_OPT="-o StrictHostKeyChecking=no -o ConnectTimeout=5 -i ${SSH_KEY_OUT_FILE}"
KMC_MK_FILE="/opt/huawei/dj/paas/kmc_master_key"
BIN_DIR=/opt/huawei/dj/bin/kmc
KEY_ID=0

function log_error()
{
    logger -id -n localhost -p local1.error -t update_kmc_key "$1"
}

function log_info()
{
    logger -id -n localhost -p local1.info -t update_kmc_key "$1"
}

function safe_del()
{
    local file_name=$1

    if [[ "${file_name%*/}" != `realpath -m "${file_name}"` ]]; then
        exit 1
    fi

    [[ ! -f ${file_name} ]] && return
    #安全删除重复执行三次
    shred -z ${file_name}
    shred -z ${file_name}
    shred -uzf ${file_name}
}

function exit_cleanup()
{
    safe_del ${KMC_MK_FILE}
    safe_del ${SSH_KEY_OUT_FILE}
    if [[ $1 -ne 0 ]];then
        echo "[ERROR] Update KMC master key failed."
    fi
    exit $1
}

function generate_mk()
{
    KEY_ID=$(csbs_python -c "from kmc import kmc;print(kmc.API().create_mk(0))")
    if [[ $? -ne 0 ]];then
        log_error "Acticve KMC master key failed."
        exit_cleanup 1
    fi
    csbs_python -c "from kmc import kmc;kmc.API().export_mk_file(\"${KMC_MK_FILE}\", \"${KEY_ID}\")"
    if [[ $? -ne 0 ]];then
        log_error "Export KMC master key failed."
        exit_cleanup 1
    fi
    echo "[INFO] Generate KMC master key success."
    log_info "Generate KMC master key success."
}

function import_master_key()
{
    log_info "Begin import KMC master key."
    if [[ ! -f ${KMC_MK_FILE} ]];then
        log_error "Not found key ${KMC_MK_FILE} when import KMC master key."
        exit_cleanup 1
    fi
    csbs_python -c "from kmc import kmc;kmc.API().import_mk_file(\"${KMC_MK_FILE}\", \"${KEY_ID}\")"
    if [[ $? -ne 0 ]];then
        log_error "Import KMC master key file failed."
        exit_cleanup 1
    fi
    csbs_python -c "from kmc import kmc;kmc.API().active_mk(0, \"${KEY_ID}\")"
    if [[ $? -ne 0 ]];then
        log_error "Active KMC master key file failed."
        exit_cleanup 1
    fi
    log_info "Import KMC master key success."
}



function sync_file()
{
    local node_ip_list=`get_info.py --manage_ip list|tr ',' ' '`
    local local_ip=`get_info.py --manage_ip`
    local profile=`get_info.py profile`
    local count=1
    gen_ssh_key
    for node_ip in ${node_ip_list}
    do
        if [[ ${local_ip} == ${node_ip} ]]; then
            continue
        else
            scp ${SSH_OPT} ${KMC_MK_FILE} djmanager@[${node_ip}]:/home/djmanager
            if [[ $? -ne 0 ]];then
                log_error "Scp master failed when import KMC master key to ${node_ip}, please ensure the network is normal and the password is correct."
                exit_cleanup 1
            fi
            ssh ${SSH_OPT} djmanager@${node_ip} ${BIN_DIR}/kmc_update_master_key.sh -c ${KEY_ID}
            if [[ $? -ne 0 ]];then
                log_error "Import KMC master key to ${node_ip} failed, please ensure the network is normal and the password is correct."
                exit_cleanup 1
            fi
        fi
        echo "[INFO] Sync KMC master key success on node ${node_ip}."
        log_info "Sync KMC master key to ${node_ip} success."
    done
    csbs_python -c "from kmc import kmc;kmc.API().active_mk(0, \"${KEY_ID}\")"
    if [[ $? -ne 0 ]];then
        log_error "Active KMC master key file failed."
        exit_cleanup 1
    fi
    echo "[INFO] Update KMC master key success."
    log_info "Update KMC master key success."
}

function check() {
    local check_regex="^[0-9a-zA-Z]+$"
    if [[ ! "$1" =~ ${check_regex} ]];then
        log_error "Invalid parameter."
        echo "Invalid parameter."
        exit 1
    fi
}

function main()
{
    if [ $# -eq 2 -a "$1" = '-c' ]; then
        check $2
        KEY_ID=$2
        KMC_MK_FILE="/home/djmanager/kmc_master_key"
        import_master_key
    else
        generate_mk
        sync_file
    fi
    exit_cleanup 0
}

main $@
