#!/usr/bin/csbs_python
# -*- encoding:utf-8 -*-
import os
import six
import sys
import traceback
import subprocess
import shutil
from six.moves import configparser
from kmc import kmc

VALUE_CIPHER_TYPE_GENERAL = 'generalCipher'
VALUE_CIPHER_TYPE_SM = 'SMCompatible'

PRIVATE_CONF = '/opt/huawei/dj/DJSecurity/privkey/privkey.conf'

SYS_CIPHER_TYPE = {
    "config_file": '/opt/huawei/dj/cfg/sys.ini',
    "config_sections": [{
        "section_name": "SYSTEM",
        "config_items": [{"name": "cipher_type"}]
    }]
}

SYS_INI = {
    "config_file": '/opt/huawei/dj/cfg/sys.ini',
    "config_sections": [{
        "section_name": "SYSTEM",
        "config_items": [{"name": "default_user_pwd"},
                         {"name": "rabbitmq_pwd"},
                         {"name": "gaussdb_admin_pwd"}]
    }]
}

ZK_CLIENT = {
    "config_file": '/opt/fusionplatform/data/openstack/zkClient.conf',
    "config_sections": [{
        "section_name": "zookeeper_auth",
        "config_items": [{"name": "data_kmc"}]
        }, {
        "section_name": "hmac_auth",
        "config_items": [{"name": "data_kmc"}]
        }]
}


class ConfManager(object):
    def __init__(self, o_cipher_type, new_cipher_type):
        self.o_cipher_type = o_cipher_type
        self.new_cipher_type = new_cipher_type
        self.kmc_api = kmc.API()

    def _backup(self, config_file):
        """Backup files
        :param config_file:
        :return:
        """
        file_bak = '{}{}'.format(config_file, '_backup')
        self._delete_file(file_bak)
        shutil.copy2(config_file, file_bak)

    def _revert_change(self, config_file):
        """Revert change when failed"""
        file_bak = '{}{}'.format(config_file, '_backup')
        os.remove(config_file)
        shutil.copy2(file_bak, config_file)
        self._delete_file(file_bak)

    def _delete_backups(self, config_file):
        """Delete backup files when success
        :param config_file:
        :return:
        """
        file_bak = '{}{}'.format(config_file, '_backup')
        self._delete_file(file_bak)

    @staticmethod
    def _delete_file(file):
        """Delete file
        :param file:
        :return:
        """
        if os.path.exists(file):
            os.remove(file)

    def do_changes(self, config_file, config_sections, force_add=False):
        """Do changes
        :param config_file:
        :param config_sections:
        :param force_add: if set true, then the section will to add to conf
        even it's not exist.
        :return:
        """
        try:
            self._backup(config_file)
            self._do_changes(config_file, config_sections, force_add=force_add)
            self._delete_backups(config_file)
        except Exception as e:
            self._revert_change(config_file)
            raise e

    def _do_changes(self, conf_file, config_sections, force_add=False):
        """Do change job
        :param conf_file:
        :param config_sections:
        :return:
        """
        cf = configparser.ConfigParser()
        cf.read(conf_file)
        for config_section in config_sections:
            config_items = config_section['config_items']
            for config_item in config_items:
                if cf.has_option(config_section['section_name'],
                                 config_item['name']):
                    cf.set(config_section['section_name'],
                           config_item['name'],
                           six.text_type(config_item['value']))
                elif force_add:
                    cf.set(config_section['section_name'],
                           config_item['name'],
                           six.text_type(config_item['value']))
        with open(conf_file, 'w') as f:
            cf.write(f)

    def update_conf_cipher(self):
        """Update conf file
        :return:
        """
        for config_context in [SYS_INI, ZK_CLIENT]:
            config_file = config_context['config_file']
            cf = configparser.ConfigParser()
            cf.read(config_file)
            config_sections = config_context['config_sections']
            for section in config_sections:
                items = section['config_items']
                for item in items:
                    value = cf.get(section['section_name'], item['name'])
                    dep_value = self.kmc_api.decrypt(
                        kmc.KMC_DOMAIN.DEFAULT, value,
                        cipher_type=self.o_cipher_type)
                    item['value'] = self.kmc_api.encrypt(
                        kmc.KMC_DOMAIN.DEFAULT, dep_value,
                        cipher_type=self.new_cipher_type)

            self.do_changes(config_file, config_sections)

    def update_private_conf(self):
        """Update privkey.conf
        :return:
        """
        with open(PRIVATE_CONF, 'r') as f:
            txt = f.read()
        dep_value = self.kmc_api.decrypt(kmc.KMC_DOMAIN.DEFAULT,
                                         txt.split()[1],
                                         cipher_type=self.o_cipher_type)
        en_value = self.kmc_api.encrypt(kmc.KMC_DOMAIN.DEFAULT,
                                        dep_value,
                                        cipher_type=self.new_cipher_type)
        new_txt = ' '.join(['encrypt_password', en_value])
        with open(PRIVATE_CONF, 'w') as f:
            f.write(new_txt)

    @staticmethod
    def exec_cmd(cmd, raise_ex=True):
        """
        :param cmd:
        :param raise_ex:
        :return:
        """
        ret = subprocess.Popen(cmd, shell=False, stdout=subprocess.PIPE,
                               stdin=subprocess.PIPE, stderr=subprocess.PIPE)
        ret.wait(300)
        if ret.returncode != 0 and raise_ex:
            msg = 'exec cmd {} failed, ret = {}'.format(' '.join(cmd),
                                                        ret.returncode)
            raise Exception(msg)

    def update_id_rsa(self):
        """Update id_rsa
        :return:
        """
        self.exec_cmd(['rm', '-f', '/home/djmanager/.ssh/id_rsa.tmp'])
        self.kmc_api.file_decrypt(kmc.KMC_DOMAIN.DEFAULT,
                                  '/home/djmanager/.ssh/id_rsa',
                                  '/home/djmanager/.ssh/id_rsa.tmp',
                                  cipher_type=self.o_cipher_type)
        if not os.path.exists('/home/djmanager/.ssh/id_rsa.tmp'):
            msg = 'Decrypt ssh file failed.'
            raise Exception(msg)

        self.exec_cmd(['rm', '-f', '/home/djmanager/.ssh/id_rsa'])

        self.kmc_api.file_encrypt(kmc.KMC_DOMAIN.DEFAULT,
                                  '/home/djmanager/.ssh/id_rsa.tmp',
                                  '/home/djmanager/.ssh/id_rsa',
                                  cipher_type=self.new_cipher_type)
        if not os.path.exists('/home/djmanager/.ssh/id_rsa'):
            msg = 'Encrypt ssh file failed.'
            raise Exception(msg)

        self.exec_cmd(['chown', 'djmanager:openstack',
                       '/home/djmanager/.ssh/id_rsa'])

        self.exec_cmd(['rm', '-f', '/home/djmanager/.ssh/id_rsa.tmp'],
                      raise_ex=False)

    def update_sys_cipher(self):
        """Update sys file cipher type
        :return:
        """
        config_file = SYS_CIPHER_TYPE['config_file']
        config_sections = SYS_CIPHER_TYPE['config_sections']
        items = config_sections[0]['config_items']
        items[0]['value'] = self.new_cipher_type
        self.do_changes(config_file, config_sections, force_add=True)
        # switch cipher type, update kmc root key
        self.kmc_api.update_root_key()


if __name__ == "__main__":
    try:
        if len(sys.argv) < 3:
            raise Exception('Para invalid, at least 3 params.')

        method = sys.argv[1]
        if method not in ['update_sys_cipher', 'update_encrypt_data']:
            raise Exception('Para method invalid.')

        cipher_type = sys.argv[2]
        if cipher_type not in [VALUE_CIPHER_TYPE_GENERAL,
                               VALUE_CIPHER_TYPE_SM]:
            raise Exception('Para cipher_type invalid.')
        original_type = VALUE_CIPHER_TYPE_GENERAL if \
            cipher_type == VALUE_CIPHER_TYPE_SM else VALUE_CIPHER_TYPE_SM

        server_action = ConfManager(original_type, cipher_type)
        if method == 'update_sys_cipher':
            server_action.update_sys_cipher()
        else:
            server_action.update_conf_cipher()
            server_action.update_private_conf()
            server_action.update_id_rsa()

    except KeyboardInterrupt:
        sys.exit(1)
    except Exception:
        print(traceback.format_exc())
        sys.exit(1)
