#!/bin/bash
DJSecurity_dir="/opt/huawei/dj/DJSecurity"
LOG_FILE="/var/log/huawei/dj/services/system/digital_certificate/check_digital_certificate.log"
SENDALARM_PATH="/opt/omm/oma/tools/sendAlarm"
source /opt/huawei/dj/inst/bin/utils.sh

#告警ID
ALARM_ID_EXPIRING=13279 #即将过期
ALARM_ID_EXPIRED=13278 #已经过期
REGION_IAM_CERT_ALARM_ID=10761 #iam证书错误告警id
FSP_CERT_ALARM_ID=10762 #fsp证书错误告警id

#告警类型
ALARM_TYPE_FAULT="0"
ALARM_TYPE_RECOVER="1"
#告警重发间隔
ALARM_SEND_INTERVAL=`expr 12 \* 60 \* 60`

#告警时间
ALARM_TIME='30'
#7天内即将过期发送严重告警，时间:604800s
CRITICAL_ALARM_TIME=604800
#告警级别
CRITICAL_ALARM_LEVEL=1
MAJOR_ALARM_LEVEL=2
#检查周期
CHECK_CYC='1'
#检查开关
CHECK_OFF_OR_ON='on'
#月份
MONTH="Jan Feb May Apr Mar Jun Jul Aug Sep Oct Nov Dec"
month_list=($MONTH)


# 检查证书时间
# $1: 证书路径
_check_certificate_date()
{
    local cert_path="$1"
    local alarm_time="$2"

    local alarm_time_second=`expr ${alarm_time} \* 86400`

    if [[ ! -f "${cert_path}" ]];then
        return
    fi
    
    openssl x509 -subject_hash -fingerprint -in "${cert_path}" -noout -text &>/dev/null
    if [[ $? -ne 0 ]];then
        return
    fi

    local diff_time=$(get_cert_diff_time ${cert_path})
    if [[ ${diff_time} -lt 0 ]];then
        # 证书已经过期，发布或者转换成已经过期告警（针对即将过期证书，要先撤销这条告警）
        log_info "${cert_path}:The certificate has expired."
        recover_expiring_alarm ${CRITICAL_ALARM_LEVEL} ${cert_path}; recover_ret=$?
        send_expired_alarm_to_oc ${cert_path}; send_ret=$?
        return ${recover_ret} && ${send_ret}
    elif [[ ${diff_time} -ge 0 ]] && [[ ${diff_time} -le ${CRITICAL_ALARM_TIME} ]];then
        # 7天内即将过期发送紧急告警
        log_info "${cert_path}:The certificate is about to expire in 7 days."
        recover_expiring_alarm ${MAJOR_ALARM_LEVEL} ${cert_path}; recover_ret=$?
        send_expiring_alarm ${CRITICAL_ALARM_LEVEL} ${cert_path}; send_ret=$?
        return ${recover_ret} && ${send_ret}
    elif [[ ${diff_time} -ge ${CRITICAL_ALARM_TIME} ]] && [[ ${diff_time} -le ${alarm_time_second} ]];then
        # 30天内即将过期发送重要告警
        log_info "${cert_path}:The certificate is about to expire."
        recover_expired_alarm ${cert_path}; recover_ret=$?
        send_expiring_alarm ${MAJOR_ALARM_LEVEL} ${cert_path}; send_ret=$?
        return ${recover_ret} && ${send_ret}
    fi

    log_info "${cert_path}:The certificate is normal."
    recover_expired_alarm ${cert_path}; recover_ret=$?
    recover_expiring_alarm ${CRITICAL_ALARM_LEVEL} ${cert_path}; cri_ret=$?
    recover_expiring_alarm ${MAJOR_ALARM_LEVEL} ${cert_path}; major_ret=$?
    return ${recover_ret} && ${cri_ret} && ${major_ret}
}

function get_cert_time() {
    local cert_path=$1
    local cer_date=`openssl x509 -subject_hash -fingerprint -in "${cert_path}" -noout -text | grep -w 'Not After'`
    local cer_date_arr=(${cer_date})
    # 证书时间秒数
    local cer_date_time=`date --date="${cer_date_arr[3]} ${cer_date_arr[4]} ${cer_date_arr[6]} ${cer_date_arr[5]}" +%s`
    echo ${cer_date_time}
}

function get_cert_diff_time()
{
    local cer_date_time=$(get_cert_time $1)
    # 当前时间秒数
    local cur_date_time=`date +%s`
    local diff_time=`expr ${cer_date_time} - ${cur_date_time}`
    echo "${diff_time}"
}

function get_cer_time_show() {
    local cer_date_time=$(get_cert_time $1)
    local cer_date_time_disp=`date -d @${cer_date_time} "+%Y-%m-%d"`
    echo ${cer_date_time_disp}
}

# 检查证书是否错误，发送告警
function check_cert_error()
{
    local cert_path=$1
    local service=$2
    local ssl_count=`check_karbor_connect | grep ${service} | grep SSLError | wc -l`
    if [[ ${ssl_count} -gt 0 ]];then
        log_info "${cert_path}:The certificate is error, send alarm."
        send_cert_sslerror_alarm ${service} ${cert_path}
    else
         recover_cert_sslerror_alarm ${service} ${cert_path}
    fi
    return 0
}

function get_alarm_service()
{
    local file_name=$1
    if [[ ${file_name} == "trust_iam.cer" ]]
    then
        echo "IAM"
    elif [[ ${file_name} == "nova_ca.crt" ]]
    then
        echo "Nova"
    fi
}

function send_cert_sslerror_alarm_to_oc()
{
    local alarm_id=$1
    local cert_path=$2
    local node_ip=`get_info.py --manage_ip`
    local node_name=`get_info.py --hostname`
    local cur_date_time=`date +%s`
    ${SENDALARM_PATH} ${alarm_id} 0 1 "" "${cur_date_time}" "CPS" "certificate" "${node_name};${cert_path}" "${node_ip}"
    if [[ $? -ne 0 ]];then
        log_error "${cert_path}:The certificate error alarm send failed"
        return 1
    fi
    log_info "${cert_path}:The certificate error alarm send success"
    return 0
}

function send_cert_sslerror_alarm()
{
    local service=$1
    local cert_path=$2
    if [[ ${service} == "IAM" ]];then
        alarm_id=${REGION_IAM_CERT_ALARM_ID}
    else
        alarm_id=${FSP_CERT_ALARM_ID}
    fi
    send_cert_sslerror_alarm_to_oc ${alarm_id} ${cert_path}
    return $?
}

function recover_cert_sslerror_alarm()
{
    local service=$1
    local cert_path=$2
    if [[ ${service} == "IAM" ]];then
        local alarm_id=${REGION_IAM_CERT_ALARM_ID}
    else
        local alarm_id=${FSP_CERT_ALARM_ID}
    fi
    local node_ip=`get_info.py --manage_ip`
    local node_name=`get_info.py --hostname`
    local cur_date_time=`date +%s`
    ${SENDALARM_PATH} ${alarm_id} 1 1 "" "${cur_date_time}" "CPS" "certificate" "${node_name};${cert_path}" "${node_ip}"
    if [[ $? -ne 0 ]];then
        log_error "${cert_path}:The certificate error recover alarm send failed"
        return 1
    fi
    log_info "${cert_path}:The certificate error recover alarm send success"
    return 0
}

function send_expired_alarm_to_oc()
{
    local cert_path=$1
    local cur_date_time=`date +%s`
    local node_name=`get_info.py --hostname`
    local node_ip=`get_info.py --manage_ip`
    ${SENDALARM_PATH} 13278 0 1 "" "${cur_date_time}" "CPS" "certificate" "${node_name};${cert_path}" "${node_ip}"
    if [[ $? -ne 0 ]];then
        log_error "${cert_path}:The certificate expired alarm send failed"
        return 1
    fi
    log_info "${cert_path}:The certificate expired alarm send success"
    return 0
}

function recover_expired_alarm()
{
    local cert_path=$1
    local cur_date_time=`date +%s`
    local node_name=`get_info.py --hostname`
    local node_ip=`get_info.py --manage_ip`
    ${SENDALARM_PATH} 13278 1 1 "" "${cur_date_time}" "CPS" "certificate" "${node_name};${cert_path}" "${node_ip}"
    if [[ $? -ne 0 ]];then
        log_error "${cert_path}:The certificate expired recover alarm send failed"
        return 1
    fi
    log_info "${cert_path}:The certificate expired recover alarm send success"
    return 0
}

function send_expiring_alarm_to_oc()
{
    local alarm_level=$1
    local cert_path=$2
    local cur_date_time=`date +%s`
    local node_name=`get_info.py --hostname`
    local node_ip=`get_info.py --manage_ip`
    local cer_date_time_disp=$(get_cer_time_show ${cert_path})
    ${SENDALARM_PATH} 13279 0 ${alarm_level} "" "${cur_date_time}" "CPS" "certificate" "${node_name};${cert_path}" "${node_ip};${cer_date_time_disp}"
    if [[ $? -ne 0 ]];then
        log_error "${cert_path}:The certificate is about to expire alarm send failed"
        return 1
    fi
    log_info "${cert_path}:The certificate is about to expire alarm send success"
    return 0
}

function send_expiring_alarm()
{
    local alarm_level=$1
    local cert_path=$2
    local cur_date_time=`date +%s`
    local node_name=`get_info.py --hostname`
    local node_ip=`get_info.py --manage_ip`
    local cer_date_time_disp=$(get_cer_time_show ${cert_path})
    ${SENDALARM_PATH} 13279 0 ${alarm_level} "" "${cur_date_time}" "CPS" "certificate" "${node_name};${cert_path}" "${node_ip};${cer_date_time_disp}"
    if [[ $? -ne 0 ]];then
        log_error "${cert_path}:The certificate is about to expire alarm send failed"
        return 1
    fi
    log_info "${cert_path}:The certificate is about to expire alarm send success"
    return 0
}

function recover_expiring_alarm()
{
    local alarm_level=$1
    local cert_path=$2
    local cur_date_time=`date +%s`
    local node_name=`get_info.py --hostname`
    local node_ip=`get_info.py --manage_ip`
    local cer_date_time_disp=$(get_cer_time_show ${cert_path})
    ${SENDALARM_PATH} 13279 1 ${alarm_level} "" "${cur_date_time}" "CPS" "certificate" "${node_name};${cert_path}" "${node_ip};${cer_date_time_disp}"
    if [[ $? -ne 0 ]];then
        log_error "${cert_path}:The certificate is about to expire recover alarm send failed"
        return 1
    fi
    log_info "${cert_path}:The certificate is about to expire recover alarm send success"
    return 0
}

function get_cert_dir(){
    for element in `ls $1`
    do
        local dir_or_file=$1"/"${element}
        if [[ -d ${dir_or_file} ]]
        then
            get_cert_dir ${dir_or_file} $2
        else
            local file_name=${dir_or_file##*/}
            local service=`get_alarm_service ${file_name}`
            if [[ "${file_name##*-}" == "cert.pem" ]]
            then
                echo ${dir_or_file}
                _check_certificate_date ${dir_or_file} $2
            elif [[ "${file_name##*/}" == "trust_iam.cer" ]]
            then
                echo ${dir_or_file}
                _check_certificate_date ${dir_or_file} $2
                check_cert_error ${dir_or_file} ${service}
            elif [[ "${file_name##*/}" == "nova_ca.crt" ]]
            then
                echo ${dir_or_file}
                _check_certificate_date ${dir_or_file} $2
                check_cert_error ${dir_or_file} ${service}
            fi
        fi
    done
}

_main()
{
    #读数据库操作
    local key="certificate.alarm.date"
    local ALARM_TIME=$(csbs_python -c "from basesdk import cms_info;print(cms_info.get_cms_info(\"${key}\"))")
    if [[ "${ALARM_TIME}" == "" ]];then
        ALARM_TIME="30"
    fi
    get_cert_dir ${DJSecurity_dir} ${ALARM_TIME}
}

_main
