#!/bin/bash

# 获取脚本所在的当前路径
cd $(dirname $0)
declare -r gCurPath="$PWD"
cd - >/dev/null

############################
#gaussdb控制脚本：
#        stop_gaussdb_force stop_gaussdb start_gaussdb_single
#        start_gaussdb_primary start_gaussdb_standby 
#        status  rebuild  promote  dbState  localRole
############################
DB_CONTROL_PY="/etc/gaussdb/gaussdb/dbControl.py"
source /home/db_func.sh


function get_private_key_pwd()
{
     encrypt_password=$(cat /opt/huawei/dj/DJSecurity/privkey/privkey.conf | awk '{print $2}')
     export ENCRYPT_DATA=${encrypt_password}
     csbs_python -c "import kmc.kmc;A=kmc.kmc.API();print(A.decrypt(0))"
}

function config_gaussdb_data()
{
    [[ -f /opt/fusionplatform/data/gaussdb/server.key.cipher ]] && return 0
    mkdir -p /opt/fusionplatform/data/gaussdb
    privkey=$(get_private_key_pwd)
    su - gaussdba -s "/bin/bash" -c "gs_guc encrypt -M server -K ${privkey}"
    if [ $? -ne 0 ];then
        exit 1
    fi
    su - gaussdba -s "/bin/bash" -c "gs_guc encrypt -M client -K ${privkey}"
    if [ $? -ne 0 ];then
        exit 1
    fi
    /usr/bin/cp -f /opt/fusionplatform/data/gaussdb/*.key.* /opt/fusionplatform/data/cert
    chown -hR gaussdba:dbgrp /opt/fusionplatform/data
    su -s /bin/sh gaussdba -c "chmod 600 /opt/fusionplatform/data/cert/*"
}


function _decript_pas_gaussdba()
{
    export ENCRYPT_DATA="${gaussdbaPass}"
    gaussdbaPass=`csbs_python -c "from kmc import kmc;print(kmc.API().decrypt(0))"`
}

######################################################################
#   FUNCTION   : getStatus_single
#   DESCRIPTION: 单机数据库状态查询函数
#   脚本返回码 : 
#   单主模式 或 双主模式
#    declare -i db_normal=0           #   正常运行     
#    declare -i db_abnormal=1         #   运行异常     
#    declare -i db_stopped=2          #   停止         
#    declare -i db_unknown=3          #   状态未知     
#    declare -i db_starting=4         #   正在启动     
#    declare -i db_stopping=5         #   正在停止     
#    declare -i db_primary=6          #   主正常运行   
#    declare -i db_standby=7          #   备正常运行   
#    declare -i db_activating=8       #   正在升主     
#    declare -i db_deactivating=9     #   正在降备     
#    declare -i db_notsupported=10    #   动作不存在   
#    declare -i db_repairing=11       #   正在修复
######################################################################
getStatus_single()
{
    local dbinfo=""
    local -i retVal=$r_success
    
    # 查询数据库是否正常运行    
    # 数据库未启动
    if ! isDBRunning; then
        log_error "[getStatus_single] db no runnig now."
        return $db_stopped
    fi
    
    # 获取数据库状态
    dbinfo=$(getDBStatusInfo)
    
    # 从数据库的查询结果中分离出各项信息: LOCAL_ROLE, DB_STATE, DETAIL_INFORMATION, PEER_ROLE. 前面四个变量会在getDBState赋值。
    getDBState "$dbinfo"; retVal=$?             # 执行命令，立即获取返回值。写在同一行，是为了避免后续修改时，在两个语句中插入了其他命令，导致获取失败
    
    # 根据数据库的角色进行判断，正常情况下，只会出现Normal和空（即获取失败）
    case "$LOCAL_ROLE" in
        "$gs_r_normal")
            log_info "[getStatus_single] db current role Normal. [$dbinfo]"
            return $db_normal
            ;;

        # 以下三种状态，虽然在单机下不可能出现，但是需要进行异常处理，这三种状态下，先判断是否在重建中。
        "$gs_r_primary"|"$gs_r_standby"|"$gs_r_cstandby")
            log_warn "[getStatus_single] db current role error[$LOCAL_ROLE], need further check. [$dbinfo]"
            ;;
        
        # Pending角色下的数据库，可以直接状态变更，所以返回未启动
        "$gs_r_pending")
            log_warn "[getStatus_single] db current role Pending, need restart. [$dbinfo]"
            return $db_abnormal
            ;;
        
        # 其他角色不是数据库的正常角色，可能是获取失败了，认为数据库没有启动
        "$gs_r_unknown"|*)
            log_error "[getStatus_single] db current role[$LOCAL_ROLE] error! [$dbinfo]"
            return $db_abnormal
            ;;
    esac     
    
    # 以下判断是在本端数据库角色为主、备、级联备的情况的判断，属于异常处理。
    case "$DB_STATE" in
        # 以下几种状态，表示数据库状态不稳定，待稳定后再进行下一步处理，以避免出问题。
        "$gs_s_rebuilding")
            log_warn "[getStatus_single] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $db_repairing
            ;;
            
        "$gs_s_demoting")
            log_warn "[getStatus_single] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $db_deactivating
            ;;

        "$gs_s_waiting"|"$gs_s_promoting")
            log_warn "[getStatus_single] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $db_activating
            ;;
        
        "$gs_s_starting")
            log_warn "[getStatus_single] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $db_starting
            ;;
        
        # 其他状态，直接重启为单机
        "$gs_s_normal"|"$gs_s_needrepair"|"$gs_s_unknown"|*)
            log_warn "[getStatus_single] db state[$DB_STATE] stable now, need restart. [$dbinfo]"
            return $db_abnormal
            ;;
    esac
    
    # 数据库正常运行
    return $db_normal
}

######################################################################
#   FUNCTION   : getStatus_double
#   DESCRIPTION: 双机数据库状态查询函数
#   脚本返回码 : 
#   主备模式
#   declare -i db_normal=0           #   正常运行     
#   declare -i db_abnormal=1         #   运行异常     
#   declare -i db_stopped=2          #   停止         
#   declare -i db_unknown=3          #   状态未知     
#   declare -i db_starting=4         #   正在启动     
#   declare -i db_stopping=5         #   正在停止     
#   declare -i db_primary=6          #   主正常运行   
#   declare -i db_standby=7          #   备正常运行   
#   declare -i db_activating=8       #   正在升主     
#   declare -i db_deactivating=9     #   正在降备     
#   declare -i db_notsupported=10    #   动作不存在   
#   declare -i db_repairing=11       #   正在修复
######################################################################
getStatus_double()
{
    local dbinfo=""
    local -i retVal=0
    local -i outResult=$r_success                    # 返回的结果，用于在主备用情况下，暂存结果。
    
    # 查询数据库是否正常运行    
    # 数据库未启动
    if ! isDBRunning; then
        log_error "[getStatus_double] db no runnig now."
        
        # 如HA期待的数据库角色为主,插件认DB正确就应为主,此时启动为主
        if [ "$runState" == "active" ]; then
            log_warn "[getStatus_double] HA wantted primary,DB no runnig,restart to primary!!!"
            restartToPrimary;retVal=$?
            return $db_stopped
        fi
        
        # 数据库没有运行，有可能是启动不了，检查是否需要重建
        isNeedRebuild; retVal=$?
        [ $retVal -eq $db_normal ] || return $retVal
        return $db_stopped
    fi
    
    # 获取数据库状态
    dbinfo=$(getDBStatusInfo)

    # 从数据库的查询结果中分离出各项信息: LOCAL_ROLE, DB_STATE, DETAIL_INFORMATION, PEER_ROLE. 前面四个变量会在getDBState赋值。
    getDBState "$dbinfo"; retVal=$?

    # 根据数据库的角色进行判断，正常情况下，只会出现 Pendig, Primary, Standby 和空（即获取失败）
    case "$LOCAL_ROLE" in
        # Pending状态下的数据库，可以直接状态变更，所以返回期望状态的反状态
        "$gs_r_pending")
            log_warn "[getStatus_double] db current role[$LOCAL_ROLE], current ha role[$runState]."
            return $db_normal
            ;;
            
        # 以下两种状态，需要先判断是否正在重建
        "$gs_r_primary")
            outResult=$db_primary
            log_info "[getStatus_double] db current role[$LOCAL_ROLE], need further check."
            ;;
        
        "$gs_r_standby")
            outResult=$db_standby
            log_info "[getStatus_double] db current role[$LOCAL_ROLE], need further check."
            ;;
            
        # 以下角色不是双机会出现的角色，直接重启即可
        "$gs_r_normal"|"$gs_r_cstandby"|"$gs_r_unknown"|*)
            log_error "[getStatus_double] db current role[$LOCAL_ROLE] error! [$dbinfo]"
            return $db_stopped
            ;;
    esac

    # 当前为HA的状态为未知时，数据库主备需要返回为正常
    [ "$runState" == "unknown" ] && outResult=$db_normal
    
    # 以下判断是在本端数据库角色为主、备、级联备的情况的判断，需要判断数据库的状态
    case "$DB_STATE" in
        # 以下几种状态，表示数据库状态不稳定，待稳定后再进行下一步处理，以避免出问题。
        "$gs_s_rebuilding")
            log_warn "[getStatus_double] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $db_abnormal #HA advice when standby is repairing, return excepion to it.
            ;;
            
        "$gs_s_demoting")
            log_warn "[getStatus_double] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $db_deactivating
            ;;

        "$gs_s_waiting"|"$gs_s_promoting")
            log_warn "[getStatus_double] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $db_activating
            ;;
        
        "$gs_s_starting")
            log_warn "[getStatus_double] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $db_starting
            ;;
            
        # 数据库需要重建，则需要考虑对端情况
        "NeedRepair")
            # 重建状态不一定都是需要修复的
            if echo "$DETAIL_INFORMATION" | grep -Ew "$gs_r_canrepair"; then
                log_warn "[getStatus_double] db state[$DB_STATE] need repair now, and current role[$LOCAL_ROLE]. [$dbinfo]"
                # 如果数据库角色为主，则不先不进行其他处理
                if [ "$LOCAL_ROLE" == "Primary" ];then 
                    return $db_activating
                else
                # 否则，返回需要修复
                    return $db_abnormal
                fi
            else
                log_warn "[getStatus_double] db state[$DB_STATE] need repair now, but no need to repair, DETAIL_INFORMATION[$DETAIL_INFORMATION]. [$dbinfo]"
                return $outResult
            fi
            ;;
                    
        # 其他状态，直接重启跳转到需要的状态
        "$gs_s_normal"|"$gs_s_unknown"|*)
            log_info "[getStatus_double] db state[$DB_STATE] stable now, can goto want role."

            return $outResult
            ;;
    esac
    
    # 数据库正常运行，返回当前状态
    return $db_normal
}


######################################################################
#   FUNCTION   : getStatus
#   DESCRIPTION: 数据库状态查询函数
######################################################################
getStatus()
{
    bash /opt/huawei/dj/etc/gaussdb/sync_monitor.sh check_progress>/dev/null 2>&1
    # 数据库的状态检查大体步骤如下，具体处理单双机有一些区别
    # Step1: 数据库不在位，需要返回不在位    
    # Step2: 数据库在位，检查是否pending，是则返回与期望状态相反    
    # Step3: 数据库在位，处于中间状态，返回相应的中间态    
    # Step4: 数据库在位，检查是否需要重建，如果需要重建，则数据库角色为备端返回需要重建    
    # Step5: 数据库在位，不需要重建，如果数据库角色正确，返回OK    
    # Step6: 数据库在位，数据库角色不正确，返回需要激活或去激活    
    local -i retVal=$r_success
    if [ "$gsCurMode" == "$gsdbSingle" ]; then
        getStatus_single ; retVal=$?
    else
        getStatus_double ; retVal=$?
    fi
    return $retVal
}


######################################################################
#   FUNCTION   : doStart_single
#   DESCRIPTION: 单机数据库启动函数，只将数据启动为Normal
######################################################################
doStart_single()
{
    local -i retVal=$r_success
    
    # 启动前，先检查当前数据库是否已经启动为Normal
    if checkLocalRole "Normal"; then
        log_info "[doStart_single] db had been start already."
        return $r_success
    fi
    
    # 启动数据库
    restartToNormal; retVal=$?
    
    return $retVal
}

######################################################################
#   FUNCTION   : doStart_double
#   DESCRIPTION: 双机数据库启动函数
######################################################################
doStart_double()
{
    local -i retVal=$r_success

    # 启动前，先检查当前数据库是否已经启动为Primary|standby|Pending
    if checkLocalRole "Primary|standby|Pending"; then
        log_info "[doStart_double] db had been start already."
        return $r_success
    fi
        
    # 启动数据库
    restartToPending; retVal=$?
    
    return $retVal
}


######################################################################
#   FUNCTION   : doStart
#   DESCRIPTION: 数据库启动函数
######################################################################
doStart()
{
    local -i retVal=$r_success
  
    # 未启动，则根据单机双机配置启动为相应的模式
    if [ "$gsCurMode" == "$gsdbSingle" ]; then
        doStart_single ; retVal=$?
    else
        doStart_double ; retVal=$?
    fi    
    
    return $retVal
}

######################################################################
#   FUNCTION   : doStop
#   DESCRIPTION: 数据库停止函数
######################################################################
doStop()
{
    local -i retVal=$r_success

    # 停止前，先检查当前数据库是否已经停止
    if ! isDBRunning; then
        log_info "[doStop]db is not running now."
        return $r_success
    fi
    
    # 停止数据库
    su gaussdba -s "/bin/bash" -c "gs_ctl stop -m fast"; retVal=$?
    
    return $r_success
}


######################################################################
#   FUNCTION   : doStopforce
#   DESCRIPTION: 数据库强制停止函数
######################################################################
doStopforce()
{
    local -i retVal=$r_success

    # 停止前，先检查当前数据库是否已经停止
    if ! isDBRunning; then
        log_info "[doStopforce]db is not running now."
        return $r_success
    fi

    # 强制停止数据库
    su gaussdba -s "/bin/bash" -c "gs_ctl stop -m immediate"; retVal=$?
    
    return $r_success
}

######################################################################
#   FUNCTION   : doRepair_single
#   DESCRIPTION: 单机数据修复函数
######################################################################
doRepair_single()
{
    # 数据库角色不正在，在此修复。
    # 如果当前不角色不正确，则进行重启，但是如果正在处理修复阶段，则先等待。
    local dbinfo=""
    local -i retVal=$r_success
    
    # 查询数据库是否正常运行    
    # 数据库未启动
    if ! isDBRunning; then
        log_warn "[doRepair_single] db no runnig now."
        
        # 启动数据库
        doStart_single; retVal=$?
        return $retVal
    fi
    
    # 获取数据库状态
    dbinfo=$(getDBStatusInfo)
    
    # 从数据库的查询结果中分离出各项信息: LOCAL_ROLE, DB_STATE, DETAIL_INFORMATION, PEER_ROLE. 前面四个变量会在getDBState赋值。
    getDBState "$dbinfo"; retVal=$?
    
    # 根据数据库的角色进行判断，正常情况下，只会出现Normal和空（即获取失败）
    case "$LOCAL_ROLE" in
        "$gs_r_normal")
            log_info "[doRepair_single] db current role Normal. [$dbinfo]"
            return $db_normal
            ;;

        # 以下三种状态，虽然在单机下不可能出现，但是需要进行异常处理，这三种状态下，目前不进行其他判断，直接重启
        "$gs_r_primary"|"$gs_r_standby"|"$gs_r_cstandby")
            log_warn "[doRepair_single] db current role error[$LOCAL_ROLE], need restart. [$dbinfo]"
            # 重启数据库
            restartToNormal; retVal=$?
            return $retVal
            ;;
        
        # 其他角色不是数据库的正常角色，直接重启数据库
        *)
            log_error "[doRepair_single] db current role[$LOCAL_ROLE] error! [$dbinfo]"
            # 重启数据库
            restartToNormal; retVal=$?
            return $retVal
            ;;
    esac
    
    # 重启数据库
    restartToNormal; retVal=$?    
    return $retVal
}

######################################################################
#   FUNCTION   : doRepair_double
#   DESCRIPTION: 双机数据修复函数
######################################################################
doRepair_double()
{
    # 修复只做rebuild操作。数据库角色不正确，通过激活和去激活恢复
    # 修复操作只有数据库角色为备或级联备端，同时数据库状态为 NeedRepair
    # 而且数据库连接状态不为连接中，或是连接断开
    local dbinfo=""
    local -i retVal=$r_success
    
    # 查询数据库是否正常运行    
    # 数据库未启动
    if ! isDBRunning; then
        log_warn "[doRepair_double] db no runnig now."
        
        # 数据库没有运行，有可能是启动不了，检查是否需要重建
        isNeedRebuild; retVal=$?
        if [ $retVal -eq $db_abnormal ]; then
            # 调用命令进行重建
            log_warn "[doRepair_double] start to repair by [$OMM_GS_CTL build]. "
            su gaussdba -s "/bin/bash" -c "gs_ctl build -D /opt/fusionplatform/data/gaussdb/ -U gaussdba"; retVal=$?
            # 重建完成，需要清空一下记录
            recordStartResult $retVal
            log_warn "[doRepair_double] repair finish[$retVal]."
            return $retVal
        fi
        
        return $db_stopped
    fi
    
    # 获取数据库状态
    dbinfo=$(getDBStatusInfo)
    
    # 从数据库的查询结果中分离出各项信息: LOCAL_ROLE, DB_STATE, DETAIL_INFORMATION, PEER_ROLE. 前面四个变量会在getDBState赋值。
    getDBState "$dbinfo"; retVal=$?

    # 不为备状态，或是级联备，不处理
    if [ "$LOCAL_ROLE" != "$gs_r_standby" -a "$LOCAL_ROLE" != "$gs_r_cstandby" ]; then
        log_warn "[doRepair_double] LOCAL_ROLE[$LOCAL_ROLE] error. [$dbinfo]"
        return $r_success
    fi
    
    # 不是待修复状态，不处理
    if [ "$DB_STATE" != "$gs_s_needrepair" ]; then
        log_warn "[doRepair_double] current DB_STATE[$DB_STATE] no need to repair. [$dbinfo]"
        return $r_success
    fi
    
    # 详细信息不需要重建，则不重建
    if ! echo "$DETAIL_INFORMATION" | grep -Ew "$gs_r_canrepair"; then
        log_warn "[doRepair_double] current DETAIL_INFORMATION[$DETAIL_INFORMATION] error, it can not be repair. [$dbinfo]"
        return $r_success
    fi
    
    # 调用命令进行重建
    log_warn "[doRepair_double] start exc:[$OMM_GS_CTL build] to repair. [$dbinfo]"
    su gaussdba -s "/bin/bash" -c "gs_ctl build -D /opt/fusionplatform/data/gaussdb/ -U gaussdba"; retVal=$?
    log_warn "[doRepair_double] Ended to repair, GaussDB return: [$retVal]."
    
    # 重建完成，需要清空一下记录
    recordStartResult $retVal
    log_warn "$[doRepair_double] repair finish[$retVal]."

    return $retVal
}



######################################################################
#   FUNCTION   : doRepair
#   DESCRIPTION: 数据修复函数
######################################################################
doRepair()
{
    local -i retVal=$r_success

    if [ "$gsCurMode" == "$gsdbSingle" ]; then
        doRepair_single ; retVal=$?
    else
        doRepair_double ; retVal=$?
    fi        
    return $retVal
}

######################################################################
#   FUNCTION   : doDeactivate
#   DESCRIPTION: 数据库去激活函数
######################################################################
doDeactivate()
{
    # 本端没有启动，尝试启动为备
    if ! isDBRunning; then
        log_warn "[doDeactivate] db no runnig now."
        
        # 启动数据库
        restartToStandby; retVal=$?
        return $retVal
    fi

    # 获取数据库状态
    dbinfo=$(getDBStatusInfo)
    
    # 从数据库的查询结果中分离出各项信息: LOCAL_ROLE, DB_STATE, DETAIL_INFORMATION, PEER_ROLE. 前面四个变量会在getDBState赋值。
    getDBState "$dbinfo"; retVal=$?

    case "$LOCAL_ROLE" in
        # 本端pending时，直接降备
        "$gs_r_pending")
            log_warn "[doDeactivate] db current role[$LOCAL_ROLE], current ha role[$runState], need notify to standby. [$dbinfo]"

            notifyToStandby; retVal=$?
            return $retVal
            ;;
            
        # 已经是备用角色，不需要处理
        "$gs_r_standby")
            log_info "[doDeactivate] db current role[$LOCAL_ROLE] no need to change. [$dbinfo]"
            return $r_success
            ;;
        
        # 主角色下，需要检查
        "$gs_r_primary")
            log_info "[doDeactivate] db current role[$LOCAL_ROLE], need further check. [$dbinfo]"
            ;;
            
        # 以下角色不是双机会出现的角色，直接重启即可
        "$gs_r_normal"|"$gs_r_cstandby"|"$gs_r_unknown"|*)
            log_error "[doDeactivate] db current role[$LOCAL_ROLE] error! [$dbinfo]"
            restartToStandby; retVal=$?
            return $retVal
            ;;
    esac
    
    # 本端角色不正确，需要判断对端，以及重建的情况
    case "$DB_STATE" in
        # 对端在位，且状态正常，不能直接降备，需要等待对端进行swithover或是双主时本端降备
        "$gs_s_normal")
            log_warn "[doDeactivate] db state[$DB_STATE] stable now, start change to stabndby. [$dbinfo]"
            ;;
            
        # 以下几种状态，表示数据库状态不稳定，待稳定后再进行下一步处理，以避免出问题。
        "$gs_s_rebuilding"|"$gs_s_starting"|"$gs_s_demoting"|"$gs_s_promoting"|"$gs_s_waiting")
            log_warn "[doDeactivate] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $r_success
            ;;
        
        # 数据库需要重建，待修复时处理
        "$gs_s_needrepair")
            # 重建状态不一定都是需要修复的，这个需要高斯DB后续修改  lianyunwen
            if echo "$DETAIL_INFORMATION" | grep -Ew "$gs_r_canrepair"; then
                log_warn "[doDeactivate] db state[$DB_STATE] need repair now, can not start to standby. [$dbinfo]"
                return $r_success
            fi
            
            # 不需要修复的情况，直接跳转到需要的状态。
            log_warn "[doDeactivate] db state[$DB_STATE], DETAIL_INFORMATION[$DETAIL_INFORMATION] can not repair, start change to stabndby. [$dbinfo]"
            ;;
            
        # 其他状态，直接重启跳转到需要的状态
        "$gs_s_unknown"|*)
            log_info "[doDeactivate] db state[$DB_STATE] stable now, can goto want role. [$dbinfo]"
            restartToStandby; retVal=$?
            return $retVal
            ;;
    esac
    
    # 主降备，在操作前，需要看对端的状态
    if [ "" == "$PEER_ROLE" ]; then
        PriToSta; retVal=$?
        return $retVal
    fi            
    
    log_warn "[doDeactivate] db state[$DB_STATE] stable now, and peer role[$PEER_ROLE], need peer to primary first. [$dbinfo]"
    return $r_success
}


function doActivate()
{
    local outInfo=""
    outInfo=$(bash /opt/huawei/dj/etc/gaussdb/sync_monitor.sh check_status 2>&1);ret=$?
    log_info "[doActivate] db sync status [$outInfo]."
    if [ $ret -ne 0 ];then
        log_error "[doActivate] db sync status is not correct[$outInfo]."
        [[ $ret -eq 1 ]] && return $r_failure    
    fi
    # 本端没有启动，尝试启动为主
    if ! isDBRunning; then
        log_warn "[doActivate] db no runnig now."
        
        # 启动数据库
        restartToPrimary; retVal=$?
        return $retVal
    fi
    # 获取数据库状态
    dbinfo=$(getDBStatusInfo)
    # 从数据库的查询结果中分离出各项信息: LOCAL_ROLE, DB_STATE, DETAIL_INFORMATION, PEER_ROLE. 前面四个变量会在getDBState赋值。
    getDBState "$dbinfo"; retVal=$?
    
    case "$LOCAL_ROLE" in
    # 本端pending时，直接升主
    "$gs_r_pending")
        log_warn "[doActivate] db current role[$LOCAL_ROLE], current ha role[$runState], need restart. [$dbinfo]"
        # pending 数据库BUG 升主之前需要翻转数据库配置
        
        notifyToPrimary; retVal=$?
        return $retVal
        ;;
    
    # 已经是主用状态，不需要处理
    "$gs_r_primary")
        log_info "[doActivate] db current role[$LOCAL_ROLE] no need to change. [$dbinfo]"
        return $r_success
        ;;
    
    "$gs_r_standby")
        log_info "[doActivate] db current role[$LOCAL_ROLE], need further check. [$dbinfo]"
        ;;
        
    # 以下角色不是双机会出现的角色，直接重启即可
    "$gs_r_normal"|"$gs_r_cstandby"|"$gs_r_unknown"|*)
        log_error "[doActivate] db current role[$LOCAL_ROLE] error! [$dbinfo]"
        restartToPrimary; retVal=$?
        return $retVal
        ;;
    esac
    
    # 本端角色不正确，需要判断对端，以及重建的情况
    case "$DB_STATE" in
        # 对端在位，且状态正常，先尝试switchover，再尝试failover
        "$gs_s_normal")
            log_info "[doActivate] db state[$DB_STATE] stable now, start change to primary. [$dbinfo]"
            StaToPri; retVal=$?
            return $retVal
            ;;
            
        # 以下几种状态，表示数据库状态不稳定，待稳定后再进行下一步处理，以避免出问题。
        "$gs_s_rebuilding"|"$gs_s_starting"|"$gs_s_demoting"|"$gs_s_promoting"|"$gs_s_waiting")
            log_warn "[doActivate] db state[$DB_STATE] instable now, need wait. [$dbinfo]"
            return $r_success
            ;;
        
        # 数据库需要重建，待修复时处理
        "$gs_s_needrepair")
            # 重建状态不一定都是需要修复的，这个需要高斯DB后续修改  lianyunwen
            if echo "$DETAIL_INFORMATION" | grep -Ew "$gs_r_canrepair"; then
                log_warn "[doActivate] db state[$DB_STATE] need repair now, can not start to primary. [$dbinfo]"
                return $r_success
            fi
            
            # 不需要修复的情况，直接跳转到需要的状态。
            log_warn "[doActivate] db state[$DB_STATE], DETAIL_INFORMATION[$DETAIL_INFORMATION] can not repair, start change to primary. [$dbinfo]"
            restartToPrimary; retVal=$?
            return $retVal
            ;;
            
        # 其他状态，直接重启跳转到需要的状态
        "$gs_s_unknown"|*)
            log_info "[doActivate] db state[$DB_STATE] stable now, can goto want role. [$dbinfo]"
            restartToPrimary; retVal=$?
            return $retVal
            ;;
    esac
    
    return $r_success
    
}

function notifyService()
{
    return 0
}


function main()
{
    local -i retVal=0
    case "$optCommand" in
    status)
        # To Do 主备模式资源会用到 $2
        # 查询资源状态，返回码为上面列出的返回码
        getStatus; retVal=$?
        return ${retVal}
        ;;
        
    start)
        # To Do 主备模式资源不会用到该action（需要使用active）
        # 启动资源，返回码 0表示成功 1表示失败
        config_gaussdb_data
        doStart; retVal=$?
        return $retVal
        ;;
        
    stop)
        # To Do
        # 停止资源，返回码 0表示成功 1表示失败
        doStop; retVal=$?
        return $retVal
        ;;
        
    active)
        # To Do 仅主备模式资源会用到该action
        # 激活资源，返回码 0表示成功 1表示失败
        config_gaussdb_data
        doActivate; retVal=$?
        return $retVal
        ;;
    deactive)
        # To Do 仅主备模式资源会用到该action
        # 去激活资源，返回码 0表示成功 1表示失败
        config_gaussdb_data
        doDeactivate; retVal=$?

        return $retVal
        ;;
    force-stop)
        # To Do在stop失败时，会执行此操作
        # 强制停止资源，返回码 0表示成功 1表示失败
        doStopforce; retVal=$?
        return $retVal
        ;;
        

    repair)
        # To Do 此处可能需要使用 $2 
        # 如果是停止失败，会进行force-stop，不需要修复
        # 在start、active、deactive失败时，会调用此接口
        # 修复资源，返回码 0表示成功 1表示失败
        config_gaussdb_data
        doRepair; retVal=$?
        return $retVal
        ;;
    notify)
        # To Do 此处可能需要使用 $2 
        # 资源状态变更时，会调用此操作
        # 资源状态变更通知，返回码 0表示成功 1表示失败
        notifyService "$nextState" "$haName"; retVal=$?
        return $retVal
        ;;
        
    prepare)
        # To Do 此处可能需要使用 $2 
        #主备切换之前，需要做的一些准备工作
        # 资源状态变更前的准备工作，返回码 0表示成功 1表示失败
        log_info "Not support cmd[$@]."
        return $db_notsupported
        ;;
    
    *)
        #echo "Usage: $0 { stop|status|notify|force-stop|..}"
        # 返回动作不存在
        log_info "Unknown cmd[$@]."
        return $db_notsupported
        ;;
    esac

}

# 获取脚本入参
inParamNum="$#"
inParamLst="$@"

declare optCommand="$1"; shift                                          # 当前运行的命令
declare runState="$1";  shift                                           # 当前ha的运行状态

declare nextState="$1"; shift   # 资源状态变更后的状态（仅在notify时有效，其他情况都无此入参），资源变更后的状态的取值为（0 = 正常, 1 = 故障）。
declare haName="$1";    shift                                           # ha name
gaussdbaPass=$(grep gaussdb_admin_pwd /opt/huawei/dj/cfg/sys.ini | awk -F '=' '{print $NF}' | sed 's/ //g')


declare scriptName="$(basename $0)"

declare -i scriptRetVal=0

# 如果不是单机，则默认是双机配置
tail -1 /opt/fusionplatform/data/gaussdb/postgresql.conf | grep repl_force_cert_check >> /dev/null
if [ $? -ne 0 ];then
    gsCurMode="$gsdbSingle"
fi
[ "$gsCurMode" != "$gsdbSingle" ] && gsCurMode="$gsdbDouble"

_decript_pas_gaussdba

export GAUSSPASSWORD=$gaussdbaPass

main ; scriptRetVal=$?
unset GAUSSPASSWORD

exit $scriptRetVal
