#!/usr/bin/csbs_python
# -*- encoding:utf-8 -*-
import configparser
import sys
from basesdk import utils

LOG = utils.get_logger("cms_watcher")
conf_file = "/etc/karbor/karbor.conf"  # noqa


class KarborConf(object):
    def __init__(self, conffile):
        self.cf = configparser.RawConfigParser()
        self.cf.read(conffile)

    def get_from_karbor_conf(self, section_name, key):
        if self.cf.has_option(section_name, key):
            return self.cf.get(section_name, key)

    def put_to_karbor_conf(self, section_name, key, value):
        self.cf.set(section_name, key, value)

    def flush(self):
        with open(conf_file, "w") as conf:
            self.cf.write(conf)


def mapping_value(current_key, value_map, default):
    current_value = utils.get_info(current_key)
    if current_value not in value_map:
        return default
    else:
        return value_map[current_value]


def list_value(current_key, *kwarg):
    return utils.get_info(current_key, "list")


function_map = {
    "profile": [mapping_value,
                {"public": "public_cloud", "private_kvm": "private_cloud",
                 "private_recover": "private_cloud"}, "public_cloud"],
    "rabbitmq_nodes": [list_value, {}]
}


# 需要替换的采用for_replace结尾命名占位符，直接赋值的占位符设置为None
key_map = {
    "gaussdb_admin_pwd": {
        "database": {"connection": "gaussdb_admin_pwd"},
        "wf_database": {"connection": "wf_db_pwd"},
        "wf_database_async": {"connection": "wf_db_pwd"}},
    "gaussdb_float_ip": {
        "database": {"connection": "db_float_ip_for_replace"},
        "wf_database": {"connection": "db_float_ip_for_replace"},
        "wf_database_async": {"connection": "db_float_ip_for_replace"}},
    "db_ca_file": {
        "database": {"connection": "db_ca_file_for_replace"},
        "wf_database": {"connection": "db_ca_file_for_replace"}},
    "rabbitmq_nodes": {
        "oslo_messaging_rabbit": {"rabbit_hosts": "rabbitmq_ip_addr"}},
    "rabbitmq_pwd": {
        "oslo_messaging_rabbit": {"rabbit_password": "rabbitmq_pwd"}},
    "haproxy_float_ip": {
        "IAM": {"karbor_endpoint": "karbor_tenant_float_ip"}},
    "cinder_url": {
        "cinder_client": {"cinder_endpoint": "cinder_url_for_replace"}},
    "nova_url": {
        "nova_client": {"nova_endpoint": "nova_url_for_replace"}},
    "glance_url": {
        "glance_client": {"glance_endpoint": "glance_url_for_replace"}},
    "ceilometer_url": {
        "ceilometer_client": {
            "ceilometer_endpoint": "ceilometer_url_for_replace"}},
    "manage_ip": {
        "DEFAULT": {"osapi_karbor_listen": "internal_ip_for_replace"}},
    "auth_url": {"trustee": {"auth_url": "auth_url_for_replace"}},
    "default_user_pwd": {"trustee": {"password": "default_user_pwd"}},
    "cts_url": {"cts_client": {"cts_endpoint": "cts_url_for_replace"}},
    "smn_url": {"smn_client": {"smn_endpoint": "smn_url_for_replace"}},
    "cbc_url": {"cbc_client": {"cbc_endpoint": "cbc_url_for_replace"}},
    "profile": {"DEFAULT": {"scene": "scene_profile_for_replace"}}
}


def change_single(section, key, value, replace_holder, kc):
    origin_value = kc.get_from_karbor_conf(section, key)
    if replace_holder in origin_value:
        if value is None:
            value = ""
        origin_value = origin_value.replace(replace_holder, value)
        kc.put_to_karbor_conf(section, key, origin_value)
        LOG.info("Change_single section[{}], key[{}]".format(section, key))


def set_single(section, key, value, kc):
    kc.put_to_karbor_conf(section, key, value)
    LOG.info("set_single section[%s] key[%s] value[%s]" %
             (section, key, value))


def replace_conf(value, sections, kc):
    for sect in sections:
        items = sections[sect]
        for item_key in items:
            replace_holder = items[item_key]
            if replace_holder:
                change_single(sect, item_key, value, replace_holder,
                              kc)
            else:
                set_single(sect, item_key, value, kc)


def batch_change():
    kc = KarborConf(conf_file)
    for key_index in key_map:
        try:
            if key_index in function_map:
                exec_param = function_map[key_index]
                value = exec_param[0](key_index, *exec_param[1:])
            else:
                value = utils.get_info(key_index)
            sections = key_map[key_index]
            replace_conf(value, sections, kc)
        except Exception:
            LOG.exception("batch_change key[%s] Exception" % key_index)
    kc.flush()


def single_change(section, key, replace_holder, value):
    kc = KarborConf(conf_file)
    change_single(section, key, value, replace_holder, kc)
    kc.flush()


if __name__ == "__main__":
    try:
        if len(sys.argv) >= 5:
            args = sys.argv
            single_change(args[1], args[2], args[3], args[4])
        else:
            batch_change()
    except Exception as e:
        LOG.exception(e)
