# coding: utf-8
# !/usr/bin/csbs_python
# vim: tabstop=4 shiftwidth=4 softtabstop=4

"""
sync config
"""

import os
import argparse
import configparser
from basesdk import utils

LOG = utils.get_logger("cms_watcher")
_RABBITMQ = "rabbitmq"
_GAUSSDB = "gaussdb"
_KEYSTONE = "keystone"
COMP_LIST = [_RABBITMQ, _GAUSSDB, _KEYSTONE, ]


def parse_args():
    description = " "
    parser = argparse.ArgumentParser(description=description)
    parser.add_argument('--config-file', '-f',
                        nargs='+',
                        dest='config_files',
                        help="Support multiple file",
                        required=True)
    parser.add_argument('--comp', '-c',
                        dest='comp',
                        choices=COMP_LIST,
                        help="",
                        required=True)
    parser.add_argument('--account', '-a', "--username",
                        dest='account',
                        help="the username of authoritation",
                        required=False)
    parser.add_argument('--password', '-p',
                        dest='password',
                        help="the password of authoritation",
                        required=False)
    parser.add_argument("--host", "-host",
                        dest="host",
                        help="the ip of the keystone",
                        default="",
                        required=False)
    parser.add_argument("--region", "-r",
                        dest="region",
                        help="the region for keystone",
                        required=False)
    parser.add_argument('--target', '-t',
                        dest='target',
                        help="deprecated",
                        default="",
                        required=False)
    return parser.parse_args()


def update_config(config_file):
    conf = configparser.ConfigParser()
    conf.read(config_file)
    comp = args.comp
    if comp == _RABBITMQ:
        LOG.info(
            "User is updating the password to access Rabbitmq Server in %s." %
            config_file)
        update_rabbitmq_config(args, conf)
    if comp == _GAUSSDB:
        LOG.info("Usr is updating the pwd access GaussDB in %s", config_file)
        update_db_config(args, conf)
    if comp == _KEYSTONE:
        LOG.info(
            "User is updating the password to access Keystone in %s." %
            config_file)
        update_keystone_config(args, conf)
        file_name = os.path.basename(os.path.normpath(config_file)).partition(
            '.')
        if "ceilometer" in file_name:
            update_ceilometer_center(args, conf)
    with open(config_file, 'w') as fp:
        conf.write(fp)


def update_ceilometer_center(args, conf):
    section = "service_credentials"
    account = 'os_username'
    password = 'os_password'
    os_auth_url = "os_auth_url"
    if args.account:
        conf.set(section, account, args.account)
    if args.password:
        conf.set(section, password, args.password)
    if args.host:
        url = replace_url_with_host(conf.get(section, os_auth_url),
                                    args.host)
        conf.set(section, os_auth_url, url)


def update_rabbitmq_config(args, conf):
    if conf.has_section('oslo_messaging_rabbit'):
        section = "oslo_messaging_rabbit"
    else:
        section = "DEFAULT"
    account = 'rabbit_userid'
    password = 'rabbit_password'
    if args.account:
        conf.set(section, account, args.account)
    if args.password:
        conf.set(section, password, args.password)


def update_db_config(args, conf):
    try:
        section = 'database'
        url_option = 'connection'
        gaussdb_url = conf.get(section, url_option)
    except Exception:
        section = 'DEFAULT'
        url_option = 'database_connection'
        gaussdb_url = conf.get(section, url_option)
    if not (gaussdb_url.count(':') >= 3 and gaussdb_url.count('@') == 1):
        LOG.error("Error updating the password to access GaussDB.")
        return False
    gaussdb_pw = gaussdb_url.partition('@')[0].rpartition(":")[2]
    gaussdb_user = gaussdb_url.partition('@')[0].split(":")[1][2:]
    if args.account:
        gaussdb_url = gaussdb_url.replace(gaussdb_user, args.account, 1)
    if args.password:
        gaussdb_url = gaussdb_url.replace(gaussdb_pw, args.password)
    conf.set(section, url_option, gaussdb_url)


def update_keystone_config(args, conf):
    section = "keystone_authtoken"
    account = 'admin_user'
    password = 'admin_password'
    if args.account:
        conf.set(section, account, args.account)
    if args.password:
        conf.set(section, password, args.password)
    if args.host:
        update_keystone_uri(section, args.host, conf)
    if args.region:
        update_region(args, conf)

    update_keystone_trustee(args, conf)


def update_keystone_trustee(args, conf):
    section = "trustee"
    account = "username"
    password = "password"
    if conf.has_section(section):
        if args.account:
            conf.set(section, account, args.account)
        if args.password:
            conf.set(section, password, args.password)
        if args.host:
            update_keystone_uri(section, args.host, conf)
        if args.region:
            if not (args.account or args.password or args.host):
                conf.set(section, "interface", "internal")
            else:
                conf.set(section, "interface", "admin")


def update_region(args, conf):
    if conf.has_option("DEFAULT", "os_region_name"):
        conf.set("DEFAULT", "os_region_name", args.region)
    if conf.has_option("DEFAULT", "region_name_for_services"):
        conf.set("DEFAULT", "region_name_for_services", args.region)


def replace_url_with_host(url, host):
    host_old = url.partition("//")[2].partition('/')[0]
    if ":" in host_old:
        host_old = host_old.partition(":")[0]
    return url.replace(host_old, host)


def update_keystone_uri(section, host, conf):
    auth_uri = 'auth_uri'
    identity_uri = "identity_uri"
    if conf.has_option(section, "auth_host"):
        conf.set(section, "auth_host", host)

    if conf.has_option(section, auth_uri):
        url = conf.get(section, auth_uri)
        url = replace_url_with_host(url, host)
        conf.set(section, auth_uri, url)
    if conf.has_option(section, "auth_url"):
        url = conf.get(section, "auth_url")
        url = replace_url_with_host(url, host)
        conf.set(section, "auth_url", url)

    if conf.has_option(section,
                       identity_uri):
        url = conf.get(section,
                       identity_uri)
        url = replace_url_with_host(url, host)
        conf.set(section, identity_uri, url)


if __name__ == "__main__":
    # stuck 'sudo' privilege
    args = parse_args()
    if not (args.host or args.account or args.password or args.region):
        LOG.error("nothing to change.")
        exit(1)

    try:
        # update password to *.conf
        for _file in args.config_files:
            LOG.info("User is updating file %s." % os.path.basename(_file))
            if not os.path.isfile(_file):
                LOG.error("The file name is invalid.")
                exit(1)
            update_config(_file)

        LOG.info("Update password successfully.")
        exit(0)
    except Exception:
        LOG.exception("Error updating password.")
        exit(1)
