#!/bin/bash
CURRENT_DIR=$(dirname $(readlink -f $0))
LINE_ROOT=$(dirname $(dirname $(dirname ${CURRENT_DIR})))
source ${LINE_ROOT}/bin/utils.sh
LOG_FILE="/var/log/huawei/dj/services/system/upgradeagent/karbor_upgrade.log"

function backup_files()
{
    backup_or_rollback back ${CURRENT_DIR}/backup_files/package.tar.gz ${CURRENT_DIR}/upgrade_files.conf
    if [ $? -ne 0 ]; then
        log_error "Backup karbor failed."
        return 1
    fi
    log_info "Backup karbor success."
    backup_image ${CURRENT_DIR}/backup_files/ "karbor"
    if [ $? -ne 0 ];then
        log_error "Error to save karbor image."
        return 0
    fi
    log_info "Backup karbor image success."
    return 0
}

function set_upgrade_flag()
{
    should_rollback write karbor
    if [[ $? -ne 0 ]];then
        log_error "Failed to write karbor upgrade flag"
        return 1
    fi
    log_info "Success to write karbor upgrade flag."
    return 0
}

function backup_db_version()
{
    docker exec -u root karborapi bash -c "/usr/bin/kangaroo-manage db_version > /opt/huawei/dj/paas/kangaroo_db_version.bk" >> ${LOG_FILE} 2>&1
    if [[ $? -ne 0 ]]; then
        log_error "Backup kangaroo db version failed."
    else
        log_info "Backup kangaroo db version success."
    fi
}


function execute_cmd_with_retry()
{
    # 由于升级割接数据等为了复用我们的业务代码，需要进入容器执行命令，但是容器
    # 可能出现退出（当前机制不合理导致），因此在这里增加重试来提升可靠性。
    cmd=$1
    docker exec -u root karborapi bash -c "${cmd}" >> ${LOG_FILE} 2>&1
    if [[ $? != 0 ]];then
        log_error "Failed to execute ${cmd} in karbor-api container."
        sleep 15
    else
        return 0
    fi
    docker exec -u root karborprotection bash -c "${cmd}" >> ${LOG_FILE} 2>&1
    if [[ $? != 0 ]];then
        log_error "Failed to execute ${cmd} in karbor-protection container."
        sleep 15
    else
        return 0
    fi
    docker exec -u root karboroperation bash -c "${cmd}" >> ${LOG_FILE} 2>&1
    if [[ $? != 0 ]];then
        log_error "Failed to execute ${cmd} in karbor-operation container."
        exit 1
    fi
}

function migrate_data()
{
    #  用于升级后数据割接
    execute_cmd_with_retry "kangaroo-manage build_protection_summary"
    execute_cmd_with_retry "kangaroo-manage add_image_type"
    execute_cmd_with_retry "kangaroo-manage data_migrate -a update_ckp_item_project_id"
    execute_cmd_with_retry "kangaroo-manage db_import_expired_at"
    execute_cmd_with_retry "kangaroo-manage add_indexes"
    execute_cmd_with_retry "kangaroo-manage convert_trigger_execute_time --force false"
    execute_cmd_with_retry "kangaroo-manage data_migrate -a add_scheduled_operation_id"
}

function upgrade_karbor()
{
    sh ${CURRENT_DIR}/../uninstall.sh
    log_info "uninstall old karbor success."
    sh ${CURRENT_DIR}/../merge.sh
    if [ $? -ne 0 ];then
        log_error "Merge new karbor failed."
        return 1
    fi
    log_info "Merge new karbor success."
    sh ${CURRENT_DIR}/../install.sh
    if [ $? -ne 0 ];then
        log_error "Install new karbor failed."
        return 1
    fi
    log_info "Install new karbor success."
    sh ${CURRENT_DIR}/../start.sh
    if [ $? -ne 0 ];then
        log_error "Start new karbor failed."
        return 1
    fi
    log_info "Start new karbor success."
}

function main()
{
    set_upgrade_flag; CHECK_RESULT $? "Failed to execute set_upgrade_flag."
    backup_files; CHECK_RESULT $? "Failed to execute backup_files."
    backup_db_version; CHECK_RESULT $? "Failed to execute backup_db_version."
    upgrade_karbor; CHECK_RESULT $? "Failed to execute merge_new_image."
    migrate_data; CHECK_RESULT $? "Failed to execute convert_database."
}

main
log_info "Upgrade karbor success."
exit 0