#!/usr/bin/csbs_python

"""
Command-line interface to the CPS service API.
"""
import requests
import sys
import os
import json
import getpass
from basesdk import utils
from checkArb import CheckArb
from kmc import kmc
from sys import argv

kmc_api = kmc.API()
arb_conf = "/usr/local/bin/ha/arbitrationtool/conf/arb.properties"
DEF_PORT = '2379'
DEF_USER = 'arbiterClient'
DEF_CERT = "/home/djmanager/ca.crt"
CUR_DIR = os.path.dirname(os.path.abspath(__file__))


class ConfigArb(object):
    def __init__(self):
        super(ConfigArb, self).__init__()

    @staticmethod
    def write_conf(key, value):
        run_cmds = [['cat', arb_conf], ['grep', key]]
        code, out = utils.run_cmds(run_cmds)
        if out and len(out.split('\n')) != 1:
            raise Exception("Read %s failed" % key)
        if not out:
            run_cmds = ['sed', '-i', '1i %s=%s' % (key, value), arb_conf]
        else:
            run_cmds = ['sed', '-i', 's/%s/%s=%s/g' %
                        (out, key, value), arb_conf]
        code, out = utils.run_cmd(run_cmds)
        if code:
            raise Exception("Config %s failed" % key)

    @staticmethod
    def config_other_node(node_ip, parameters=""):
        if not node_ip:
            raise Exception("Node not exist")
        """send quest to sync agent to config arb."""
        url = 'https://[%s]:28688/scagent/v1/command' % node_ip
        body = {
            "command": {
                "name": os.path.join(CUR_DIR, "configArb.sh"),
                "service": "karbor",
                "parameters": parameters
            }
        }
        data = json.dumps(body)
        headers = {
            'Content-Type': 'application/json'
        }
        kwargs = {
            'verify': "/opt/huawei/dj/DJSecurity/server-ca/ca-cert.pem",
            "cert": (
                "/opt/huawei/dj/DJSecurity/server-cert/karbor/karbor-cert.pem",
                "/opt/huawei/dj/DJSecurity//server-cert/karbor/"
                "karbor-key.pem"),
            'timeout': 180
        }
        ret = requests.request('PUT', url, headers=headers, data=data,
                               **kwargs)
        if ret.status_code <= 300 \
                and not ret.json().get('command').get('code'):
            return
        raise Exception("Config on node(%s) failed(%s)." % (node_ip, ret.text))

    @staticmethod
    def ip_check(value):
        code, out = utils.run_cmd(['karbor_ping', '-c', '3', value])
        if code:
            raise Exception("Check ip(%s) failed." % value)

    @staticmethod
    def show_paramer():
        print("csbs_python %s disable" % argv[0])
        print("csbs_python %s <ips> <local_dc> <remote_dc>" % argv[0])
        sys.exit(1)

    def disable_arb(self):
        if len(argv) == 2:
            self.config_other_node(utils.get_ha_peer_node(), "disable remote")
        if len(argv) > 2 and argv[2] != "remote":
            self.show_paramer()
        self.write_conf('arb.enable', 'false')
        print("Disable arbitration service successful.")
        sys.exit(0)

    def check_and_config(self, ips, local, remote, pwd):
        ip_list = list()
        for ip in ips.split(','):
            self.ip_check(ip)
            ip_list.append("[%s]:%s" % (ip, DEF_PORT))
        CheckArb().query_arb(','.join(ip_list), DEF_USER, pwd, local, remote)
        self.write_conf('arb.ips', ','.join(ip_list))
        self.write_conf('arb.localDC', local)
        self.write_conf('arb.remoteDC', remote)
        self.write_conf('arb.username', DEF_USER)
        self.write_conf('arb.password', kmc_api.encrypt(0, pwd))
        self.write_conf('arb.enable', 'true')
        print("Config arbitration service successful.")
        sys.exit(0)

    def enable_arb(self):
        if len(argv) < 4:
            self.show_paramer()
        if len(argv) == 4:
            pwd = getpass.getpass("Enter password of user(%s):\n" % DEF_USER)
            param = "%s %s %s %s" % (argv[1], argv[3], argv[2], pwd)
            self.config_other_node(utils.get_ha_peer_node(), param)
        else:
            pwd = argv[4]
        self.check_and_config(argv[1], argv[2], argv[3], pwd)

    def main(self):
        if len(argv) < 2:
            self.show_paramer()
        if argv[1] == "disable":
            self.disable_arb()
        else:
            self.enable_arb()


if __name__ == '__main__':
    ConfigArb().main()
