#! /bin/bash
RABBITMQCTL_PATH="/usr/local/lib/rabbitmq/sbin/rabbitmqctl"
RABBITMQ_CFG_PATH="/usr/local/lib/rabbitmq/etc/rabbitmq/rabbitmq.config"
RABBITMQ_ENV_PATH="/usr/local/lib/rabbitmq/etc/rabbitmq/rabbitmq-env.conf"
RABBITMQ_COOKIE_PATH="/home/openstack/.erlang.cookie"
RABBITMQ_MODIFY_PWD="/usr/local/lib/rabbitmq/HW_FS_Extend/etc/modify_password.py"
RABBITMQ_REPLACE_PWD="/usr/local/lib/rabbitmq/HW_FS_Extend/etc/replace_password.py"

function log_info()
{
    logger -id -n localhost -p local1.info -t "rabbitmq" "$1"
}

function log_error()
{
    logger -id -n localhost -p local1.error -t "rabbitmq" "$1"
}

function delete_pwd_from_config() {
    csbs_python ${RABBITMQ_REPLACE_PWD} encrypt
}

function check_result_and_safe_exit()
{
    if [[ $1 -ne 0 ]];then
        log_error "$2"
        delete_pwd_from_config
        exit $1
    fi
}

function encrypt_private_key()
{
    csbs_python ${RABBITMQ_REPLACE_PWD} decrypt
    if [[ $? -ne 0 ]];then
        log_error "Decrypt private key password failed."
        exit 1
    fi
}

function init_rabbit_user()
{
    # ensure vhosts is OK
    ${RABBITMQCTL_PATH} add_vhost "/"
    check_result_and_safe_exit "$?" "ERROR | add rabbitmq vhosts failed."
    log_info "INFO | The rabbitmq vhosts \"/\" is ok."

    # ensure user is OK
    ${RABBITMQCTL_PATH} add_user rabbit ''
    ret=$? && [[ ${ret} -eq 70 ]] && ret=0
    check_result_and_safe_exit "${ret}" "ERROR | add rabbitmq user failed."
    log_info "INFO | The user rabbit is OK."

    # ensure user administrator is OK
    ${RABBITMQCTL_PATH} set_user_tags rabbit administrator
    check_result_and_safe_exit "$?" "ERROR | add rabbit's administrator failed."
    log_info "INFO | The user rabbit's administrator is OK."

    # ensure rabbit permissions is OK
    ${RABBITMQCTL_PATH} set_permissions rabbit ".*" ".*" ".*"
    check_result_and_safe_exit "$?" "ERROR | add abbitmq rabbit permissions failed."
    log_info "INFO | The rabbitmq rabbit permissions \".*\" is OK."

    # ensure user password is OK
    csbs_python ${RABBITMQ_MODIFY_PWD}
    check_result_and_safe_exit "$?" "ERROR | modify_password rabbit failed."
    log_info "INFO | modify_password rabbit success."
    return 0
}

function init_server_params()
{
    memtotal=`cat /proc/meminfo |grep MemTotal |awk '{print $2}'`
    memtotalMB=`expr ${memtotal} / 1024`
    frac=`awk 'BEGIN{x=8192/'${memtotalMB}'*'1.3'; if (x < 1) {printf "%.4f", x} else {printf "0.4"}}'`

    ${RABBITMQCTL_PATH} set_vm_memory_high_watermark "${frac}"
    check_result_and_safe_exit "$?" "ERROR | rabbitmqctl set_vm_memory_high_watermark failed."
    log_info "INFO | set_vm_memory_high_watermark success."

    ${RABBITMQCTL_PATH} set_policy ha-allqueue "^"  '{"ha-mode":"all","ha-sync-mode":"automatic"}' --priority 1 --apply-to exchanges
    check_result_and_safe_exit "$?" "ERROR | rabbitmqctl set_policy ha-allqueue failed."
    log_info "INFO | set_policy ha-allqueue success."

    ${RABBITMQCTL_PATH} set_policy ha_length_ttl_meter "^metering.sample$" '{"ha-mode":"all","ha-sync-mode":"automatic", "max-length":720000, "message-ttl": 86400000}' --priority 1 --apply-to queues
    check_result_and_safe_exit "$?" "ERROR | rabbitmqctl set_policy ha_length_ttl_meter failed."
    log_info "INFO | set_policy ha_length_ttl_meter success."

    ${RABBITMQCTL_PATH} set_policy ha_length_ttl '^(?!metering\.sample).+' '{"ha-mode":"all","ha-sync-mode":"automatic", "max-length":59600, "message-ttl": 86400000}' --priority 1 --apply-to queues
    check_result_and_safe_exit "$?" "ERROR | rabbitmqctl set_policy ha_length_ttl failed."
    log_info "INFO | set_policy ha_length_ttl success."
    return 0
}

function join_cluster()
{
    encrypt_private_key

    # ensure rabbit cluster is running
    ${RABBITMQCTL_PATH} -n $1 status
    check_result_and_safe_exit "$?" "ERROR | $1 is not running."
    log_info "INFO | $1 is running."

    # ensure rabbit app is stop
    ${RABBITMQCTL_PATH} stop_app
    check_result_and_safe_exit "$?" "ERROR | stop rabbit app failed."
    log_info "INFO | stop rabbit app success."

    # ensure rabbit cluster_status is ok
    ${RABBITMQCTL_PATH} reset
    check_result_and_safe_exit "$?" "ERROR | reset rabbit app failed."
    log_info "INFO | reset rabbit app success."

    # ensure rabbit join_cluster is ok
    ${RABBITMQCTL_PATH} join_cluster $1
    check_result_and_safe_exit "$?" "ERROR | join_cluster $1 failed."
    log_info "INFO | join_cluster $1 app success."

    # ensure rabbit app is ok
    ${RABBITMQCTL_PATH} start_app
    check_result_and_safe_exit "$?" "ERROR | start rabbit app failed."
    log_info "INFO | start rabbit app success."

    delete_pwd_from_config
}

function init_cluser_status()
{
    # ensure rabbit app is stop
    ${RABBITMQCTL_PATH} stop_app
    check_result_and_safe_exit "$?" "ERROR | stop rabbit app failed."
    log_info "INFO | stop rabbit app success."

    # ensure rabbit cluster_status is ok
    ${RABBITMQCTL_PATH} reset
    check_result_and_safe_exit "$?" "ERROR | reset rabbit app failed."
    log_info "INFO | reset rabbit app success."

    # ensure rabbit app is ok
    ${RABBITMQCTL_PATH} start_app
    check_result_and_safe_exit "$?" "ERROR | start rabbit app failed."
    log_info "INFO | start rabbit app success."

}


function waitServerStarted()
{
    for i in $(seq 450); do
        if [[ ! -f "${RABBITMQ_COOKIE_PATH}" ]]; then
            log_info "INFO | cookie is not yet created"
            sleep 10
            continue
        fi
        ${RABBITMQCTL_PATH} status >/dev/null 2>&1
        [[ $? -eq 0 ]] && return 0
        sleep 10
    done
    check_result_and_safe_exit "1" "ERROR | rabbitmq is not running."
}

function restart_rabbitmq()
{
    log_info "Try to start the server, even if it is already started"
    encrypt_private_key
    bash /usr/local/lib/rabbitmq/sbin/rabbitmq-plugins disable rabbitmq_management >> /dev/null 2>&1
    bash /usr/local/lib/rabbitmq/sbin/rabbitmq-server -detached >> /dev/null 2>&1

    log_info "Wait rabbitmq running."
    waitServerStarted

    log_info "Init rabbitmq user info."
    sleep 10
    init_rabbit_user

    log_info "Init rabbitmq server params."
    sleep 10
    init_server_params

    log_info "Init rabbitmq cluser status."
    sleep 10
    init_cluser_status

    delete_pwd_from_config
    log_info "Restart rabbitmq success."
    return 0
}


