#!/bin/bash
if [ ${UID} -ne 0 ];then
  sudo $(dirname $(readlink -f $0))/$(basename $(readlink -f $0)) $@;exit $?
fi

source /opt/huawei/dj/inst/bin/utils.sh
ZOOKEEPER_SERVER="/usr/local/bin/zookeeper/zookeeper/bin/zkServer.sh"
ZOOKEEPER_LOG_PATH="/var/log/huawei/dj/services/system/zookeeper"
ZOOKEEPER_DATA_PATH="/opt/huawei/dj/etc/zookeeper/data"
LOG_FILE="${ZOOKEEPER_LOG_PATH}/zookeeperControl.log"
SERV_IMG_VERSION=$(read_microservice_ver /opt/huawei/dj/versions/zookeeper.ver)
ZOOKEEPER_IMG="zookeeper:${SERV_IMG_VERSION}"
ServiceState=
component="zookeeper"

function showhelp()
{
    echo "Usage: $prog :"
    echo "  -S ServiceState <RUN|STOP|CHANGE>"
    echo "  -A RESTART"
    echo "  -CS CompontState"
    echo "  -h show this help"
    exit 1
}

function args_err()
{
    echo -e "${prog}: illegal option"
    showhelp
    exit 1
}

function stopHeartBeat()
{
    heartBeat STOP zookeeper
    if [[ $? -ne 0 ]]; then
        log_error "Unregister heartbeat of zookeeper failed."
        return 1
    fi
    log_info "Unregister heartbeat of zookeeper success."
}

function heartBeatReg()
{
    heartBeat START zookeeper
    if [[ $? -ne 0 ]]; then
        log_error "Register heartbeat of zookeeper failed."
        return 1
    fi
    log_info "Register heartbeat of zookeeper success."
}


function start_zookeeper()
{
    log_info "Begin to start ${component}...."
    #if has container instance, start it or skip
    ret_info=`docker ps -q -f name=${component}`
    CHECK_RESULT $? "Get docker info of ${component} failed."
    if [ ! -z "${ret_info}" ];then
        log_info "Container ${component} is running, skip."
        return 0
    fi
    ret_info=`docker ps -a -q -f name=${component}`
    CHECK_RESULT $? "Get docker info of ${component} failed."
    if [ ! -z  "${ret_info}" ];then
        log_info "Try to start ${component}..."
        run_container ${ret_info}
        sleep 10
        new_info=`docker ps -q -f name=${component}`
        if [ ! -z "${new_info}" ];then
            log_info "Start container ${component} successfully."
            return 0
        fi
        docker rm -f ${ret_info} >/dev/null 2>&1
    fi
    start_instance
}

function run_container()
{
    local container_id=$1
    local retry=0
    while [ ${retry} -lt 3 ]
    do
        docker start ${container_id}
        if [ "$?" == "0" ];then
            log_info "start ${component} container successfully."
            return 0
        fi
        CHECK_RESULT $? "start ${component} container failed, retry."
        c_id=`docker ps -a -q --no-trunc -f name=${component}`
        if [ "$?" == "0" ] && [ -n "${c_id}" ];then
            if [ -d "/run/containerd/${c_id}" ];then
                rm -fr "/run/containerd/${c_id}"
            fi
        fi
        retry=$(( ${retry} + 1 ))
    done
}


function start_instance()
{
    local docker_version=`docker -v|awk -F'[ ,]' '{print $3}'|awk -F'.' '{print $2}'`
    CHECK_RESULT $? "Get docker version failed."
    docker_version=`expr ${docker_version} + 0`
    CHECK_RESULT $? "Docker version [${docker_version}] is invalid."
    log_info "Docker version is ${docker_version}."
    local docker_security_opt="--memory 1024m --cpu-shares 1024  --security-opt=no-new-privileges --pids-limit 1024 --ulimit nofile=2048:2048"
    docker run -d --net=host --name ${component} \
        -v ${ZOOKEEPER_LOG_PATH}:/var/log/huawei/dj/zookeeper \
        -v ${ZOOKEEPER_DATA_PATH}:/usr/local/bin/zookeeper/zookeeper/data \
        -v /var/log/huawei/dj/services/system/kmc:/var/log/huawei/dj/services/system/kmc \
        -v /etc/localtime:/etc/localtime:ro \
        -v /usr/bin/get_info.py:/usr/bin/get_info.py:ro \
        -v /opt/huawei/dj/cfg:/opt/huawei/dj/cfg:ro \
        -v /etc/hosts:/etc/hosts \
        -v /opt/huawei/dj/paas:/opt/huawei/dj/paas:ro \
        -v /opt/fusionplatform/data/openstack:/opt/fusionplatform/data/openstack:ro \
        ${docker_security_opt} \
        "${ZOOKEEPER_IMG}" bash /usr/local/bin/zookeeper/zookeeper/bin/checkState.sh >/dev/null 2>&1
    CHECK_RESULT $? "create container failed"
    log_info "container create successfully."
}

function component_run()
{
    log_info "${component}Control: ======component_run begin"
    start_zookeeper
    heartBeatReg
    log_info "${component}Control: ======component_run end"
}

function component_stop()
{
    log_info "${component}Control: ======component_stop begin"
    stopHeartBeat
    ret_info=`docker ps -a -q -f name=${component}`
    CHECK_RESULT $? "Get docker info of ${component} failed."
    if [ -z "${ret_info}" ];then
        log_info "There is no container instance, skip."    
    else
        docker stop ${component}
        CHECK_RESULT $? "Stop component ${component} failed."    
    fi
    log_info "Stop component ${component} successfully."
    log_info "${component}Control: ======component_stop end"
}

function component_restart()
{
    log_info "${component}Control: ======component_restart begin"
    component_stop
    component_run
    log_info "${component}Control: ======component_end end"
}

function checkRun()
{
    ret_info=`docker ps -q -f name=${component}`
    if [ $? -ne 0 -o -z "${ret_info}" ];then
        log_error "Query status of ${component} failed."
        exit 1
    fi
    zk_stus=$(docker exec ${component} bash ${ZOOKEEPER_SERVER} status | grep -w "^Mode:" | awk '{print $NF}')
    if [[ "${zk_stus}" == "standalone" ]];then
        exit 0
    fi
    if [[ "${zk_stus}" == "leader" ]];then
        exit 2
    fi
    if [[ "${zk_stus}" == "follower" ]];then
        exit 3
    fi
    exit 1
}

function chown_chmod_log_dir()
{
    mkdir -p ${ZOOKEEPER_LOG_PATH} ${ZOOKEEPER_DATA_PATH}/dev/null 2>&1
    chown -h openstack:openstack  ${ZOOKEEPER_LOG_PATH} ${ZOOKEEPER_DATA_PATH}>/dev/null 2>&1
    chmod 770 ${ZOOKEEPER_DATA_PATH}>/dev/null 2>&1
    chmod 750 ${ZOOKEEPER_LOG_PATH} >/dev/null 2>&1
}

log_info "INFO | #################begin exec zookeeperControl########################."
chown_chmod_log_dir
while true; do
    case "$1" in
        -S) ServiceState=$2;shift 2;break;;
        -CS) CompStatus=$2;shift 2;break;;
        -A) Action=$2;shift 2;break;;
        -h) showhelp $0;break;;
        *) shift; break;;
    esac
done
if [ "RESTART" = "${Action}" ]; then
    component_restart
elif [ "RUN" = "${ServiceState}" ]; then
    component_run
elif [ "STOP" = "${ServiceState}" ]; then
    component_stop
elif [ "CHANGE" = "${ServiceState}" ]; then
    component_restart
elif [ "CompontState" = "${CompStatus}" ];then
    checkRun
else
    args_err
fi
exit 0
