//+build linux

package cryptoapi

/*
#cgo CFLAGS: -fstack-protector-all -D_FORTIFY_SOURCE=2 -O2 -Wformat=2 -Wfloat-equal -Wshadow -I/usr/local/seccomponent/include
#cgo LDFLAGS: -L/usr/local/seccomponent/lib -lsc-secrypto
#include <stdlib.h>
#include <stdio.h>
#include "sc_cryptoapi.h"
#include "sc_errcode.h"
#include "sc_type.h"
*/
import "C"
import (
	"errors"
	"strconv"
	"unsafe"
)

const ALG_DEFAULT = 0
const ALG_AES256_CBC = 7
const ALG_AES128_GCM = 8
const ALG_AES256_GCM = 9
const ALG_SM4_CBC = 10
const ALG_SM4_CTR = 11
const ALG_SHA256 = 1028
const ALG_SM3 = 1031
const ALG_HMAC_SHA256 = 2052
const ALG_HMAC_SM3 = 2055
const ALG_PBKDF2_HMAC_SHA256 = 3076
const ALG_PBKDF2_HMAC_SM3 = 3079

const DEFAULT_DOMAIN_ID = 0
//256 bit / 8 =32 byte
const DEFAULT_KEY_LEN = 32
const KEYID_AUTO_GET = 0
const SEC_TRUE = 1
const SEC_FALSE = 0
const SEC_SUCCESS = 0

func Initialize(confFile string) (err error) {
	var pconf = C.CString(confFile)
	defer C.free(unsafe.Pointer(pconf))
	ret := C.SCC_Initialize(pconf)
	if ret == SEC_SUCCESS {
		return
	} else if ret == 1001 {
		err = errors.New("Initialize Failed. Error=ERR_NO_INIT,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 1002 {
		err = errors.New("Initialize Failed. Error=ERR_REPEAT_INIT,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 1003 {
		err = errors.New("Initialize Failed. Error=ERR_INIT_SYS_LOG_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 1004 {
		err = errors.New("Initialize Failed. Error=ERR_INIT_LOG_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 1005 {
		err = errors.New("Initialize Failed. Error=ERR_INIT_CFG_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 2001 {
		err = errors.New("Initialize Failed. Error=ERR_INIT_KMC_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 2002 {
		err = errors.New("Initialize Failed. Error=ERR_INIT_DOMAIN_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 2003 {
        // Ӳʼʧܣʼɹʹ
	    err = errors.New("Initialize Failed. Error=ERR_INIT_KMC_HARD_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else {
		err = errors.New("Initialize Failed. ErrCode=" + strconv.Itoa(int(ret)))
	}
	return
}

func Finalize() (err error) {
	ret := C.SCC_Finalize()
	if ret != SEC_SUCCESS {
		err = errors.New("Finalize Failed. ErrCode=" + strconv.Itoa(int(ret)))
	}
	return
}

func EncryptByAlgorithmDomain(algorithmId int, domainId int, plainText string) (cipherText string, err error) {
	var pcipher *C.char
	var cipherLen C.int
	var plainLen = C.int(len(plainText))
	var pplainText = C.CString(plainText)
	defer C.free(unsafe.Pointer(pplainText))
	ret := C.SCC_EncryptByAlgorithmDomain(C.int(algorithmId),C.int(domainId), pplainText, plainLen, &pcipher, &cipherLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call EncryptByAlgorithmDomain failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pcipher))
		cipherText = C.GoString(pcipher)
		return
	}
}

func EncryptByAlgorithm(algorithmId int, plainText string) (cipherText string, err error) {
	cipherText, err = EncryptByAlgorithmDomain(algorithmId,DEFAULT_DOMAIN_ID, plainText)
    return
}

func EncryptByDomain(domainId int, plainText string) (cipherText string, err error) {
	cipherText, err = EncryptByAlgorithmDomain(ALG_DEFAULT,domainId, plainText)
    return
}

func Encrypt(plainText string) (cipherText string, err error) {
	cipherText, err = EncryptByAlgorithmDomain(ALG_DEFAULT,DEFAULT_DOMAIN_ID, plainText)
	return
}

func DecryptByDomain(domainId int, cipherText string) (plainText string, err error) {
	var pplain *C.char
	var plainLen C.int
	var ciperLen = C.int(len(cipherText))
	var pcipherText = C.CString(cipherText)
	defer C.free(unsafe.Pointer(pcipherText))
	ret := C.SCC_DecryptByDomain(C.int(domainId), pcipherText, ciperLen, &pplain, &plainLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call DecryptByDomain failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pplain))
		plainText = C.GoString(pplain)
		return
	}
}

func Decrypt(cipherText string) (plainText string, err error) {
	plainText, err = DecryptByDomain(DEFAULT_DOMAIN_ID, cipherText)
	return
}

func SetDomain(domainId int) {
	C.SCC_SetDomain(C.int(domainId))
	return
}

func ResetDomain() {
	C.SCC_ResetDomain()
	return
}

func ReEncryptByAlgorithmDomain(algorithmId int, domainId int, oldcipherText string) (newcipherText string, err error) {
	var pnewcipher *C.char
	var newcipherLen C.int
	var oldciperLen = C.int(len(oldcipherText))
	var poldcipherText = C.CString(oldcipherText)
	defer C.free(unsafe.Pointer(poldcipherText))
	ret := C.SCC_EncryptByAlgorithmDomain(C.int(algorithmId),C.int(domainId), poldcipherText, oldciperLen, &pnewcipher, &newcipherLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call ReEncryptByAlgorithmDomain failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pnewcipher))
		newcipherText = C.GoString(pnewcipher)
		return
	}
}

func ReEncryptByAlgorithm(algorithmId int, oldcipherText string) (newcipherText string, err error) {
	newcipherText, err = ReEncryptByAlgorithmDomain(algorithmId,DEFAULT_DOMAIN_ID, oldcipherText)
    return
}

func ReEncryptByDomain(domainId int, oldcipherText string) (newcipherText string, err error) {
	newcipherText, err = ReEncryptByAlgorithmDomain(ALG_DEFAULT,domainId, oldcipherText)
    return
}

func ReEncrypt(oldcipherText string) (newcipherText string, err error) {
	newcipherText, err = ReEncryptByAlgorithmDomain(ALG_DEFAULT,DEFAULT_DOMAIN_ID, oldcipherText)
	return
}

func ActiveDomainNewKey(domainId int) (err error) {
	ret := C.SCC_ActiveNewKey(C.int(domainId))
	if ret != SEC_SUCCESS {
		err = errors.New("call ActiveNewKey failed. ErrCode=" + strconv.Itoa(int(ret)))

	}
	return
}

func ActiveNewKey() (err error) {
	err = ActiveDomainNewKey(DEFAULT_DOMAIN_ID)
	return
}

func GenSecKeyByLen(keyLen int) (keyBase64 string, err error) {
	var pkey *C.char
	var keyLenResult C.int
	ret := C.SCC_GenSecKey(C.int(keyLen), &pkey, &keyLenResult)
	if ret != SEC_SUCCESS {
		err = errors.New("call GenSecKey failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pkey))
		keyBase64 = C.GoString(pkey)
		return
	}
}

func GenSecKey() (keyBase64 string, err error) {
	keyBase64, err = GenSecKeyByLen(DEFAULT_KEY_LEN)
	return
}

func RegisterKeyById(domainId int, keyId int, keyBase64 string) (err error) {
	var keyLen = C.int(len(keyBase64))
	var pkeyBase64 = C.CString(keyBase64)
	defer C.free(unsafe.Pointer(pkeyBase64))
	ret := C.SCC_RegisterKey(C.int(domainId), C.int(keyId), pkeyBase64, keyLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call RegisterKey failed. ErrCode=" + strconv.Itoa(int(ret)))
	}
	return
}

func RegisterKey(domainId int, keyBase64 string) (err error) {
	err = RegisterKeyById(domainId, KEYID_AUTO_GET, keyBase64)
	return
}

func GetDomainKeyByID(domainId int, keyId int) (keyBase64 string, err error) {
	var pkey *C.char
	var keyLen C.int
	ret := C.SCC_GetKeyByID(C.int(domainId), C.int(keyId), &pkey, &keyLen, SEC_TRUE)
	if ret != SEC_SUCCESS {
		err = errors.New("call GetKeyByID failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pkey))
		keyBase64 = C.GoString(pkey)
		return
	}
}

func GetActiveKey() (keyBase64 string, err error) {
	keyBase64, err = GetDomainKeyByID(DEFAULT_DOMAIN_ID, KEYID_AUTO_GET)
	return
}

func GetDomainActiveKey(domainId int) (keyBase64 string, err error) {
	keyBase64, err = GetDomainKeyByID(domainId, KEYID_AUTO_GET)
	return
}

func GetMaxMkID(domainId int) (keyId int, err error) {
	var keyIdResult C.int
	ret := C.SCC_GetMaxMkID(C.int(domainId), &keyIdResult)
	if ret != SEC_SUCCESS {
		err = errors.New("call GetMaxMkID failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		keyId = int(keyIdResult)
		return
	}
}

func IsExistMkID(domainId int, keyId int) (existInd bool, err error) {
	var existResult C.int
	ret := C.SCC_IsExistMkID(C.int(domainId), C.int(keyId), &existResult)
	if ret != SEC_SUCCESS {
		err = errors.New("call IsExistMkID failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		existInd = true
		if existResult == SEC_FALSE {
			existInd = false
		}
		return
	}
}

func GetPublicKey() (keyBase64 string, err error) {
	var pPublickey *C.char
	ret := C.SCC_GetPublicKey(&pPublickey)
	if ret != SEC_SUCCESS {
		err = errors.New("call GetPublicKey failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pPublickey))
		keyBase64 = C.GoString(pPublickey)
		return
	}
}

func PublicKeyEncrypt(plainText string, publicKey string) (cipherText string, err error) {
	var pcipher *C.char
	var cipherLen C.int
	var plainLen = C.int(len(plainText))
	var pplainText = C.CString(plainText)
	var ppublicKey = C.CString(publicKey)
	defer C.free(unsafe.Pointer(pplainText))
	defer C.free(unsafe.Pointer(ppublicKey))
	ret := C.SCC_PublicKeyEncrypt(pplainText, plainLen, ppublicKey, &pcipher, &cipherLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call PublicKeyEncrypt failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pcipher))
		cipherText = C.GoString(pcipher)
		return
	}
}

func PrivateKeyDecrypt(cipherText string) (plainText string, err error) {
	var pplain *C.char
	var plainLen C.int
	var ciperLen = C.int(len(cipherText))
	var pcipherText = C.CString(cipherText)
	defer C.free(unsafe.Pointer(pcipherText))
	ret := C.SCC_PrivateKeyDecrypt(pcipherText, ciperLen, &pplain, &plainLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call PrivateKeyDecrypt failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pplain))
		plainText = C.GoString(pplain)
		return
	}
}

func ResetRSAKey() (err error) {
	ret := C.SCC_ResetRSAKey()
	if ret != SEC_SUCCESS {
		err = errors.New("call ResetRSAKey failed. ErrCode=" + strconv.Itoa(int(ret)))
	}
	return
}

func CreateHashCode(algorithmId int, plainText string) (hashCode string, err error) {
	var phashCode *C.char
	var hashCodeLen C.int
	var plainLen = C.int(len(plainText))
	var pplainText = C.CString(plainText)
	defer C.free(unsafe.Pointer(pplainText))
	ret := C.SCC_CreateHashCode(C.int(algorithmId), pplainText, plainLen, &phashCode, &hashCodeLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call CreateHashCode failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(phashCode))
		hashCode = C.GoString(phashCode)
		return
	}
}

func CheckHashCode(algorithmId int, plainText string, hashCode string) (checkRet bool, err error) {
	var checkResult C.int
	var plainTextLen = C.int(len(plainText))
	var pplainText = C.CString(plainText)
	var hashCodeLen = C.int(len(hashCode))
	var phashCode = C.CString(hashCode)
	defer C.free(unsafe.Pointer(pplainText))
	defer C.free(unsafe.Pointer(phashCode))

	ret := C.SCC_CheckHashCode(C.int(algorithmId), pplainText, plainTextLen, phashCode, hashCodeLen, &checkResult)
	if ret != SEC_SUCCESS {
		err = errors.New("call CheckHashCode failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		checkRet = true
		if checkResult == SEC_FALSE {
			checkRet = false
		}
		return
	}
}

func CreateHmacCode(algorithmId int, domainId int, plainText string) (hmacCode string, err error) {
	var phmacCode *C.char
	var hmacCodeLen C.int
	var plainLen = C.int(len(plainText))
	var pplainText = C.CString(plainText)
	defer C.free(unsafe.Pointer(pplainText))
	ret := C.SCC_CreateHmacCode(C.int(algorithmId), C.int(domainId), pplainText, plainLen, &phmacCode, &hmacCodeLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call CreateHmacCode failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(phmacCode))
		hmacCode = C.GoString(phmacCode)
		return
	}
}

func CheckHmacCode(domainId int, plainText string, hmacCode string) (checkRet bool, err error) {
	var checkResult C.int
	var plainTextLen = C.int(len(plainText))
	var pplainText = C.CString(plainText)
	var hmacCodeLen = C.int(len(hmacCode))
	var phmacCode = C.CString(hmacCode)
	defer C.free(unsafe.Pointer(pplainText))
	defer C.free(unsafe.Pointer(phmacCode))

	ret := C.SCC_CheckHmacCode(C.int(domainId), pplainText, plainTextLen, phmacCode, hmacCodeLen, &checkResult)
	if ret != SEC_SUCCESS {
		err = errors.New("call CheckHmacCode failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		checkRet = true
		if checkResult == SEC_FALSE {
			checkRet = false
		}
		return
	}
}

func CreateFileHmacCode(algorithmId int, domainId int, filePath string) (hmacCode string, err error) {
	var phmacCode *C.char
	var hmacCodeLen C.int
	var pfilePath = C.CString(filePath)
	defer C.free(unsafe.Pointer(pfilePath))
	ret := C.SCC_CreateFileHmacCode(C.int(algorithmId), C.int(domainId), pfilePath, &phmacCode, &hmacCodeLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call CreateFileHmacCode failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(phmacCode))
		hmacCode = C.GoString(phmacCode)
		return
	}
}

func CheckFileHmacCode(domainId int, filePath string, hmacCode string) (checkRet bool, err error) {
	var checkResult C.int
	var pfilePath = C.CString(filePath)
	var hmacCodeLen = C.int(len(hmacCode))
	var phmacCode = C.CString(hmacCode)
	defer C.free(unsafe.Pointer(pfilePath))
	defer C.free(unsafe.Pointer(phmacCode))

	ret := C.SCC_CheckFileHmacCode(C.int(domainId), pfilePath, phmacCode, hmacCodeLen, &checkResult)
	if ret != SEC_SUCCESS {
		err = errors.New("call CheckFileHmacCode failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		checkRet = true
		if checkResult == SEC_FALSE {
			checkRet = false
		}
		return
	}
}

func ProtectPwd(algorithmId int, password string) (cipherText string, err error) {
	var pcipherText *C.char
	var cipherLen C.int
	var plainLen = C.int(len(password))
	var pplainText = C.CString(password)
	defer C.free(unsafe.Pointer(pplainText))
	ret := C.SCC_ProtectPwd(C.int(algorithmId), pplainText, plainLen, &pcipherText, &cipherLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call ProtectPwd failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		defer C.free(unsafe.Pointer(pcipherText))
		cipherText = C.GoString(pcipherText)
		return
	}
}

func VerifyPwd(password string, cipherText string) (checkRet bool, err error) {
	var checkResult C.int
	var plainTextLen = C.int(len(password))
	var pplainText = C.CString(password)
	var cipherLen = C.int(len(cipherText))
	var pcipherText = C.CString(cipherText)
	defer C.free(unsafe.Pointer(pplainText))
	defer C.free(unsafe.Pointer(pcipherText))

	ret := C.SCC_VerifyPwd(pplainText, plainTextLen, pcipherText, cipherLen, &checkResult)
	if ret != SEC_SUCCESS {
		err = errors.New("call VerifyPwd failed. ErrCode=" + strconv.Itoa(int(ret)))
		return
	} else {
		checkRet = true
		if checkResult == SEC_FALSE {
			checkRet = false
		}
		return
	}
}

func UpdateRootKey() (err error) {
    ret := C.SCC_UpdateRootKey()
    if ret == SEC_SUCCESS {

    } else if ret == 2005 {
        err = errors.New("Update hard root key failed. ErrCode=" + strconv.Itoa(int(ret)))
    } else {
        err = errors.New("Update root key failed. ErrCode=" + strconv.Itoa(int(ret)))
    }
    return
}
