#!/bin/bash
set +x

G_CURRENT_PATH="$(cd "$(dirname "$0")" && pwd)"

source ${G_CURRENT_PATH}/common.sh

G_USER_NAME="fsp"
G_CURRENT_ROLE=1  #0:eBackup driver, 1:not eBackup driver
G_UPGRADE_TEMP_DIR="/tmp/upgrade"
G_NODES_PWD_FILE="${G_UPGRADE_TEMP_DIR}/servers.txt"
G_CURRENT_NODE_FILE="${G_UPGRADE_TEMP_DIR}/localhost.txt"
G_NODE_PWD_TMP_FILE="${G_UPGRADE_TEMP_DIR}/server_tmp_info.txt"
G_NODES_TMP_FILE="${G_UPGRADE_TEMP_DIR}/nodes.txt"
G_HCP_MAIN_PATH=""
G_DRIVER_VERSION_FILE_PATH=""

G_CURRENT_NODE_IP=""

G_ENCTOOL_LIBPATH="${G_UPGRADE_TEMP_DIR}/enctool"
G_ENCTOOL="${G_ENCTOOL_LIBPATH}/AES2KMC"

source ${G_CURRENT_PATH}/log.sh

export LD_LIBRARY_PATH=${LD_LIBRARY_PATH}:${G_ENCTOOL_LIBPATH}
export PATH=$PATH:/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/root/bin:/sbin

#******************************************************************#
# Function: get_driver_install_path
# Description: Get eBackup driver install path and version file path.
# Input Parameters: 
#   None
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function get_driver_install_path()
{
    G_HCP_MAIN_PATH=""
    G_DRIVER_VERSION_FILE_PATH=""
    if [ -f "/usr/lib/python2.7/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python2.7/site-packages/cinder/backup/"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python2.7/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib/python2.7/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python2.7/site-packages/cinder/backup/"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python2.7/site-packages/cinder/backup/drivers/version.conf"
        return 0
    elif [ -f "/usr/lib64/python2.7/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib64/python2.7/site-packages/cinder/backup/"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib64/python2.7/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib64/python2.7/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib64/python2.7/site-packages/cinder/backup/"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib64/python2.7/site-packages/cinder/backup/drivers/version.conf"
        return 0
    elif [ -f "/usr/lib64/python2.6/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib64/python2.6/site-packages/cinder/backup/"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib64/python2.6/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib64/python2.6/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib64/python2.6/site-packages/cinder/backup/"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib64/python2.6/site-packages/cinder/backup//drivers/version.conf"
        return 0
    elif [ -f "/usr/lib/python2.6/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python2.6/site-packages/cinder/backup/"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python2.6/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib/python2.6/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python2.6/site-packages/cinder/backup/"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python2.6/site-packages/cinder/backup/drivers/version.conf"
        return 0
    fi
    
    log_error "This is not eBackup driver node."
    return 1
}

#******************************************************************#
# Function: get_current_role
# Description: Get current role(0: eBackup driver, 1: not eBackup driver)
# Input Parameters: 
#   None
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function get_current_role()
{
    log_info "get current role"
    get_driver_install_path
    if [ $? -ne 0 ];then
        G_CURRENT_ROLE=1 #not eBackup driver
        log_error "This is not eBackup driver node."
    fi
    
    log_info "The driver version file is:${G_DRIVER_VERSION_FILE_PATH}."
    G_CURRENT_ROLE=0 #eBackup driver

    return 0
}

#******************************************************************#
# Function: check_role
# Description: Check the role of current node.It must be AdminNode.
# Input Parameters: 
#    None
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function check_role()
{
    get_current_role
    
    if [[ "${G_CURRENT_ROLE}" != "0" ]];then
        echo -e "\e[0;31;1mError:This operation must be done on eBackup driver.\e[0m"
        log_error "This operation must be done on eBackup driver."
        return 1
    fi
    return 0
}

#****************************************************#
# Function: check_ip_format
# Description: check the ip format
# Input Parameters:
#   $1: ip address
# Returns: 
#   0 success
#   1 failed
#****************************************************#
function check_ip_format()
{
    local L_IP="$1"
    echo "${L_IP}" | grep -E "^(((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|((([0-9A-Fa-f]{1,4}:){7}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}:[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){5}(:[0-9A-Fa-f]{1,4}){1,2})|(([0-9A-Fa-f]{1,4}:){4}(:[0-9A-Fa-f]{1,4}){1,3})|(([0-9A-Fa-f]{1,4}:){3}(:[0-9A-Fa-f]{1,4}){1,4})|(([0-9A-Fa-f]{1,4}:){2}(:[0-9A-Fa-f]{1,4}){1,5})|(([0-9A-Fa-f]{1,4})?:(:[0-9A-Fa-f]{1,4}){1,6})|(([0-9A-Fa-f]{1,4}:){1,6}:)|(::)|(([0-9A-Fa-f]{1,4}:){6}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){5}:(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){4}:([0-9A-Fa-f]{1,4}:){0,1}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){3}:([0-9A-Fa-f]{1,4}:){0,2}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){2}:([0-9A-Fa-f]{1,4}:){0,3}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4})?::([0-9A-Fa-f]{1,4}:){0,4}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})))$" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "IP(${L_IP}) format is wrong."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: test_connection
# Description: Test connection between local host and remote host
# Input Parameters: 
#   $1: remote host ip
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function test_connection()
{
    local L_HOST_IP="$1"
    check_ip_format "${L_HOST_IP}"
    if [ $? -ne 0 ];then
        log_error "IP format is wrong."
        return 1
    fi
    
    local L_RETRIES=3
    local L_PING_COMMAND="ping"
    echo "${L_HOST_IP}" | grep ":" >/dev/null 2>&1
    if [ $? -eq 0 ];then
        L_PING_COMMAND="ping6"
    fi
    
    while [ ${L_RETRIES} -gt 0 ]
    do
        ${L_PING_COMMAND} "${L_HOST_IP}" -c1 -w 2 >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            break
        fi

        L_RETRIES=`expr ${L_RETRIES} - 1`
        if [ ${L_RETRIES} -le 0 ]; then
            log_error "test connection to ${L_HOST_IP} failed"
            return 1
        fi

        sleep 1
    done
    return 0
}

#****************************************************#
# Function: check_yes_or_no
# Description: check yes or no
# Input Parameters: 
#    $1: display info
#    $2: default select
# Returns:
#    0 success (y)
#    1 failed  (n)
#****************************************************#
function check_yes_or_no()
{
    local L_INPUT_INFO=$1
    local L_DEFAULT_SELECT=$2
    local L_YES_OR_NO=""
    
    while true
    do
        echo -ne "${L_INPUT_INFO}"
        read L_YES_OR_NO
        
        if [[ -z "${L_YES_OR_NO}" && ! -z "${L_DEFAULT_SELECT}" ]];then
            L_YES_OR_NO="${L_DEFAULT_SELECT}"
            break
        fi

        if [[ "Y" == "$L_YES_OR_NO" || "y" == "${L_YES_OR_NO}" ]];then
            break;
        elif [[ "N" == "$L_YES_OR_NO" || "n" == "${L_YES_OR_NO}" ]];then
            break;
        else
            echo "Error input, please input 'Y' or 'N'."
        fi
    done
    
    if [[ "Y" == "$L_YES_OR_NO" || "y" == "$L_YES_OR_NO" ]];then
        return 0
    else
        return 1
    fi
}


#****************************************************#
# Function: print_all_nodes 
# Description: print om ip of all ebackup nodes
# Input Parameters:
#   None
# Returns:
#   0 success
#   1 failed
#****************************************************#
function print_all_nodes()
{
    local L_current_node="$1"
    local line_no=1
    local lim1="  "
    local lim2="\n"
    local last_lim=${lim1}
    printf "${L_current_node}${lim1}"
    for line in `cat ${G_NODES_TMP_FILE}`
    do
        om_ip="$line"
        check_ip_format "${om_ip}"
        if [ $? -ne 0 ];then
            continue
        fi
        line_no=$(($line_no+1))
        printf "${om_ip}"
        if [  $(($line_no % 5)) -eq 0 ];then
            last_lim=${lim2}
        else
            last_lim=${lim1}
        fi
        printf "${last_lim}"
    done
    if [ "${last_lim}" == "${lim1}" ];then
        printf "\n"
    fi
    echo "Total Nodes:${line_no}"
    return 0
}

#****************************************************#
# Function: get_all_om_ip 
# Description: get om ip of all ebackup nodes
# Input Parameters:
#   None
# Returns: 
#   0 success
#   1 failed
#****************************************************#
function get_all_om_ip()
{
    > ${G_NODES_TMP_FILE}
    local template_names=`cps template-list | sed 's/ //g' |awk -F "|" '{print $3}' |grep "cinder-backup"`
    for template_name in ${template_names}
    do
        log_info "template_name:${template_name}"
        local backup_driver=`cps template-ext-params-show --service cinder ${template_name} 2>/dev/null | sed 's/ //g' | egrep "^\|default.backup_driver\|.*\|$" | awk -F "|" '{ print $3}' |sed 's/ //g'`
        if [ -z  ${backup_driver} ];then
            continue
        fi
        if [ "${backup_driver}" != "cinder.backup.drivers.hwsebackup" ];then
            continue
        fi
        log_info "The template name(${template_name})is installed."
        cps template-instance-list --service cinder ${template_name} |sed 's/ //g' |grep "cinder-backup" |awk -F "|" '{print $6 }' 2>/dev/null >> "${G_NODES_TMP_FILE}"
        if [ $? -ne 0 ];then
            log_error "Get om ip failed."
            return 1
        fi
    done
    local tmp=`cat ${G_NODES_TMP_FILE} | sort -u`
    > ${G_NODES_TMP_FILE}
    for line in ${tmp}
    do
        if [ "${line}" == "${G_CURRENT_NODE_IP}" ];then
            log_info "The node(${line}) is current node."
            continue
        fi
        check_ip_format "${line}"
        if [ $? -ne 0 ];then
            log_error "The ip(${line}) format is wrong."
            continue
        fi
        log_info "The node IP is: ${line}."
        echo ${line} >> ${G_NODES_TMP_FILE}
    done
    
    return 0
}

function check_current_node()
{
    check_ip_format "$1"
    if [ $? -ne 0 ];then
        echo "The IP format is wrong."
        return 1
    fi
    
    ip a | grep -w "$1" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        echo "The input IP does not belong to the current node."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: get_all_driver_node
# Description: Get all eBackup driver node.
# Input Parameters:
#    None
# Returns:
#******************************************************************#
function get_all_driver_node()
{
    while true
    do
        printf "Please input current Fusionsphere openstack cinder node OM IP:"
        read G_CURRENT_NODE_IP
        check_current_node "${G_CURRENT_NODE_IP}"
        if [ $? -eq 0 ];then
            log_info "The current node IP is: ${G_CURRENT_NODE_IP}."
            echo "$G_CURRENT_NODE_IP" > "${G_CURRENT_NODE_FILE}"
            break
        fi
    done
        
    echo "Get all eBackup driver ip..."
    get_all_om_ip
    if [ $? -ne 0 ];then
        log_error "Get all om ip failed."
        return 1
    fi
    
    print_all_nodes "${G_CURRENT_NODE_IP}"
    
    return 0
}

#******************************************************************#
# Function: test_login
# Description: test login use hcp and root
# Input Parameters: 
#    $1: ip of test login
#    $2: the password of hcp
#    $3: the password of root
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function test_login()
{
    local L_NODE_IP="$1"
    local L_HCP_PASSWORD="$2"
    local L_ROOT_PASSWORD="$3"
    log_info "Test login ${L_NODE_IP}."
    
    test_connection "${L_NODE_IP}"
    if [ $? -ne 0 ];then
        log_error "Test connection to ${L_NODE_IP} failed."
        return 1
    fi
    
    local L_HCP_PWD_SAVE=`echo -e "${L_HCP_PASSWORD}\n${L_HCP_PASSWORD}" | ${G_ENCTOOL} enc | sed -n '$p'`
    local L_ROOT_PWD_SAVE=`echo -e "${L_ROOT_PASSWORD}\n${L_HCP_PASSWORD}" | ${G_ENCTOOL} enc | sed -n '$p'`
    echo "${L_ROOT_PWD_SAVE}" >"${G_NODE_PWD_TMP_FILE}"
    echo "${L_HCP_PWD_SAVE}" >>"${G_NODE_PWD_TMP_FILE}"
    chown -h ${G_USER_NAME} "${G_NODE_PWD_TMP_FILE}"
    
    chown -h ${G_USER_NAME} "${G_UPGRADE_TEMP_DIR}" -R >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify ${G_UPGRADE_TEMP_DIR} permission failed."
        return 1
    fi
    
    su ${G_USER_NAME} -c "expect -f ${G_UPGRADE_TEMP_DIR}/expectForRemote \"test_login\" \"${L_NODE_IP}\" \"${G_USER_NAME}\" \"${G_NODE_PWD_TMP_FILE}\""
    local L_RET=$?
    #delete the password tmp file
    rm "${G_NODE_PWD_TMP_FILE}" >/dev/null 2>&1
    if [ ${L_RET} == 41 ];then 
        echo "Failed to log in to the node[$L_NODE_IP] using the password of user ${G_USER_NAME}."
        log_error "Login ${L_NODE_IP} use ${G_USER_NAME} failed."
        return 1
    elif [ ${L_RET} == 42 ];then
        echo "Failed to log in to the node[$L_NODE_IP] using the password of user root."
        log_error "Login ${L_NODE_IP} use root failed."
        return 1
    elif [ ${L_RET} == 43 ];then
        echo "Failed to log in to the node[$L_NODE_IP] using public key, Permission denied."
        log_error "Login ${L_NODE_IP} using public key failed, Permission denied."
        return 1
    elif [ ${L_RET} == 44 ];then
        echo "Timeout to log in to the node[$node]."
        log_error "Login ${L_NODE_IP} timeout."
        return 1
    elif [ ${L_RET} == 45 ];then
        echo "Failed to log in to the node[$L_NODE_IP], Connection refused."
        log_error "Login ${L_NODE_IP} failed, Connection refused."
        return 1
    elif [ ${L_RET} == 46 ];then
        echo "Failed to log in to the node[$node] using the public key credential password of user ${G_USER_NAME}."
        log_error "Login ${L_NODE_IP} using public key failed."
        return 1
    elif [ ${L_RET} != 0 ];then
        echo "Failed to log in to the node[$L_NODE_IP]."
        log_error "Login ${L_NODE_IP} failed, RC=${L_RET}."
        return 1
    fi
    
    log_info "Test login ${L_NODE_IP} success."
    return 0
}

#******************************************************************#
# Function: get_all_nodes_password
# Description: get all nodes password
# Input Parameters: 
#    None
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function get_all_nodes_password()
{
    if [ ! -f "${G_ENCTOOL}" ];then
        log_error "The file(${G_ENCTOOL}) does not exist."
        return 1
    fi
    
    > "${G_NODES_PWD_FILE}"
    local L_ALL_NODES_TMP=`cat "${G_NODES_TMP_FILE}" 2>/dev/null`
    if [ "" == "${L_ALL_NODES_TMP}" ];then
        log_info "The eBackup driver nodes is empty."
        rm "${G_NODES_TMP_FILE}" >/dev/null 2>&1
        return 0
    fi
    
    local have_some_node_skip=0
    check_yes_or_no "Do you have some nodes to skip (y/n)? (default: n)" "n"
    if [ $? -eq 0 ];then
        have_some_node_skip=1
    fi
    
    echo "===================================================="
    local L_HCP_PASSWORD=""
    local L_ROOT_PASSWORD=""
    check_yes_or_no "Are passwords of users root on all nodes the same and those of users ${G_USER_NAME} on all nodes the same (y/n)? (default: y)" "y"
    if [ $? -eq 0 ];then
        while true
        do
            printf "Enter the password of user ${G_USER_NAME} or the passphrase of user ${G_USER_NAME}'s private key file of all eBackup driver nodes:"
            read -s L_HCP_PASSWORD
            echo "" 
            if [ "" != "${L_HCP_PASSWORD}" ];then
                break
            fi
        done
        while true
        do
            printf "Enter the password of user root of all eBackup driver nodes:"
            read -s L_ROOT_PASSWORD
            echo ""
            if [ "" != "${L_ROOT_PASSWORD}" ];then
                break
            fi
        done
        
        #test connect 
        for node in `cat ${G_NODES_TMP_FILE}`
        do
            if [ "${node}" == "${G_CURRENT_NODE_IP}" ];then
                log_info "The node(${line}) is current node."
                continue
            fi
            
            if [ ${have_some_node_skip} -eq 1 ];then
                check_yes_or_no "Are you want to skip this node(${node}) (y/n)? (default:n)" "n"
                if [ $? -eq 0 ];then
                    continue
                fi
            fi
            
            #test connect
            test_login "${node}" "${L_HCP_PASSWORD}" "${L_ROOT_PASSWORD}"
            if [ $? -ne 0 ];then
                log_error "Login ${node} failed."
                return 1
            fi
            
            #write the password to tmp file
            local L_HCP_PWD_SAVE=`echo -e "${L_HCP_PASSWORD}\n${L_HCP_PASSWORD}" | ${G_ENCTOOL} enc | sed -n '$p'`
            local L_ROOT_PWD_SAVE=`echo -e "${L_ROOT_PASSWORD}\n${L_ROOT_PASSWORD}" | ${G_ENCTOOL} enc | sed -n '$p'`
            echo "${node}|${L_HCP_PWD_SAVE}|${L_ROOT_PWD_SAVE}" >> "${G_NODES_PWD_FILE}"
            
            log_info "Test connection node(${node}) success."
            echo "Test connection node(${node}) success."
        done
    else
        for node in `cat ${G_NODES_TMP_FILE}`
        do
            if [ "${node}" == "${G_CURRENT_NODE_IP}" ];then
                log_info "The node(${line}) is current node."
                continue
            fi
            
            cat "${G_NODES_PWD_FILE}" | grep -w "${node}" >/dev/null 2>&1
            if [ $? -eq 0 ];then
                echo "The node(${node}) is already configured."
                continue
            fi
            
            if [ ${have_some_node_skip} -eq 1 ];then
                check_yes_or_no "Are you want to skip this node(${node}) (y/n)? (default:n)" "n" 
                if [ $? -eq 0 ];then
                    continue
                fi
            fi
            
            #input the password of hcp and root
            local L_HCP_PWD_SAVE=""
            local L_ROOT_PWD_SAVE=""
            while true
            do
                #input password
                while true
                do
                    printf "Enter the password of user ${G_USER_NAME} or the passphrase of user ${G_USER_NAME}'s private key file of the node(${node}):" 
                    read -s L_HCP_PASSWORD
                    echo "" 
                    if [ "" != "${L_HCP_PASSWORD}" ];then
                        break
                    fi
                done
                while true
                do
                    printf "Enter the password of user root of the node(${node}):" 
                    read -s L_ROOT_PASSWORD
                    echo ""
                    if [ "" != "${L_ROOT_PASSWORD}" ];then
                        break
                    fi
                done
                
                #test connect
                test_login "${node}" "${L_HCP_PASSWORD}" "${L_ROOT_PASSWORD}"
                if [ $? -ne 0 ];then
                    log_warn "Login ${node} failed."
                    continue
                fi
                
                L_HCP_PWD_SAVE=`echo -e "${L_HCP_PASSWORD}\n${L_HCP_PASSWORD}" | ${G_ENCTOOL} enc | sed -n '$p'`
                L_ROOT_PWD_SAVE=`echo -e "${L_ROOT_PASSWORD}\n${L_ROOT_PASSWORD}" | ${G_ENCTOOL} enc | sed -n '$p'`
                break
            done
            
            echo "${node}|${L_HCP_PWD_SAVE}|${L_ROOT_PWD_SAVE}" >> "${G_NODES_PWD_FILE}"
            
            echo "===================================================="
        done
    fi
    
    rm "${G_NODES_TMP_FILE}" >/dev/null 2>&1
    return 0
}

#******************************************************************#
# Function: copy_unzip_package
# Description: Copy the package to /tmp/upgrade and decompress
# Input Parameters: 
#    None
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function copy_unzip_package()
{
    local max_size=$((1024*1024*1024))

    #create /tmp/upgrade
    if [ ! -d ${G_UPGRADE_TEMP_DIR} ];then
        log_info "mkdir -p ${G_UPGRADE_TEMP_DIR} dir"
        mkdir -p ${G_UPGRADE_TEMP_DIR} >/dev/null 2>&1
        if [ $? -ne 0 ];then
            echo -e "\e[0;33;1mWarning:Create ${G_UPGRADE_TEMP_DIR} failed.\e[0m"
            log_error "Create ${G_UPGRADE_TEMP_DIR} failed."
            return 1
        fi
    fi
    
    #copy package to /tmp/upgrade
    echo "${G_CURRENT_PATH}" | grep -E "^${G_UPGRADE_TEMP_DIR}/?\$" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        echo "Copy the file ${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME} to ${G_UPGRADE_TEMP_DIR}/."
        log_info "Copy ${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME} to ${G_UPGRADE_TEMP_DIR}/."
        cp -P --remove-destination "${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME}" "${G_UPGRADE_TEMP_DIR}/" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            echo "Copy ${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME} to ${G_UPGRADE_TEMP_DIR}/ failed."
            log_error "Copy ${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME} to ${G_UPGRADE_TEMP_DIR}/ failed."
            return 1
        fi
        
        #decompress package
        echo "Decompress the file ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME}."
        log_info "Decompress ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME}."
        
        total_size=$(tar -tvf "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME}" | awk 'BEGIN{sum=0}{sum+=$3}END{print int(sum)}')
        if [ ${total_size} -gt ${max_size} ];then
            echo "compressed file is too large"
            return 1
        fi

        tar -zxvf "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME}" -C "${G_UPGRADE_TEMP_DIR}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            echo "Decompress ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME} failed."
            log_error "tar -zxvf ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME} failed."
            return 1
        fi
    fi
    
    return 0
}

function init_kmc_log_dir()
{
    mkdir -p "${G_UPGRADE_TEMP_DIR}/logs" >/dev/null 2>&1
    touch "${G_UPGRADE_TEMP_DIR}/logs/script.log" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "touch ${G_UPGRADE_TEMP_DIR}/logs/script.log failed."
        return 1
    fi
    chown -h ${G_USER_NAME} "${G_UPGRADE_TEMP_DIR}" -R >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify ${G_UPGRADE_TEMP_DIR} permission failed."
        return 1
    fi
    
    return 0
}

function main()
{
    # check role
    check_role
    if [ $? -ne 0 ];then
        log_error "This operation must be done on eBackup driver node."
        echo "Failed to configure."
        return 1
    fi
    
    init_kmc_log_dir
    if [ $? -ne 0 ];then
        log_error "Init kmc log dir failed."
        echo "Failed to configure."
        return 1
    fi
    
    #check the cps environment variables
    if [[ -z $CPS_USERNAME || -z $CPS_PASSWORD ]];then
        echo "Please set the cps environmental variables first."
        return 1
    fi
    
    # copy path file to /tmp/upgrade and decompress package
    copy_unzip_package
    if [ $? -ne 0 ];then
        log_error "Copy or decompress package failed."
        echo "Failed to configure."
        return 1
    fi
    
    # get all register node
    get_all_driver_node
    if [ $? -ne 0 ];then
        log_error "Get all register node failed."
        echo "Failed to configure."
        return 1
    fi
    
    # get all register node
    get_all_nodes_password
    if [ $? -ne 0 ];then
        log_error "Get the password of those nodes failed."
        echo "Failed to configure."
        return 1
    fi
    
    echo "Successed to configure."
    return 0
}

main
exit $?
