#!/bin/bash
set +x

G_CURRENT_PATH="$(cd "$(dirname "$0")" && pwd)"
source ${G_CURRENT_PATH}/common.sh

G_USER_NAME="fsp"
G_CURRENT_ROLE=1  #0:eBackup driver, 1:not eBackup driver
G_UPGRADE_TEMP_DIR="/tmp/upgrade"
G_CURRENT_NODE_FILE="${G_UPGRADE_TEMP_DIR}/localhost.txt"
G_SERVERS_CONF="${G_UPGRADE_TEMP_DIR}/servers.txt"
G_NODE_PWD_TMP_FILE="${G_UPGRADE_TEMP_DIR}/server_tmp_info.txt"
G_PATCH_FILES_PATH="${G_UPGRADE_TEMP_DIR}/patch_file.txt"

G_UPGRADE_NODE_RESULT="${G_UPGRADE_TEMP_DIR}/upgrade_nodes_result" #has upgraded nodes
G_ROLLBACK_NODE_RESULT="${G_UPGRADE_TEMP_DIR}/rollback_nodes_result" #has roll backed nodes
G_BACKUP_RESULT="${G_UPGRADE_TEMP_DIR}/backup_result" #backup self result
G_UPGRADE_FINAL_RESULT="${G_UPGRADE_TEMP_DIR}/upgrade_final_result" #current node upgrade result

G_HCP_MAIN_PATH=""
G_DRIVER_VERSION_FILE_PATH=""
G_DJ_ENV_FILE="/etc/huawei/dj/cfg/sys.ini"
G_CINDER_SERVICES_FILE="${G_UPGRADE_TEMP_DIR}/cinder_backup_services.txt"

G_ENCTOOL_LIBPATH="${G_UPGRADE_TEMP_DIR}/enctool"

G_ERR_PATCH_FILE_NOT_EXIST=80

source ${G_CURRENT_PATH}/log.sh

export LD_LIBRARY_PATH=${LD_LIBRARY_PATH}:${G_ENCTOOL_LIBPATH}
export PATH=$PATH:/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/root/bin:/sbin

#******************************************************************#
# Function: usage
# Description: show usage of this scripts
# Input Parameters: N/A
# Returns: N/A
#******************************************************************#
function usage()
{ 
    cat << END                                                             
Usage:
    sh patch.sh upgrade              Upgrade eBackup driver on all nodes.
    sh patch.sh upgrade_self         Upgrade eBackup driver.
    sh patch.sh rollback             Roll back eBackup driver on all nodes.
    sh patch.sh rollback_self        Roll back eBackup driver.
    sh patch.sh clean                Clean upgrade tmporary file on all nodes.
    sh patch.sh clean_self           Clean upgrade tmporary file.
END
}

function is_dj_env()
{
    if [ -f ${G_DJ_ENV_FILE} ]; then
        log_info "it is DJ env."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: get_driver_install_path
# Description: Get eBackup driver install path and version file path.
# Input Parameters: 
#   None
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function get_driver_install_path()
{
    G_HCP_MAIN_PATH=""
    G_DRIVER_VERSION_FILE_PATH=""
    if [ -f "/usr/lib/python3.7/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python3.7/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python3.7/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib/python3.7/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python3.7/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python3.7/site-packages/cinder/backup/drivers/version.conf"
        return 0
    elif [ -f "/usr/lib/python2.7/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python2.7/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python2.7/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib/python2.7/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python2.7/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python2.7/site-packages/cinder/backup/drivers/version.conf"
        return 0
    elif [ -f "/usr/lib64/python2.7/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib64/python2.7/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib64/python2.7/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib64/python2.7/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib64/python2.7/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib64/python2.7/site-packages/cinder/backup/drivers/version.conf"
        return 0
    elif [ -f "/usr/lib64/python2.6/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib64/python2.6/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib64/python2.6/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib64/python2.6/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib64/python2.6/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib64/python2.6/site-packages/cinder/backup//drivers/version.conf"
        return 0
    elif [ -f "/usr/lib/python2.6/site-packages/cinder/backup/drivers/ebackupversion.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python2.6/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python2.6/site-packages/cinder/backup/drivers/ebackupversion.conf"
        return 0
    elif [ -f "/usr/lib/python2.6/site-packages/cinder/backup/drivers/version.conf" ];then
        G_HCP_MAIN_PATH="/usr/lib/python2.6/site-packages"
        G_DRIVER_VERSION_FILE_PATH="/usr/lib/python2.6/site-packages/cinder/backup/drivers/version.conf"
        return 0
    fi
    
    log_error "This is not eBackup driver node."
    return 1
}

#******************************************************************#
# Function: get_current_role
# Description: Get current role(0: eBackup driver node, 1: not eBackup driver node)
# Input Parameters: 
#   None
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function get_current_role()
{
    log_info "get current role"
    get_driver_install_path
    if [ $? -ne 0 ];then
        G_CURRENT_ROLE=1 #not eBackup driver
        log_error "This is not eBackup driver node."
    fi
    
    log_info "The driver version file is:${G_DRIVER_VERSION_FILE_PATH}."
    G_CURRENT_ROLE=0 #eBackup driver
    G_DIRVER_BACKUP_ROOT_DIR="${G_HCP_MAIN_PATH}/cinder/backup/ebackup_driver_backup/${G_CURRENT_VERSION}"
    
    return 0
}

#******************************************************************#
# Function: test_connection
# Description: Test connection between local host and remote host
# Input Parameters: 
#   $1: remote host ip
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function test_connection()
{
    local L_RETRIES=3
    local L_PING_COMMAND="ping"
    echo "$1" | grep ":" >/dev/null
    if [ $? -eq 0 ];then
        L_PING_COMMAND="ping6"
    fi
    while [ ${L_RETRIES} -gt 0 ]
    do
        ${L_PING_COMMAND} "$1" -c1 -w 2 >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            break
        fi

        L_RETRIES=`expr ${L_RETRIES} - 1`
        if [ ${L_RETRIES} -le 0 ]; then
            log_error "test connection to $1 failed"
            return 1
        fi

        sleep 1
    done
    return 0
}

#******************************************************************#
# Function: send_cmd
# Description: execute command use expect on remote node
# Input Parameters: 
#    $1: Operate type(dispatch/upgrade/...)
#    $2: Node IP
#    $3: User name (hcp)
#    $4: Password of the user(hcp)
#    $5: Password of the user(root)
#    $6: the parameter for execute patch.sh(upload/upgrade/...)
#    $7: the parameter1 for execute patch.sh
#    $8: the parameter2 for execute patch.sh
#    $9: the parameter3 for execute patch.sh
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function send_cmd()
{
    local L_OPES=$1
    local L_HOST_IP=$2
    
    log_info "Test node(${L_HOST_IP}) connection."
    test_connection "${L_HOST_IP}"
    if [ $? -ne 0 ];then
        log_error "Execute command(${L_OPES}) failed, test connection to ${L_HOST_IP} failed."
        return 1
    fi
    
    log_info "Execute command(${L_OPES}) on node(${L_HOST_IP})..."
    #write the pwd to the tmp file
    local L_HCP_PWD_SAVE="$4"
    local L_ROOT_PWD_SAVE="$5"
    echo "${L_ROOT_PWD_SAVE}" >"${G_NODE_PWD_TMP_FILE}"
    echo "${L_HCP_PWD_SAVE}" >>"${G_NODE_PWD_TMP_FILE}"
    chown -h ${G_USER_NAME} "${G_NODE_PWD_TMP_FILE}"
    
    chown -h ${G_USER_NAME} "${G_UPGRADE_TEMP_DIR}" -R >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify ${G_UPGRADE_TEMP_DIR} permission failed."
        return 1
    fi
    
    #execute cmd use expect
    su ${G_USER_NAME} -c "expect -f ${G_UPGRADE_TEMP_DIR}/expectForRemote '${L_OPES}' '${L_HOST_IP}' '$3' '${G_NODE_PWD_TMP_FILE}' '$6' '$7' '$8' '$9'"
    local L_RET=$? 
    if [ ${L_RET} -ne 0 ];then
        log_error "Execute command(${L_OPES}) on node(${L_HOST_IP}) failed, ret=${L_RET}."
    else
        log_info "Execute command(${L_OPES}) on node(${L_HOST_IP}) successfully."
    fi
    
    #delete the password tmp file
    rm "${G_NODE_PWD_TMP_FILE}" >/dev/null 2>&1
    return ${L_RET}
}

#******************************************************************#
# Function: check_operator
# Description: check current operator is root user or not
# Input Parameters: 
#   NA
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function check_operator()
{
    local L_CUR_USER=`whoami`
    if [ "${L_CUR_USER}" != "root" ]; then
        log_error "The operator user is not root."
        echo -e "\e[0;31;1mError:You have to be root in order to run those scripts.\e[0m"
        exit 1
    fi
}

#******************************************************************#
# Function: do_rollback
# Description: Roll back self
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function do_rollback()
{
    log_info "Begin roll back..."
    
    local L_PATCH_INFO="$1"
    if [[ "" != "${L_PATCH_INFO}" && -f "${L_PATCH_INFO}" ]];then
        log_info "Delete file ${L_PATCH_INFO}."
        rm "${L_PATCH_INFO}" > /dev/null 2>&1
    fi
    
    rollback_other_before
    if [ $? -ne 0 ];then
        log_error "Rollback other before failed."
        return 1
    fi
    
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            continue
        fi
        
        echo "${line}" | grep "^${G_PATCH_FILE_PATH_PREFIX}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            continue
        fi
        
        local L_lib=`echo ${line} | awk -F' ' '{print $1}'`
        if [ "X${L_lib}" == "X" ];then
            log_warn "The node(${line}) is empty."
            continue
        fi
        L_lib="${G_HCP_MAIN_PATH}/${L_lib}"
        local L_mode=`echo ${line} | awk -F' ' '{print $5}'`
        
        rollback_copy_file "${L_lib}" "${L_mode}"
        local L_copy_ret=$?
        if [ 0 -ne ${L_copy_ret} ];then
            log_error "Roll back file failed."
            return 1
        fi
    done < ${G_PATCH_FILES_PATH}
    
    rollback_other_after
    if [ $? -ne 0 ];then
        log_error "Rollback other after failed."
        return 1
    fi
    
    restart_process
    if [ $? -ne 0 ];then
        log_error "Restart process failed."
        return 1
    fi
    
    log_info "Roll back success."
    return 0
}

function rollback_other_before()
{
    log_info "Roll back other."
    
    return 0
}

function rollback_other_after()
{
    log_info "Roll back other."
    
    return 0
}

function rollback_copy_file()
{
    local L_lib="$1"
    local L_lib_backup="${G_DIRVER_BACKUP_ROOT_DIR}/${L_lib}"
    local L_mode="$2"
    
    if [ -f "${L_lib_backup}" ];then
        log_info "rm ${L_lib}."
        rm "${L_lib}" > /dev/null 2>&1

        log_info "Copy ${L_lib_backup} to ${L_lib}."
        cp -afP --remove-destination "${L_lib_backup}" "${L_lib}" > /dev/null 2>&1
        if [ $? -ne 0 ];then
            log_error "Copy ${L_lib_backup} to ${L_lib} failed."
            return 1
        fi
        
        return 0
    fi
    
    if [ "new" == "${L_mode}" ];then
        log_info "The file ${L_lib} is new add, need to delete."
        rm "${L_lib}" >/dev/null 2>&1
        return 0
    fi
    
    log_info "The file ${L_lib} has not backup."
    return 0
}

#******************************************************************#
# Function: backup_self
# Description: Backup the files, that need to be upgrade
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function backup_self()
{
    log_info "Begin backup self."
    
    if [ -f "${G_BACKUP_RESULT}" ];then
        local L_BACKUP_RET=`cat ${G_BACKUP_RESULT} 2>/dev/null`
        if [ "0" == "${L_BACKUP_RET}" ];then
            log_info "The last backup self has successfully."
            return 0
        fi
    fi
    
    #delete backup dir
    log_info "rm -rf ${G_DIRVER_BACKUP_ROOT_DIR}."
    rm -rf "${G_DIRVER_BACKUP_ROOT_DIR}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_warn "Delete ${G_DIRVER_BACKUP_ROOT_DIR} failed."
    fi
    
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            continue
        fi
        
        echo "${line}" | grep "^${G_PATCH_FILE_PATH_PREFIX}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            continue
        fi
        
        local L_lib=`echo ${line} | awk -F' ' '{print $1}'`
        if [ "X${L_lib}" == "X" ];then
            log_warn "The file(${line}) is empty."
            continue
        fi
        L_lib="${G_HCP_MAIN_PATH}/${L_lib}"
        
        backup_copy_file "${L_lib}"
        local L_copy_ret=$?
        if [ 0 -ne ${L_copy_ret} ];then
            log_error "Backup file failed."
            return 1
        fi
    done < ${G_PATCH_FILES_PATH}
    
    backup_other_file
    if [ $? -ne 0 ];then
        log_error "Backup other file failed."
        return 1
    fi
    
    echo "0" > "${G_BACKUP_RESULT}"
    if [ $? -ne 0 ];then
        log_error "Write backup result failed."
        return 1
    fi
    log_info "Backup self success."
    return 0
}

function backup_other_file()
{
    log_info "Backup other file."
    
    return 0
}

function backup_copy_file()
{
    local L_lib="$1"
    local L_lib_backup="${G_DIRVER_BACKUP_ROOT_DIR}/${L_lib}"
    local L_lib_backup_dir=""
    L_lib_backup_dir=`dirname "${L_lib_backup}" 2>/dev/null`
    if [ $? -ne 0 ];then
        log_error "Get backup directory ${L_lib_backup_dir} failed."
        return 1
    fi
    if [ "" == "${L_lib_backup_dir}" ];then
        log_error "The backup directory is empty."
        return 1
    fi
    
    mkdir -p "${L_lib_backup_dir}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Create backup directory ${L_lib_backup_dir} failed."
        return 1
    fi
    
    if [ -f "${L_lib_backup}" ];then
        log_info "rm ${L_lib_backup}."
        rm "${L_lib_backup}" > /dev/null 2>&1
    fi
    
    if [ ! -f "${L_lib}" ];then
        log_info "The file ${L_lib} no need to backup, the file does not exist."
        return 0
    fi
    
    log_info "Copy file ${L_lib} to ${L_lib_backup}."
    cp -afP --remove-destination "${L_lib}" "${L_lib_backup}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Backup ${L_lib} to ${L_lib_backup} failed."
        return 1
    fi
    if [ ! -f "${L_lib_backup}" ];then
        log_error "Backup ${L_lib} failed, the ${L_lib_backup} does not exist."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: do_upgrade
# Description: replace the file
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function do_upgrade()
{
    log_info "Begin do upgrade."
    
    upgrade_other_before "$1" "$2" "$3"
    if [ $? -ne 0 ];then
        log_error "Upgrade other before failed."
        return 1
    fi
    
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            continue
        fi
        
        echo "${line}" | grep "^${G_PATCH_FILE_PATH_PREFIX}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            continue
        fi
        
        local L_lib=`echo ${line} | awk -F' ' '{print $1}'`
        if [ "X${L_lib}" == "X" ];then
            log_warn "The file(${line}) is empty."
            continue
        fi
        local L_patch_lib_file_path="${L_lib}"
        L_lib="${G_HCP_MAIN_PATH}/${L_lib}"
        
        upgrade_copy_file "${L_lib}" "${line}"
        local L_copy_ret=$?
        if [ 0 -ne ${L_copy_ret} ];then
            log_error "Upgrade file failed."
            return 1
        fi
    done < ${G_PATCH_FILES_PATH}
    
    upgrade_other_after "$1" "$2" "$3"
    if [ $? -ne 0 ];then
        log_error "Upgrade other after failed."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: upgrade_other_before
# Description: upgrade other file
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_other_before()
{
    log_info "Upgrade other."
    
    return 0
}

#******************************************************************#
# Function: upgrade_other_after
# Description: upgrade other file
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_other_after()
{
    log_info "Upgrade other."
    
    return 0
}

function upgrade_copy_file()
{
    local L_lib="$1"
    local L_line="$2"
    local L_mode=`echo ${L_line} | awk -F' ' '{print $5}'`
    
    if [ "modify" == "${L_mode}" ];then
        log_info "This file ${L_lib} will be modify, no need replace."
        return 0
    fi
    
    log_info "rm file ${L_lib}."
    rm "${L_lib}" > /dev/null 2>&1
    local L_del_ret=$?
    if [ "delete" == "${L_mode}" ];then
        if [ 0 -ne ${L_del_ret} ];then
            log_error "Delete file ${L_lib} failed."
            return 1
        fi
        
        log_info "Delete file ${L_lib} success."
        return 0
    fi
    
    log_info "Copy file ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}/${L_patch_lib_file_path} to ${L_lib}."
    cp -P --remove-destination "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}/${L_patch_lib_file_path}" "${L_lib}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Copy ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}/${L_patch_lib_file_path} to ${L_lib} failed."
        return 1
    fi
    
    log_info "Modify permission of the file(${L_lib})."
    echo "${L_line}" | awk "{ system(\"chown \"\$3\":\"\$4\" ${L_lib}\");system(\"chmod  \"\$2\" ${L_lib}\")}" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify permission of the file(${L_lib}) failed."
        return 1
    fi
    
    #verify
    log_info "Verify file ${L_lib}."
    local L_src_file_sha256=$(sha256sum "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}/${L_patch_lib_file_path}" 2>/dev/null | awk -F' ' '{print $1}')
    local L_dest_file_sha256=$(sha256sum "${L_lib}" 2>/dev/null | awk -F' ' '{print $1}')
    if [[ "${L_src_file_sha256}" != "${L_dest_file_sha256}" || "" == "${L_dest_file_sha256}" ]];then
        log_error "Verify ${L_lib} failed, srcSHA256=${L_src_file_sha256}, destSHA256=${L_dest_file_sha256}."
        return 1
    fi
    
    return 0
}

function check_service_status()
{
    > "${G_CINDER_SERVICES_FILE}"
    local L_SERVICE_MODES=`ls /etc/cinder/components_info/|grep "^cinder-backup"`
    if [ $? -ne 0 ];then
        log_error "there no cinder-backup service."
        return 1
    fi
    
    local L_IS_PASS=0
    for L_SERVICE_MODE in ${L_SERVICE_MODES}
    do
        ps -ef |grep "cinder-backup"|grep "config-file"|grep -E "(${L_SERVICE_MODE}\.conf)|(${L_SERVICE_MODE}_.*\.conf)" >/dev/null 2>&1
        if [ $? != 0 ];then
            log_warn "${L_SERVICE_MODE} is not running."
            continue
        else
            log_info "${L_SERVICE_MODE} is running, but it maybe restart,so we need to check it three times."
            sleep 2
            frist_time_pid=`ps -ef |grep "cinder-backup"|grep "config-file"|grep -E "(${L_SERVICE_MODE}\.conf)|(${L_SERVICE_MODE}_.*\.conf)"|awk '{print $2}'`
            log_info "frist_time_pid:${frist_time_pid}"
            sleep 3
            second_time_pid=`ps -ef |grep "cinder-backup"|grep "config-file"|grep -E "(${L_SERVICE_MODE}\.conf)|(${L_SERVICE_MODE}_.*\.conf)"|awk '{print $2}'` 
            log_info "second_time_pid:${second_time_pid}"
            sleep 3
            third_time_pid=`ps -ef |grep "cinder-backup"|grep "config-file"|grep -E "(${L_SERVICE_MODE}\.conf)|(${L_SERVICE_MODE}_.*\.conf)"|awk '{print $2}'`
            log_info "third_time_pid:${third_time_pid}"
            if [[ $frist_time_pid -eq $second_time_pid ]] && [[ $second_time_pid -eq $third_time_pid ]];then 
               L_IS_PASS=$(expr $L_IS_PASS + 1)
               echo "$L_SERVICE_MODE" >> "${G_CINDER_SERVICES_FILE}"
               log_info "cinder backup(${L_SERVICE_MODE}) is running."
               continue
            else
               log_warn "${L_SERVICE_MODE} is not running."
               continue
            fi
        fi
    done
    if [ ${L_IS_PASS} -eq 0 ];then
        log_error "There have no cinder backup service is running."
        return 1
    fi
    
    log_info "There have same cinder backup service is running."
    return 0
}

#******************************************************************#
# Function: restart_process
# Description: Restart cinder-backup
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_process()
{
    local L_RET=0
    
    is_dj_env
    L_RET=$?
    if [ ${L_RET} -eq 1 ]; then
        for i in `ps -e | grep "cinder-backup" | awk '{print $1}'`
        do
            kill -9 $i
        done 
    else
        local L_SERVICE_MODES=`cat ${G_CINDER_SERVICES_FILE} 2>/dev/null`
        if [ "" == "${L_SERVICE_MODES}" ];then
            log_error "there no cinder-backup service."
            return 1
        fi
        
        for L_SERVICE_MODE in ${L_SERVICE_MODES}
        do
            log_info "Restart cinder backup service:${L_SERVICE_MODE}."
            ${L_SERVICE_MODE}Control -A RESTART
        done
    fi
    
    ### post check
    check_service_status_verify
    L_RET=$?
    if [ ${L_RET} -ne 0 ]; then
       log_error "check_service_status_verify return failed."
       return ${L_RET}
    fi
    
    log_info "Restart process success."
    return 0
}

#******************************************************************#
# Function: check_service_status_verify
# Description: check process status
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function check_service_status_verify()
{
    if [ ! -f "${G_CINDER_SERVICES_FILE}" ];then
        log_error "the ${G_CINDER_SERVICES_FILE} is not exists."
        return 1
    fi
    local L_SERVICE_MODES=`cat ${G_CINDER_SERVICES_FILE} 2>/dev/null`
    if [ "" == "${L_SERVICE_MODES}" ];then
        log_error "there no cinder-backup service." 
        return 1
    fi
    
    local L_CHECK_MAX_TIMES=12
    local L_IS_PASS=0
    local L_serviceNums=0
    for L_SERVICE_MODE in ${L_SERVICE_MODES}
    do
        L_trytime=0
        L_serviceNums=$(expr $L_serviceNums + 1)
        while [[ ${L_trytime} -lt ${L_CHECK_MAX_TIMES} ]]
        do
            ps -ef |grep "cinder-backup"|grep "config-file"|grep -E "(${L_SERVICE_MODE}\.conf)|(${L_SERVICE_MODE}_.*\.conf)" >/dev/null 2>&1
            if [ $? != 0 ];then
                log_error "${L_SERVICE_MODE} is not running, retry ${L_trytime}."
                sleep 5
                L_trytime=$(expr $L_trytime + 1)
                continue
            else
                log_info "${L_SERVICE_MODE} is running, but it maybe restart,so we need to check it three times."
                frist_time_pid=`ps -ef |grep "cinder-backup"|grep "config-file"|grep -E "(${L_SERVICE_MODE}\.conf)|(${L_SERVICE_MODE}_.*\.conf)"|awk '{print $2}'`
                log_info "frist_time_pid:${frist_time_pid}"
                sleep 3
                second_time_pid=`ps -ef |grep "cinder-backup"|grep "config-file"|grep -E "(${L_SERVICE_MODE}\.conf)|(${L_SERVICE_MODE}_.*\.conf)"|awk '{print $2}'` 
                log_info "second_time_pid:${second_time_pid}"
                sleep 3
                third_time_pid=`ps -ef |grep "cinder-backup"|grep "config-file"|grep -E "(${L_SERVICE_MODE}\.conf)|(${L_SERVICE_MODE}_.*\.conf)"|awk '{print $2}'`
                log_info "third_time_pid:${third_time_pid}"
                if [[ $frist_time_pid -eq $second_time_pid ]] && [[ $second_time_pid -eq $third_time_pid ]];then 
                   L_IS_PASS=$(expr $L_IS_PASS + 1)
                   log_info "cinder backup(${L_SERVICE_MODE}) restart success."
                   break
                else
                   sleep 5 
                   L_trytime=$(expr $L_trytime + 1)
                   log_info "L_trytime:${L_trytime}"
                   continue
                fi
            fi
        done
        if [ ${L_trytime} -ge ${L_CHECK_MAX_TIMES} ];then
            log_error "${L_SERVICE_MODE} service is not running."
            return 1
        fi
    done
    if [[ ${L_IS_PASS} -ne ${L_serviceNums} ]];then
        log_error "cinder backup service number is wrong. L_IS_PASS:${L_IS_PASS},L_serviceNums:${L_serviceNums}." 
        return 1
    fi
    
    log_info "cinder backup service verify ok."
    return 0
}

#******************************************************************#
# Function: check_already_patched
# Description: check the node already patched
# Input Parameters: 
#   Null
# Returns:
#   0 already patched
#   1 not patch
#******************************************************************#
function check_already_patched()
{
    if [ -f "${G_UPGRADE_FINAL_RESULT}" ];then
        local L_UPGRADE_RET=`cat ${G_UPGRADE_FINAL_RESULT} 2>/dev/null`
        if [ "0" == "${L_UPGRADE_RET}" ];then
            log_info "The last upgrade has successfully."
            return 0
        fi
    fi
    
    local L_INDEX=0
    if [ ! -d ${G_DIRVER_BACKUP_ROOT_DIR} ]; then
        log_info "This node not patch, need patch."
        return 1
    fi
    L_INDEX=$(ls ${G_DIRVER_BACKUP_ROOT_DIR} | grep "^patch.*info$" | awk -F '.' '{print $1}'| awk -F '_' '{print $2}'| sort -n | sed -n '$p')
    if [ "" == "${L_INDEX}" ];then
        L_INDEX=1
    fi
    local L_PATCH_INFO="${G_DIRVER_BACKUP_ROOT_DIR}/patch_${L_INDEX}.info"
    if [ ! -f "${L_PATCH_INFO}" ];then
        log_info "This node not patch, because the patch info file(${L_PATCH_INFO}) does not exist, need patch."
        return 1
    fi

    cat "${L_PATCH_INFO}" | egrep "^Version:${G_PATCH_VERSION}\$" >/dev/null 2>&1
    if [ $? -eq 0 ];then
        log_info "This node has been patched."
        return 0
    fi
    
    log_info "This node not patch, need patch."
    return 1
}

#******************************************************************#
# Function: check_version
# Description: check version
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function check_version()
{
    dos2unix "${G_DRIVER_VERSION_FILE_PATH}" >/dev/null 2>&1
    local L_SRC_VER=$(cat "${G_DRIVER_VERSION_FILE_PATH}" 2>/dev/null | grep "eBackupVersion" | awk -F '=' '{print $2}')
    if [ "${G_CURRENT_VERSION}" != "${L_SRC_VER}" ];then
        log_error "Current version is not [${G_CURRENT_VERSION}], it is [${L_SRC_VER}]."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: upgrade_failed_handle
# Description: upgrade failed handle
# Input Parameters: 
#    $1: patch file path
# Returns:
#******************************************************************#
function upgrade_failed_handle()
{
    log_info "Upgrade failed."
    local L_PATCH_FILE_PATH="$1"
    do_rollback "${L_PATCH_FILE_PATH}"
    if [ $? -ne 0 ];then
        echo "Roll back failed."
        log_error "Rollback failed."
    fi
}

#******************************************************************#
# Function: upgrade_self
# Description: upgrade self
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_self()
{
    log_info "Begin upgrade self."
    
    #check the operator user is whether or not root
    check_operator
    
    #Get current role(0: eBackup driver node, 1: not eBackup driver node) and install path and version file path
    get_current_role
    
    #version check
    check_version
    if [ $? -ne 0 ];then
        log_error "Check version failed."
        return 1
    fi
    
    #check the node already patched
    check_already_patched
    if [ $? -eq 0 ];then
        log_info "This node has been patched."
        echo "This node has been patched."
        return 0
    fi
    
    #check file $G_PATCH_FILES_PATH
    if [ ! -f "${G_PATCH_FILES_PATH}" ];then
        log_error "The file ${G_PATCH_FILES_PATH} does not exist."
        return 1
    fi
    
    #check process status
    check_service_status
    if [ $? -ne 0 ];then
        log_error "Check service status failed."
        return 1
    fi
    
    #backup
    log_info "Backup self."
    backup_self
    if [ $? -ne 0 ];then
        log_error "Backup failed."
        return 1
    fi
    
    #upgrade
    log_info "Do upgrade."
    do_upgrade "$1" "$2" "$3"
    if [ $? -ne 0 ];then
        log_error "Upgrade failed."
        upgrade_failed_handle ""
        return 1
    fi
    
    #restart process
    log_info "Restart process."
    restart_process
    if [ $? -ne 0 ];then
        log_error "Restart process failed."
        upgrade_failed_handle ""
        return 1
    fi
    
    #create patch version file
    local L_DRIVER_INSTALL_PATH="${G_HCP_MAIN_PATH}/cinder/backup"
    if [ ! -d "${L_DRIVER_INSTALL_PATH}" ];then
        log_error "The directory ${L_DRIVER_INSTALL_PATH} does not exist."
        upgrade_failed_handle ""
        return 1
    fi
    local L_INDEX=0
    L_INDEX=$(ls "${G_DIRVER_BACKUP_ROOT_DIR}" | grep "^patch.*info$" | awk -F '.' '{print $1}'| awk -F '_' '{print $2}'| sort -n | sed -n '$p')
    if [ -z "$L_INDEX" ];then
        L_INDEX=1
    else
        let L_INDEX+=1
    fi
    
    local L_PATCH_INFO="${G_DIRVER_BACKUP_ROOT_DIR}/patch_${L_INDEX}.info"
    local L_PATCH_FILES=$(find "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}" -type f | grep "cinder" | awk '{match($0,"cinder"); print substr($0,RSTART)}')
    > ${L_PATCH_INFO}
    for patch_file in ${L_PATCH_FILES}
    do
        local L_patch_file=""
        L_patch_file=`grep "^${patch_file} " "${G_PATCH_FILES_PATH}" 2>/dev/null`
        if [ $? -ne 0 ];then
            log_error "The patch file(${patch_file}) not in ${G_PATCH_FILES_PATH}."
            upgrade_failed_handle "${L_PATCH_INFO}"
            return 1
        fi
    done
    
    #copy patch text file
    cp -P --remove-destination "${G_PATCH_FILES_PATH}" "${L_PATCH_INFO}" >/dev/null 2>&1
    if [ 0 -ne $? ];then
        log_error "Copy file ${G_PATCH_FILES_PATH} to ${L_PATCH_INFO} failed."
        upgrade_failed_handle "${L_PATCH_INFO}"
        return 1
    fi
    
    echo -e "\nVersion:${G_PATCH_VERSION}" >> ${L_PATCH_INFO}
    if [ 0 -ne $? ];then
        log_error "Write version to file ${L_PATCH_INFO} failed."
        upgrade_failed_handle "${L_PATCH_INFO}"
        return 1
    fi
    
    echo "Date:`date "+%Y-%m-%d %H:%M:%S"`" >> ${L_PATCH_INFO}
    if [ 0 -ne $? ];then
        log_warn "Write date to file ${L_PATCH_INFO} failed."
    fi
    
    #copy roll back script to /opt/huawei-data-protection/ebackup/bin
    log_info "Copy rollback file."
    cp -P --remove-destination "${G_UPGRADE_TEMP_DIR}/patch_rollback.sh" "${L_DRIVER_INSTALL_PATH}/" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_warn "Copy roll back file ${G_UPGRADE_TEMP_DIR}/patch_rollback.sh to ${L_DRIVER_INSTALL_PATH}/ failed."
    fi
    
    echo "0" > ${G_UPGRADE_FINAL_RESULT}
    return 0
}

#******************************************************************#
# Function: rollback_self
# Description: roll back
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function rollback_self()
{
    log_info "Begin roll back self."
    if [ ! -f "${G_CURRENT_PATH}/patch_rollback.sh" ];then
        log_error "The roll back file does not exist."
        return 1
    fi
    
    sh "${G_CURRENT_PATH}/patch_rollback.sh" "rollback_self" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Roll back failed."
        return 1
    fi
    
    log_info "Roll back success."
    return 0
}

#******************************************************************#
# Function: clean_self
# Description: clean the tmp file of upgrade
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function clean_self()
{
    log_info "Begin clean ${G_UPGRADE_TEMP_DIR}."
    if [ -d "${G_UPGRADE_TEMP_DIR}" ];then
        log_info "rm -rf ${G_UPGRADE_TEMP_DIR}."
        rm -rf "${G_UPGRADE_TEMP_DIR}" >/dev/null 2>&1
    fi
    if [ -f "/home/${G_USER_NAME}/${G_UPGRADE_PKG_NAME}" ];then
        log_info "rm /home/${G_USER_NAME}/${G_UPGRADE_PKG_NAME}."
        rm "/home/${G_USER_NAME}/${G_UPGRADE_PKG_NAME}" >/dev/null 2>&1
    fi
    if [ -f "${G_NODE_PWD_TMP_FILE}" ];then
        log_info "rm ${G_NODE_PWD_TMP_FILE}."
        rm "${G_NODE_PWD_TMP_FILE}" >/dev/null 2>&1
    fi
    
    return 0
}

function init_kmc_log_dir()
{
    mkdir -p "${G_UPGRADE_TEMP_DIR}/logs" >/dev/null 2>&1
    touch "${G_UPGRADE_TEMP_DIR}/logs/script.log" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "touch ${G_UPGRADE_TEMP_DIR}/logs/script.log failed."
        return 1
    fi
    chown -h ${G_USER_NAME} "${G_UPGRADE_TEMP_DIR}" -R >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify ${G_UPGRADE_TEMP_DIR} permission failed."
        return 1
    fi
    
    return 0
}

function pretreatment()
{
    #check role
    get_current_role
    if [ ${G_CURRENT_ROLE} -ne 0 ];then
        echo -e "\e[0;31;1mError:This operation must be done on eBackup driver node.\e[0m"
        log_error "This operation must be done on eBackup driver node."
        return 1
    fi
    
    init_kmc_log_dir
    if [ $? -ne 0 ];then
        log_error "Init kmc log dir failed."
        return 1
    fi
    
    #check file $G_SERVERS_CONF
    if [ ! -f "${G_SERVERS_CONF}" ];then
        echo -e "\e[0;31;1mError:Config must be executed before upgrading.\e[0m"
        log_error "Config must be executed before upgrading."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: upgrade_all_nodes
# Description: upgrade all nodes
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_all_nodes()
{
    log_info "Begin upgrade all nodes."
    
    pretreatment
    if [ $? -ne 0 ];then
        log_error "Pretreatment failed."
        return 1
    fi
    
    #version check
    check_version
    if [ $? -ne 0 ];then
        log_error "Check version failed."
        return 1
    fi
    
    #upgrade all node
    local L_HAS_SAME_NODE_FAIL=0
    local L_current_ip=`cat "${G_CURRENT_NODE_FILE}" 2>/dev/null`
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            log_info "The line(${line}) was commented."
            continue
        fi

        local L_host_ip=`echo ${line} | awk -F'|' '{print $1}'`
        if [ "X${L_host_ip}" == "X" ];then
            log_warn "The node(${line}) is empty."
            continue
        fi

        if [ "X${L_host_ip}" == "X${L_current_ip}" ];then
            log_info "The node(${L_host_ip}) is current node."
            continue
        fi
        
        if [ -f "${G_UPGRADE_NODE_RESULT}" ];then
            grep "^${L_host_ip}$" "${G_UPGRADE_NODE_RESULT}" >/dev/null 2>&1
            if [ $? -eq 0 ];then
                log_info "The node(${L_host_ip}) already upgrade."
                continue
            fi
        fi
        
        local L_hcp_pwd=`echo ${line} | awk -F'|' '{print $2}'`
        local L_root_pwd=`echo ${line} | awk -F'|' '{print $3}'`
        
        echo -n "Upload the patch package to ${L_host_ip}..."
        log_info "Upload patch package to node ${L_host_ip}."
        send_cmd "dispatch" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "${G_UPGRADE_PKG_NAME}"
        if [ $? -ne 0 ];then
            log_error "Upload patch package to node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            echo "failed"
            return 1
        fi
        echo "success"
        log_info "Upload patch package to ${L_host_ip} success."
        
        echo -n "Upgrade ${L_host_ip}..."
        log_info "Upgrade node ${L_host_ip}."
        send_cmd "upgrade" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "upgrade_self" "$1" "$2" "$3"
        if [ $? -ne 0 ];then
            log_error "Upgrade node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            echo "failed"
            return 1
        fi
        echo "success"
        log_info "Upgrade ${L_host_ip} success."
        
        echo -n "Clean ${L_host_ip}..."
        log_info "Clean node ${L_host_ip}."
        send_cmd "clean" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "clean_self"
        if [ $? -ne 0 ];then
            log_warn "Clean node ${L_host_ip} failed."
            echo "failed"
            continue
        fi
        echo "success"
        log_info "Clean ${L_host_ip} success."
        
        echo "${L_host_ip}" >> "${G_UPGRADE_NODE_RESULT}"
    done < ${G_SERVERS_CONF}
    
    #upgrade self
    echo -n "Upgrade ${L_current_ip}..."
    log_info "Upgrade self."
    upgrade_self "$1" "$2" "$3" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Upgrade self failed."
        echo "failed"
        return 1
    fi
    echo "success"
    
    #check whether or not have same node failed
    if [ "0" != "${L_HAS_SAME_NODE_FAIL}" ];then
        log_error "There have same node upgrade failed, so no clean self."
        return 1
    fi
    
    #clean self
    echo -n "Clean ${L_current_ip}..."
    log_info "Clean self."
    clean_self >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Clean self failed."
        echo "failed"
        return 0
    fi
    echo "success"
    
    log_info "Upgrade success."
    return 0
}

#******************************************************************#
# Function: rollback_all_nodes
# Description: roll back all nodes
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function rollback_all_nodes()
{
    log_info "Begin roll back all nodes."
    
    pretreatment
    if [ $? -ne 0 ];then
        log_error "Pretreatment failed."
        return 1
    fi
    
    #roll back all node
    local L_HAS_SAME_NODE_FAIL=0
    local L_current_ip=`cat "${G_CURRENT_NODE_FILE}" 2>/dev/null`
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            log_info "The line(${line}) was commented."
            continue
        fi

        local L_host_ip=`echo ${line} | awk -F'|' '{print $1}'`
        if [ "X${L_host_ip}" == "X" ];then
            log_warn "The node(${line}) is empty."
            continue
        fi

        if [ "X${L_host_ip}" == "X${L_current_ip}" ];then
            log_info "The node(${L_host_ip}) is current node."
            continue
        fi
        
        if [ -f "${G_ROLLBACK_NODE_RESULT}" ];then
            grep "^${L_host_ip}$" "${G_ROLLBACK_NODE_RESULT}" >/dev/null 2>&1
            if [ $? -eq 0 ];then
                log_info "The node(${L_host_ip}) already rollback."
                continue
            fi
        fi
        
        local L_hcp_pwd=`echo ${line} | awk -F'|' '{print $2}'`
        local L_root_pwd=`echo ${line} | awk -F'|' '{print $3}'`
        
        echo -n "Upload the patch package to ${L_host_ip}..."
        log_info "Upload patch package to node ${L_host_ip}."
        send_cmd "dispatch" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "${G_UPGRADE_PKG_NAME}"
        if [ $? -ne 0 ];then
            log_error "Upload patch package to node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            echo "failed"
            continue
        fi
        echo "success"
        log_info "Upload patch package to ${L_host_ip} success."
        
        echo -n "Roll back ${L_host_ip}..."
        log_info "Roll back node ${L_host_ip}."
        send_cmd "rollback" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "rollback_self"
        if [ $? -ne 0 ];then
            log_error "Roll back node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            echo "failed"
            continue
        fi
        echo "success"
        log_info "Roll back ${L_host_ip} success."
        
        echo -n "Clean ${L_host_ip}..."
        log_info "Clean node ${L_host_ip}."
        send_cmd "clean" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "clean_self"
        if [ $? -ne 0 ];then
            log_warn "Clean node ${L_host_ip} failed."
            echo "failed"
            continue
        fi
        echo "success"
        log_info "Clean ${L_host_ip} success."
        
        echo "${L_host_ip}" >> "${G_ROLLBACK_NODE_RESULT}"
    done < ${G_SERVERS_CONF}
    
    #roll back self
    echo -n "Roll back ${L_current_ip}..."
    log_info "Roll back self."
    rollback_self >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Roll back self failed."
        echo "failed"
        return 1
    fi
    echo "success"
    
    #check whether or not have same node failed
    if [ "0" != "${L_HAS_SAME_NODE_FAIL}" ];then
        log_error "There have same node roll back failed, so no clean self."
        return 1
    fi
    
    #clean self
    echo -n "Clean ${L_current_ip}..."
    log_info "Clean self."
    clean_self >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Clean self failed."
        echo "failed"
        return 0
    fi
    echo "success"
    
    log_info "Roll back success."
    return 0
}

#******************************************************************#
# Function: clean_all_nodes
# Description: clean all nodes
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function clean_all_nodes()
{
    log_info "Begin clean all nodes."
    
    pretreatment
    if [ $? -ne 0 ];then
        log_error "Pretreatment failed."
        return 1
    fi
    
    local L_HAS_SAME_NODE_FAIL=0
    local L_current_ip=`cat "${G_CURRENT_NODE_FILE}" 2>/dev/null`
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            log_info "The line(${line}) was commented."
            continue
        fi

        local L_host_ip=`echo ${line} | awk -F'|' '{print $1}'`
        if [ "X${L_host_ip}" == "X" ];then
            log_warn "The node(${line}) is empty."
            continue
        fi

        if [ "X${L_host_ip}" == "X${L_current_ip}" ];then
            log_info "The node(${L_host_ip}) is current node."
            continue
        fi
        
        echo -n "Clean ${L_host_ip}..."
        
        local L_hcp_pwd=`echo ${line} | awk -F'|' '{print $2}'`
        local L_root_pwd=`echo ${line} | awk -F'|' '{print $3}'`
        
        log_info "Clean node ${L_host_ip}."
        send_cmd "clean" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "clean_self"
        local L_clean_ret=$?
        if [ ${L_clean_ret} -eq ${G_ERR_PATCH_FILE_NOT_EXIST} ];then
            log_info "The patch file does not exist, no need clean."
        elif [ ${L_clean_ret} -ne 0 ];then
            log_error "Clean node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            echo "failed"
            continue
        fi
        echo "success"
    done < ${G_SERVERS_CONF}
    
    #check whether or not have same node failed
    if [ "0" != "${L_HAS_SAME_NODE_FAIL}" ];then
        log_error "There have same node clean failed, so no clean self."
        return 1
    fi
    
    #clean self
    echo -n "Clean ${L_current_ip}..."
    log_info "Clean self."
    clean_self
    if [ $? -ne 0 ];then
        log_error "Clean self failed."
        echo "failed"
        return 1
    fi
    echo "success"
    
    log_info "Clean success."
    return 0
}

#******************************************************************#
# Function: upgrade_parameter_check
# Description: check parameter
# Input Parameters: 
#   $1 parameter 1
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_parameter_check()
{
    log_info "check the upgrade parameters."
    if [[ "" == "$1" ]];then
        return 0
    fi
    
    echo -e "\e[0;31;1mError:Invalid parameter.\e[0m"
    usage
    log_error "Invalid parameters."
    exit 1
}

#******************************************************************#
# Function: main
# Description: Main workflow
# Input Parameters: 
#   $1 operation
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function main()
{
    local L_OPERATION=$1
    case "${L_OPERATION}" in
        upgrade)
            upgrade_parameter_check "$2"
            log_info "Begin upgrade"
            echo "Upgrading, please wait for a moment......"
            upgrade_all_nodes "$2"
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "upgrade failed"
                echo "Upgrade failed"
                return $L_RET
            fi
            log_info "upgrade success"
            echo "Upgrade success"
            ;;
        upgrade_self)
            upgrade_parameter_check "$2"
            log_info "upgrade self"
            echo "Upgrading, please wait for a moment......"
            upgrade_self "$2"
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "upgrade self failed"
                echo "Upgrade failed"
                return $L_RET
            fi
            log_info "upgrade self success"
            echo "Upgrade success"
            ;;
        rollback)
            log_info "Begin roll back"
            echo "Rollbacking, please wait for a moment......"
            rollback_all_nodes
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "roll back failed"
                echo "Roll back failed"
                return $L_RET
            fi
            log_info "roll back success"
            echo "Roll back success"
            ;;
        rollback_self)
            log_info "roll back self"
            echo "Rollbacking, please wait for a moment......"
            rollback_self
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "roll back self failed"
                echo "Roll back failed"
                return $L_RET
            fi
            log_info "roll back self success"
            echo "Roll back success"
            ;;
        clean)
            log_info "Begin clean"
            echo "Cleaning, please wait for a moment......"
            clean_all_nodes
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "clean failed"
                echo "Clean failed"
                return $L_RET
            fi
            log_info "clean success"
            echo "Clean success"
            ;;
        clean_self)
            log_info "clean self"
            echo "Cleaning, please wait for a moment......"
            clean_self
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "clean self failed"
                echo "Clean failed"
                return $L_RET
            fi
            log_info "clean self success"
            echo "Clean success"
            ;;
        *)
            echo -e "\e[0;31;1mError:Invalid parameter.\e[0m"
            usage
            log_error "Invalid operation."
            return 1
            ;;
    esac
    
    return 0
}

main "$@"
exit $?
