# -*- coding: UTF-8 -*-
import configparser
import os
import time

import IPy
from utils.business.project_condition_utils import get_project_conditions
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.fic_base import TestCase
from utils.common.message import Message

from utils.DBAdapter.DBConnector import BaseOps

from plugins.ResourceCheck.common.libs import common_libs
from plugins.ResourceCheck.common.libs import network_libs


class DoEbackupCheck(TestCase):
    def __init__(self, project_id, pod_id):
        super(DoEbackupCheck, self).__init__(project_id, pod_id)
        self.params = ParamUtil()
        self.db_obj = BaseOps()
        self.config_dict = self.params.get_service_cloud_param(self.pod_id, "eBackup")
        self.bonding_mode = 4 if 'lacp' in str(self.config_dict['management_plane_mode']).lower() else 1

        self.script_src_path = os.path.realpath(__file__ + "/../../../../ResourceCheck/common/scripts/Network/shells/")
        self.check_type = 'network_connectivity'
        project_api = ProjectApi()
        self.project_type = project_api.get_project_info(self.project_id).get('project_type')
        logger.info("get project type and region type succ.")
        self.cascading_ip = None
        self.cascading_mask = None
        self.bmc_info = None
        self.manage_ip_list = self.config_dict['datamover_externalom_iplist'].lower().split(',')
        self.p_storage_ip_list = self.config_dict['datamover_producestorage_iplist'].lower().split(',')
        self.b_storage_ip_list = self.config_dict['datamover_backupstorage_iplist'].lower().split(',')

    def pre_condition(self):
        pass

    def deal_with_type1(self):
        condition_dic = get_project_conditions(self.project_id)
        if condition_dic.get('Type1'):
            self.cascading_ip, self.cascading_mask = self._get_conf_openstack_cascading()
            logger.info(f"get cascading_ip and cascading_mask succ,cascading_ip:{self.cascading_ip},"
                        f"cascading_mask:{self.cascading_mask}")
            # 部署工程才检查eBackup的生产存储平面
            if self.project_type == 1:
                business_storage_net_info_dic = self.get_net_info_from_lld(self.pod_id, 'cascaded_storage0',
                                                                           'cascaded')
                logger.info('business storage net info: %s ' % business_storage_net_info_dic)
        else:
            # 部署工程才检查eBackup的生产存储平面
            if self.project_type == 1:
                network_name = self.params.get_param_value(self.pod_id, 'openstack',
                                                           'openstack_storage2_name')
                if network_name == 'storage_data0':
                    network_name = 'openstack_storage0'
                else:
                    network_name = 'openstack_storage2'
                business_storage_net_info_dic = self.get_net_info_from_lld(self.pod_id, network_name, 'openstack')
                logger.info('business storage net info: %s ' % business_storage_net_info_dic)

    def get_plane_str(self, plane_name, index):
        ip_str, vlan_str, mask_str, gw_str = ('',) * 4
        if plane_name == "management_plane":
            ip_str = self.manage_ip_list[index]
            vlan_str = self.config_dict['eBackup_externalom_tag']
            mask_str = self.config_dict['eBackup_externalom_netmask']
            gw_str = self.config_dict['eBackup_externalom_gateway']
        elif plane_name == "backup_storage_plane":
            if not self.config_dict.get('datamover_backupstorage_iplist'):
                return ip_str, vlan_str, mask_str, gw_str
            ip_str = self.b_storage_ip_list[index]
            mask_str = self.config_dict['datamover_backupstorage_netmask']
            gw_str = self.config_dict['datamover_backupstorage_gateway']
        elif plane_name == "production_storage_plane":
            if not self.config_dict.get('datamover_producestorage_iplist'):
                return ip_str, vlan_str, mask_str, gw_str
            ip_str = self.p_storage_ip_list[index]
            mask_str = self.config_dict['datamover_producestorage_netmask']
            gw_str = self.config_dict['datamover_producestorage_gateway']
        return ip_str, vlan_str, mask_str, gw_str

    def check_params(self):
        logger.info("get bmc info succ.")
        pci_obj = network_libs.NicPciCls(self.bmc_info)
        plane_list = ["management_plane", "production_storage_plane", "backup_storage_plane"]
        error_plane = []
        for index, sub_bmc_info in enumerate(self.bmc_info):
            bmc_ip_lower = sub_bmc_info['bmc_ip'].lower()
            for plane_name in plane_list:
                ip_str, vlan_str, mask_str, gw_str = self.get_plane_str(plane_name, index)
                if get_project_conditions(self.project_id).get("TenantStorFCSAN") and \
                        plane_name == "production_storage_plane" and not sub_bmc_info.get(plane_name):
                    logger.warning(f"FCSAN {plane_name} is not exist, skip check.")
                    continue
                slot_list = sub_bmc_info[plane_name].replace(' ', '').split(',')
                host_nic_list = [pci_obj.get_nic_name_by_slot(bmc_ip_lower, slot) for slot in slot_list]
                logger.info(f"{plane_name} get pci info success, start to check this plane.")
                nic_list = ','.join(host_nic_list)

                timestamp = str(int(time.time()))
                parm_dic = {'NICBONDMODEMAPS': self.bonding_mode, 'TIMESTAMP': timestamp,
                            'CHECKTYPE': 'network_connectivity', 'NICIPMAPS': '(' + ip_str + ')',
                            'NICVLANMAPS': '(' + vlan_str + ')', 'NETMASKMAPS': '(' + mask_str + ')',
                            'CHECKIPMAPS': '(' + gw_str + ')', 'ETHNICS': '(' + nic_list + ')'}

                script_src_file = self.script_src_path + '/check_network_connectivity.sh'
                common_libs.create_check_script(script_src_file, bmc_ip_lower, **parm_dic)
                try:
                    common_libs.check_all_result([bmc_ip_lower], self.check_type, timestamp)
                except Exception as e:
                    logger.error(f"check {plane_name} failed, please check the param! Error info: {e}")
                    error_plane.append(plane_name)
        if error_plane:
            raise HCCIException(653108, str(error_plane))

    def procedure(self):
        self.deal_with_type1()
        # 刷新硬件检查数据库中IP
        bmc_ip_list = []
        self.bmc_info = self.db_obj.get_ebackup_install_info(self.pod_id)
        for sub_bmc_info in self.bmc_info:
            bmc_ip_lower = sub_bmc_info['bmc_ip'].lower()
            bmc_ip_list.append(bmc_ip_lower)
        common_libs.get_ssh_ip(bmc_ip_list)
        self.check_params()

    def do_get_conf_openstack_cascading(self):
        config = configparser.ConfigParser()
        conf_path = os.path.realpath(__file__ + "/../../../conf/openstack.conf")
        config.read(conf_path)

        pro_dic = get_project_conditions(self.project_id)
        cascading_ip_mask_str = network_libs.HCSDNetCls(self.project_id,
                                                        self.pod_id).get_internal_base_network()
        if 'ManageIPV6' in pro_dic:
            cascading_ip_mask_str = cascading_ip_mask_str.replace(' ', '')
            cascading_ip_mask = cascading_ip_mask_str.split('/')
            cascading_ip = cascading_ip_mask[0]
            cascading_mask = cascading_ip_mask[1]
        else:
            cascading_ip_mask_str = cascading_ip_mask_str.replace(' ', '')
            cascading_ip_mask = cascading_ip_mask_str.split('/')
            cascading_ip = cascading_ip_mask[0]
            cascading_mask = IPy.IP(cascading_ip_mask_str).netmask()
        return cascading_ip, str(cascading_mask)

    def _get_conf_openstack_cascading(self):
        try:
            return self.do_get_conf_openstack_cascading()
        except Exception as e:
            raise Exception("get openstack cascading info faild. %s" % str(e)) from e

    @staticmethod
    def get_net_info_from_lld(pod_id, network_name, condition="openstack"):
        param_util = ParamUtil()
        _tag = param_util.get_param_value(pod_id, condition, network_name + "_tag")
        _vlan = param_util.get_param_value(pod_id, condition, network_name + "_vlan")
        if _tag is None:
            vlan = _vlan
        else:
            vlan = _tag
        net_info_dic = {
            "ip_start": param_util.get_param_value(pod_id, condition, network_name + "_start_addr"),
            "gateway": param_util.get_param_value(pod_id, condition, network_name + "_gateway"),
            "netmask": param_util.get_param_value(pod_id, condition, network_name + "_netmask"),
            "name": network_name,
            "vlan": vlan
        }
        return net_info_dic

    def post_condition(self):
        pass

    def failure(self):
        super(DoEbackupCheck, self).failure()


class EBackupCheck(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super(EBackupCheck, self).__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.db = BaseOps()
        self.implement = None

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def main_handle(self, project_id, pod_id):
        self.implement = DoEbackupCheck(project_id, pod_id)
        self.implement.procedure()
        return Message(200)

    def execute(self, project_id, pod_id):
        try:
            return self.main_handle(project_id, pod_id)
        except HCCIException as e1:
            return Message(500, e1)
        except Exception as e2:
            return Message(500, HCCIException('653012', e2))

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass
