# -*- coding:utf-8 -*-
import time
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.common import log as logger
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.ebackup_rest import EbackupRest


class ConfigHA(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        logger.init("eBackup")
        # params
        self.params = ParamUtil()
        self.__db_param_dict = self.params.get_service_cloud_param(
            pod_id, "eBackup")
        self.project_id = project_id
        self.pod_id = pod_id
        self.ssh_obj = Ssh()
        self.err_code = 0
        self.description = ''
        self.token = ''
        self.session_content = ''
        self.headers = ''
        self.rest_api = EbackupRest()
        self.datamover_iplist = None
        self.base_url = None

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def main_handle(self):
        externalom_ip_lower = self.__db_param_dict['datamover_externalom_iplist'].lower()
        iplist = externalom_ip_lower.split(',')
        self.datamover_iplist = [ipStr.lower() for ipStr in iplist]
        self.base_url = "https://%s:8088/rest/dev" % self.datamover_iplist[0]
        self.first_login_and_modify_pwd()
        self.add_ha_member()
        self.logout()
        if self.err_code != 0:
            if self.description == '':
                raise HCCIException(self.err_code)
            else:
                raise HCCIException(self.err_code, self.description)

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        if get_project_condition_boolean(project_id, 'eBackup_Proxy'):
            return Message(200)
        try:
            self.main_handle()
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(
                "Failed to install and config eBackup(datamover) ,"
                "the reason is %s" % e)
            return Message(500, e, e)
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            return Message(200)
        except HCCIException as e:
            return Message(500, e)
        except Exception as e:
            logger.error(
                "Failed to rollback:[install]2.Install and "
                "configure eBackup(datamover) ,the reason is %s" % str(e))
            return Message(500, e, e)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def update_error_info(self, err, des):
        if self.err_code == 0:
            self.err_code = err
            self.description = des

    def login(self, pwd):
        logger.info("begin to login ebackup.")
        login_url = self.base_url + '/login'
        headers = {'Accept': 'application/json'}
        # login use pwd
        data = '{"scope":0,"username":"admin","password":"' + pwd + '"}'
        rsp = self.rest_api.post(login_url, headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise HCCIException(653020)
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise HCCIException(653020)
        if errcode == 0:
            self.token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            self.session_content = cookie.split(';')[0]
            self.headers = {
                'Accept': 'application/json;version=2.2;charset=UTF-8',
                'iBaseToken': self.token,
                'Cookie': 'language=en;' + self.session_content + '; '
                'DEVICE_ID=dev; sessionIdleTime=60000; MACHINEROLE=0;'
                ' CSRF_IBASE_TOKEN=' + self.token}
            logger.info("login ebackup succ.")
        else:
            description = rsp.json()["error"]["description"]
            logger.error("login failed,description:" + description)

    def first_login_and_modify_pwd(self):
        self.login(self.__db_param_dict['eBackup_weblogin_password'])
        if self.token == '':
            # login use default pwd
            logger.info("login by default pwd.")
            self.login(
                self.__db_param_dict['eBackup_weblogin_default_password'])
            if self.token == '':
                raise HCCIException(653022)

            logger.info("login by default pwd succ.")
            # modify pwd
            modify_pwd_url = self.base_url + '/user'
            data = '{"ID":"admin","PASSWORD":"' + self.__db_param_dict[
                'eBackup_weblogin_password'] + '","OLDPASSWORD":"' + \
                self.__db_param_dict['eBackup_weblogin_default_password'] +\
                '"}'
            headers = {
                'Accept': 'application/json', 'iBaseToken': self.token,
                'Cookie': 'language=en;' + self.session_content + '; '
                'DEVICE_ID=dev; sessionIdleTime=60000; MACHINEROLE=0;'
                ' CSRF_IBASE_TOKEN=' + self.token}
            rsp = self.rest_api.put(modify_pwd_url, headers, data)
            if rsp.status_code != 200:
                logger.error("rest req return not 200.")
                raise HCCIException(653020)
            errcode = rsp.json()["error"]["code"]
            description = rsp.json()["error"]["description"]
            if errcode is None:
                raise HCCIException(653019)
            if errcode != 0:
                logger.error("modify pwd failed,description:" + description)
                raise HCCIException(653021, description)

            logger.info("modify pwd succ.")
            # login again
            self.login(self.__db_param_dict['eBackup_weblogin_password'])
            if self.token == '':
                raise HCCIException(653022)
            logger.info("login again succ.")
        logger.info("login succ.")
        self.headers = {
            'Accept': 'application/json;version=2.2;charset=UTF-8',
            'iBaseToken': self.token,
            'Cookie': 'language=en;' + self.session_content + '; '
            'DEVICE_ID=dev; sessionIdleTime=60000; MACHINEROLE=0; '
            'CSRF_IBASE_TOKEN=' + self.token}

    def is_ha_member(self, ip, is_primary):
        cmd = "su -s /bin/bash hcpprocess -c 'sh /opt/huawei-data-protection/ebackup/bin/" \
              "config_omm_ha.sh query'"
        if is_primary:
            match_str1 = "HaLocalName: ha1(active)\n"
            match_str2 = "HaPeerName:  ha2(standby)\n"
        else:
            match_str1 = "HaLocalName: ha2(standby)\n"
            match_str2 = "HaPeerName:  ha1(active)\n"
        ssh_client = self.ssh_obj.ssh_create_client(
            ip, "root", self.__db_param_dict['eBackup_os_root_password'])
        result = self.ssh_obj.ssh_exec_command_return(ssh_client, cmd)
        self.ssh_obj.ssh_close(ssh_client)
        if "last cmd result: 0\n" in result:
            logger.info("exec query ha status cmd succ.")
            if "HaMode:      double\n" in result and\
                    match_str1 in result and match_str2 in result:
                logger.info("the node(%s) has been ha mode." % ip)
                return 0
            else:
                logger.info("the node(%s) is not ha mode." % ip)
                return 1
        else:
            logger.info("exec query ha status cmd failed on node(%s)." % ip)
            self.update_error_info(653060, ip)
            return 2

    def get_node_index(self):
        url = self.base_url + '/vbackup_server'
        rsp = self.rest_api.get(url, self.headers)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.update_error_info(653020, '')
            return ''
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            self.update_error_info(653019, '')
            return ''

        if errcode != 0:
            logger.error("query server info failed,description:" + description)
            self.update_error_info(653035, description)
            return ''

        logger.info("query server list succ.")
        id_index = ''
        server_list = rsp.json()["data"]
        for sub in server_list:
            if sub['MANAGEMENTIP'] == self.datamover_iplist[1]:
                id_index = sub['ID']

        return id_index

    def add_ha_member(self):
        if len(self.datamover_iplist) != 1:
            primary_status = self.is_ha_member(self.datamover_iplist[0], True)
            standby_status = self.is_ha_member(self.datamover_iplist[1], False)
            if primary_status == 2:
                return
            if standby_status == 2:
                return
            if primary_status == 0 and standby_status == 0:
                return
            id_index = self.get_node_index()

            if not self.exec_add_ha_req(id_index):
                return
            logger.info("exec add ha req succ.")

            timeout = 0
            while timeout < 20:
                time.sleep(30)
                primary_status = self.is_ha_member(
                    self.datamover_iplist[0], True)
                standby_status = self.is_ha_member(
                    self.datamover_iplist[1], False)
                if primary_status == 0 and standby_status == 0:
                    logger.info("add ha member succ.")
                    return
                timeout += 1
            logger.info("add ha member time out.")
            self.update_error_info(653037, "time out")
        else:
            logger.info("ebackup single node, no need to add ha member.")

    def exec_add_ha_req(self, id_index):
        if id_index == '':
            self.update_error_info(653060, self.datamover_iplist[0])
            return False
        data = \
            '{"SERVERIDLIST":"' + id_index + '","FLOATIP":"' + \
            self.__db_param_dict['datamover_management_float_ip'].lower() \
            + '","GATEWAY":["' + \
            self.__db_param_dict['eBackup_externalom_gateway'] + \
            '"],"TYPE":0}'

        url = self.base_url + '/vbackup_ha/ADD'
        rsp = self.rest_api.put(url, self.headers, data)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            self.update_error_info(653020, '')
            return False
        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            self.update_error_info(653019, '')
            return False
        if errcode != 0:
            logger.error("add ha member failed,description:" + description)
            self.update_error_info(653037, description)
            return False
        return True

    def logout(self):
        if self.token == '' or self.session_content == '':
            return
        url = self.base_url + '/sessions'
        self.rest_api.delete(url, self.headers)
        logger.info("log out succ.")

# the code has been updated to python3.7
