# -*- coding:utf-8 -*-
import time
import utils.common.log as logger
from utils.DBAdapter.DBConnector import BaseOps
from utils.common.exception import HCCIException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.eBackup.common.util import Utils

RETRY_TIME_LIMIT = 10


class DatamoverStatusCheck(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DatamoverStatusCheck, self).__init__(project_id,
                                                   pod_id,
                                                   regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.param_dict = Utils.init_system_params(project_id, regionid_list[0])

    @staticmethod
    def check_is_all_ok(server_ips, ssh, check_cmd, hcp_password, root_password):
        all_ok = True
        for ebk_ip in server_ips:
            result = ssh.ssh_cmds(ebk_ip, check_cmd, "hcp", hcp_password, root_password, "", "")
            if "execute ebk_cmd result: 0" not in result:
                logger.error("datamover node(%s) is not running." % ebk_ip)
                all_ok = False
        return all_ok

    @staticmethod
    def calculate_sha256sum(datamover_ips, ssh, check_sha256sum_cmd, hcp_password, root_password, server_ips):
        md5sum_value = ""
        for ebk_ip in datamover_ips:
            result = ssh.ssh_cmds(ebk_ip, check_sha256sum_cmd, "hcp", hcp_password, root_password, "", "")
            logger.info(str(result))
            if md5sum_value:
                if md5sum_value not in str(result):
                    logger.error(f"version check failed, node({ebk_ip})version is not same with node({server_ips[0]}).")
                    return Message(500, "Datamover version check failed.")
            else:
                md5sum_value = result[0]
        return ''

    def do_execute(self):
        ssh = Ssh()
        hcp_password = self.param_dict["eBackup_hcp_pwd"]
        root_password = self.param_dict["eBackup_root_pwd"]
        server_ips = self.param_dict['eBackup_Datamover_nodes']
        server_ips = server_ips.replace(" ", "").replace("|", ";").split(";")
        logger.info("Begin to check nodes:" + str(server_ips))
        # 1.check service hcp status
        retry_times = 1
        check_cmd = "service hcp status>/dev/null;echo execute ebk_cmd result: $?"
        server_status_ok = False
        while retry_times < RETRY_TIME_LIMIT:
            is_all_ok = self.check_is_all_ok(server_ips, ssh, check_cmd, hcp_password, root_password)
            if is_all_ok:
                logger.info("StatusCheck nodes(%s) completed." % server_ips)
                server_status_ok = True
                break
            time.sleep(60)
            retry_times = retry_times + 1
        if not server_status_ok:
            logger.error("Status Check failed after 10 min.")
            return Message(500, "Datamover Status Check failed after 10 min.")
        # 2.check upgraded Version
        check_sha256sum_cmd = "sha256sum /opt/huawei-data-protection/ebackup/bin/AdminNode | awk '{print $1}'"

        datamover_groups_ips = self.param_dict['eBackup_Datamover_nodes'].replace(" ", "").split("|")
        for datamover_gourps_ip_str in datamover_groups_ips:
            datamover_ips = datamover_gourps_ip_str.split(";")
            if len(datamover_ips) == 1:
                logger.info("Single datamover node, skip check MD5.")
                continue
            res = self.calculate_sha256sum(datamover_ips, ssh, check_sha256sum_cmd, hcp_password, root_password,
                                           server_ips)
            if res != '':
                return res
        return Message(200, check_results=[])

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self.do_execute()
        except HCCIException as e:
            logger.error("StatusCheck failed: " + str(e))
            return Message(500, e)

        except Exception as e:
            logger.error("Failed to exec StatusCheck ,the reason is %s" % e)
            return Message(500, error_msg_cn="状态检查出现异常，请查看日志并重试",
                           error_msg_en="Exception occurs while Status "
                                        "Checking,please check the log and"
                                        " try again.")

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class DriverStatusCheck(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DriverStatusCheck, self).__init__(project_id, pod_id,
                                                regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region_id = regionid_list[0]
        self.params_dict = Utils.init_system_params(project_id,
                                                    regionid_list[0])
        self.fsp_passwd = self.params_dict['openstack_fsp_pwd']
        self.fsp_sudo_password = self.params_dict['openstack_root_pwd']
        self.cps_user_pwd = self.params_dict['openstack_cps_admin_pwd']
        self.region_type = BaseOps().get_region_type(project_id).lower()

    @staticmethod
    def check_is_all_ok(template_list, ssh, ssh_client):
        all_ok = True
        for sub_template in template_list:
            check_cmd = "cps template-instance-list --service cinder " + sub_template + \
                        " | grep -w 'cinder-backup' | grep  -v active | awk '{print $10}'"
            ip_list = ssh.ssh_exec_command_return_list(ssh_client, check_cmd)
            logger.info("result is %s." % str(ip_list))
            if 0 != len(ip_list):
                all_ok = False
                ip_str = ",".join(ip_list)
                logger.info("cinder-backup(%s) is not ok." % ip_str)
        return all_ok

    def do_execute(self):
        ssh = Ssh()
        reverse_proxy_ip = self.params_dict['openstack_reverse_proxy_ip']
        logger.info("reverse_proxy_ip is %s" % reverse_proxy_ip)

        set_env_cmd = "source set_env< <(echo 2;echo 'cps_admin';echo '%s')" % self.cps_user_pwd
        ssh_client = ssh.ssh_create_client(reverse_proxy_ip, "fsp", self.fsp_passwd)
        ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
        ssh.ssh_send_command(ssh_client, self.fsp_sudo_password, '#', 100)
        ssh.ssh_exec_command_return_list(ssh_client, set_env_cmd, 100)
        query_template_cmd = "cps template-list |grep 'cinder-backup-kvm' |awk '{print $4}'"
        template_list = ssh.ssh_exec_command_return_list(ssh_client, query_template_cmd)
        logger.info("template list is %s." % str(template_list))

        retry_times = 1
        ip_str = ""
        while retry_times < RETRY_TIME_LIMIT:
            is_all_ok = self.check_is_all_ok(template_list, ssh, ssh_client)
            if is_all_ok:
                logger.info("ok.")
                break
            time.sleep(60)
            retry_times = retry_times + 1
        if retry_times == RETRY_TIME_LIMIT:
            logger.error("cinder-backup-kvm(%s) is not active." % ip_str)
        return Message(200, check_results=[])

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self.do_execute()
        except HCCIException as e:
            logger.error("status check driver failed: " + str(e))
            return Message(500, e)
        except Exception as e:
            logger.error("Failed to check eBackup Driver status, the reason is %s" % e)
            return Message(500, error_msg_cn="状态检查出现异常，请查看日志并重试",
                           error_msg_en="Exception occurs while status checking,please check the log and try again.")

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)
