# -*- coding:utf-8 -*-
import base64
import json
import logging
import os
import getpass
import sys

import requests
from FSSysconf import netPolicyParse

logger = logging.getLogger('__EBACKUP_CONFIG__')
logger.setLevel(level=logging.DEBUG)
logger.propagate = False
logger.handlers = []

log_file_path = os.path.realpath(__file__ + '/../dmk_driver.log')
if not os.path.isfile(log_file_path):
    os.mknod(log_file_path)

handler = logging.FileHandler(log_file_path)
handler.setLevel(logging.DEBUG)
formatter = logging.Formatter('%(asctime)s [%(levelname)s] at %(filename)s,%(lineno)d: %(message)s')
handler.setFormatter(formatter)
logger.addHandler(handler)

VALID_VOLUME_DRIVER = ['cinder.volume.drivers.dsware.HuaweiDswareDriver',
                       'cinder.volume.drivers.huawei.vrm.vrm_driver.VRMDriver',
                       'cinder.volume.drivers.huawei.HuaweiISCSIDriver']
global CPS_BASE_URL


def get_cps_server_base_url():
    def do_get_cps_server_base_url():
        floating_ip = netPolicyParse.getComponentFloatingIp(compName="cps-server")
        floating_port = netPolicyParse.getComponentFloatingPort(compName="cps-server")
        base_url = floating_ip + ":" + floating_port
        if floating_ip.find(":") > 0:
            base_url = '[' + floating_ip + ']' + ":" + floating_port
        logger.info("Get base url by interface: %s" % base_url)
        return base_url.replace(' ', '')

    try:
        return do_get_cps_server_base_url()
    except Exception as e:
        logger.info("Not found cps-server maybe it's cascaded %s!" % e)
        file_path = "/etc/huawei/fusionsphere/cfg/serverIp.ini"
        if os.path.exists(file_path):
            return get_cps_server_base_url_from_ini(file_path)
        else:
            logger.error(f"{file_path}is not existed.")
            return ""


def get_cps_server_base_url_from_ini(file_path):
    fp = open(file_path, 'r')
    logger.info('Use config file : %s' % fp)
    base_url = ''
    while True:
        line = fp.readline().strip('\n')
        if not line:
            break
        key_value = line.split("=")
        if key_value[0].strip() == 'serverip':
            base_url = key_value[1].strip()
            base_url = base_url.replace("https://", "")
            logger.info("Get base url from serverIP.ini: %s", base_url)
    fp.close()
    return base_url.replace(' ', '')


def parse_cps_template_response(rsp, cinder_backup_templates):
    rsp_dict = rsp.json()
    for service in rsp_dict["templates"]:
        if -1 == service["name"].find("cinder-backup"):
            continue
        cinder_backup_templates.append(service["name"])


def get_cinder_backup_templates(cps_username, cps_password):
    def do_get_cinder_backup_templates():
        global CPS_BASE_URL
        logger.info("Begin to query all cinder backup template.")
        cinder_backup_templates = []
        url = "https://%s/cps/v1/services/all/componenttemplates?" \
              "commit_state=commited" % CPS_BASE_URL
        header = {
            "Content-Type": "application/json",
            "X-Auth-User": cps_username,
            "X-Auth-Password": cps_password
        }

        rsp = requests.get(url=url, headers=header, verify=False)
        if rsp.status_code == 200:
            parse_cps_template_response(rsp, cinder_backup_templates)
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")
        if 0 == len(cinder_backup_templates):
            logger.debug("There is no cinder backup template.")
            return True, []
        logger.info("Query all cinder backup template successfully.")
        return True, cinder_backup_templates

    try:
        return do_get_cinder_backup_templates()
    except Exception as e:
        logger.error("Query all cinder backup template failed.The reason is " + str(e))
        return False, []


def is_other_volume_driver_valid(other_storage_cfg):
    other_volume_driver = ''
    for _, poollist in other_storage_cfg.items():
        for _, poolcfg in poollist.items():
            if poolcfg.get('volume_driver', ''):
                other_volume_driver = poolcfg.get(
                    'volume_driver', '')
    logger.info('Other_volume_driver: %s' % other_volume_driver)
    if other_volume_driver in VALID_VOLUME_DRIVER:
        return True
    return False


def add_template_to_supported(template, template_params, supported_templates):
    volume_driver = template_params.get('volume_driver', '')
    other_storage_cfg = \
        template_params.get('other_storage_cfg', '')
    logger.debug("Template:%s, volume_driver: %s, other_storage_cfg: %s", template, volume_driver, other_storage_cfg)
    if template not in supported_templates:
        if volume_driver in VALID_VOLUME_DRIVER or \
                (other_storage_cfg and is_other_volume_driver_valid(
                    other_storage_cfg)):
            supported_templates.append(template)


def get_supported_backup_templates(cinder_backup_templates, cps_username, cps_password):
    def do_get_supported_backup_templates():
        global CPS_BASE_URL
        supported_templates = []
        logger.info("Begin to query supported backup template.")
        header = {
            "Content-Type": "application/json",
            "X-Auth-User": cps_username,
            "X-Auth-Password": cps_password
        }
        for template in cinder_backup_templates:
            logger.info('Start check template %s' % template)
            url = "https://%s/cps/v1/services/cinder/componenttemplates" \
                  "/%s/params?commit_state=commited" % (CPS_BASE_URL, template)
            logger.debug("The url is %s" % url)
            rsp = requests.get(url=url, headers=header, verify=False)
            if rsp.status_code == 200:
                template_params = rsp.json()["cfg"]
                add_template_to_supported(
                    template, template_params, supported_templates)
            else:
                logger.error("Response status code is not 200.The url is %s" % url)
                raise Exception("Response status code is not 200.")
        if 0 == len(supported_templates):
            logger.error("There are no supported templates.")
            return True, []
        logger.info("The supported templates are: %s" % str(supported_templates))
        return True, supported_templates

    try:
        return do_get_supported_backup_templates()
    except Exception as e:
        logger.error("Query supported template failed,the reason is %s" % str(e))
        return False, []


def query_cinder_template_instance(ebackup_driver_nodes, template, header):
    url = "https://%s/cps/v1/instances?service=cinder&template=%s" % (
        CPS_BASE_URL, template)
    rsp = requests.get(url=url, headers=header, verify=False)
    if rsp.status_code == 200:
        instance_list = rsp.json()["instances"]
        for host in instance_list:
            omip = host["omip"]
            if omip is not None and omip != '' and \
                    omip not in ebackup_driver_nodes:
                ebackup_driver_nodes.append(omip)
    else:
        logger.error("Response status code is not 200.The url is %s" % url)
        raise Exception("Response status code is not 200.")


def get_ebackup_driver_node2(supported_templates, cps_username, cps_password):
    def do_get_ebackup_driver_node2():
        logger.info("Begin to query eBackup driver node by supported template.")
        header = {
            "Content-Type": "application/json",
            "X-Auth-User": cps_username,
            "X-Auth-Password": cps_password
        }
        ebackup_driver_nodes = []
        for template in supported_templates:
            query_cinder_template_instance(
                ebackup_driver_nodes, template, header)
        if 0 == len(ebackup_driver_nodes):
            logger.error("There is no ebackup driver node.")
            return False, []
        logger.info("Query eBackup driver node successfully.The eBackup driver node are:" + str(ebackup_driver_nodes))
        return True, ebackup_driver_nodes

    try:
        return do_get_ebackup_driver_node2()
    except Exception as e:
        logger.error("Query eBackup driver node failed,the reason is " + str(e))
        return False, []


def query_cps_server_node(url, cps_username, cps_password):
    cps_server_node_list = []
    header = {
        "Content-Type": "application/json",
        "X-Auth-User": cps_username,
        "X-Auth-Password": cps_password
    }
    rsp = requests.get(url=url, headers=header, verify=False)
    if rsp.status_code == 200:
        rsp_dict = rsp.json()
        for cps in rsp_dict["instances"]:
            omip = cps["omip"]
            if omip is not None and omip != '':
                cps_server_node_list.append(omip)
    return cps_server_node_list


def get_cps_server_node(cps_username, cps_password):
    def do_get_cps_server_node():
        global CPS_BASE_URL
        logger.info("Begin to query cps server nodes.")
        url = "https://%s/cps/v1/instances?service=cps-server&template=cps-server" % CPS_BASE_URL
        cps_server_node_list = query_cps_server_node(
            url, cps_username, cps_password)
        if not cps_server_node_list:
            logger.info("The cps-server service name is cps")
            url = "https://%s/cps/v1/instances?service=cps&template=cps-server" % CPS_BASE_URL
            cps_server_node_list = query_cps_server_node(url, cps_username, cps_password)
            if not cps_server_node_list:
                logger.error("Response status code is not 200.The url is %s" % url)
                raise Exception("Response status code is not 200.")

        if 0 == len(cps_server_node_list):
            logger.error("There is no cps server.")
            return False, []
        logger.info("Query cps server nodes successfully.The cps server are:%s" % str(cps_server_node_list))
        return True, cps_server_node_list

    try:
        return do_get_cps_server_node()
    except Exception as e:
        logger.error("Query cps server nodes failed,the reason is %s" % str(e))
        return False, []


def check_cps_admin_password(cps_username, cps_password):
    def do_check_cps_admin_password():
        global CPS_BASE_URL
        logger.info("Begin to check the password of the %s user." % cps_username)
        url = "https://%s/cps/v1/haproxylocalurl" % CPS_BASE_URL
        header = {
            "Content-Type": "application/json",
            "X-Auth-User": cps_username,
            "X-Auth-Password": cps_password
        }

        rsp = requests.get(url=url, headers=header, verify=False)
        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            if -1 != str(rsp_dict).lower().find("failed"):
                logger.error("Authenticate Failed!")
                return False
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")
        logger.info(f"{cps_username} Authenticate successfully.")
        return True

    try:
        return do_check_cps_admin_password()
    except Exception as e:
        logger.error("Authenticate Failed!The reason is: %", str(e))
        return False


def is_cutovere_completed(rsp, url):
    if rsp.status_code == 200:
        instance_list = rsp.json()["instances"]
        is_completed = (0 != len(instance_list))
        if not is_completed:
            logger.error("Cinder backup deploy info cutover is not completed")
            return False
    else:
        logger.error(
            "Response status code is not 200.The url is %s" % url)
        raise Exception("Response status code is not 200.")
    return True


def check_cinder_backup_cutovered(supported_templates, cps_username, cps_password):
    def do_check_cinder_backup_cutovered():
        global CPS_BASE_URL
        logger.info("Begin to check if all cinder backup deploy info has been cutovered.")
        header = {
            "Content-Type": "application/json",
            "X-Auth-User": cps_username,
            "X-Auth-Password": cps_password
        }
        for template in supported_templates:
            url = "https://%s/cps/v1/instances?service=cinder&template=%s" % (CPS_BASE_URL, template)
            rsp = requests.get(url=url, headers=header, verify=False)
            if not is_cutovere_completed(rsp, url):
                return False, {"Error": "Cutover-unfinished"}

        logger.info("All cinder backup deploy info has been cutovered.")
        return True, {}

    try:
        return do_check_cinder_backup_cutovered()
    except Exception as e:
        logger.error("Check if all cinder backup deploy info has been cutovered failed,the reason is %s", str(e))
        return False, {"Error": "Cutover-exception"}


def decode_password(pass_word):
    pass_word = base64.b64decode(pass_word)
    res = check_cps_admin_password(cps_user_name, pass_word)
    return res, pass_word


if __name__ == "__main__":
    if sys.argv.__len__() != 3:
        logger.error("Parameter is Invalid.")
        print("ERROR: Parameter is Invalid.")
        exit(1)
    CPS_BASE_URL = get_cps_server_base_url()
    cps_user_name = sys.argv[1]
    operation = sys.argv[2]
    password = getpass.getpass("CPS_PASSWORD:")
    RESULT = ''
    try:
        RESULT, password = decode_password(password)
    except Exception as ex:
        logger.error(f"Decode the password of the cps_admin user failed, error:{str(ex)}.")
        print("ERROR: Decode the password of the cps_admin user failed.")
        exit(1)
    if not RESULT:
        logger.error(f"Check the password of the {cps_user_name} user failed.")
        print(f"ERROR: Authenticate Failed!Please check the password of the {cps_user_name} user")
        exit(1)
    valid_operation = ["upgrade", "check_cutover"]
    if operation not in valid_operation:
        logger.error("Operation is invalid.")
        print("ERROR: Operation is invalid.The valid operation are:" + str(valid_operation))
        exit(1)

    RESULT, backup_templates = get_cinder_backup_templates(cps_user_name, password)
    if not RESULT:
        logger.error("Get cinder backup templates failed.")
        print("ERROR: Get cinder backup templates failed.")
        exit(1)
    elif 0 == len(backup_templates):
        logger.debug("There is no cinder backup templates")
        print("DEBUG: Unsupported-eBackup.")
        exit(0)
    RESULT, supported_template = get_supported_backup_templates(backup_templates, cps_user_name, password)
    if not RESULT:
        logger.error("Get cinder backup templates which support eBackup failed.")
        print("ERROR: Get cinder backup templates which support eBackup failed.")
        exit(1)
    elif 0 == len(supported_template):
        logger.debug("There is no supported cinder backup templates")
        print("DEBUG: Unsupported-eBackup.")
        exit(0)

    if operation == "check_cutover":
        RESULT, error_msg = check_cinder_backup_cutovered(supported_template, cps_user_name, password)
        if not RESULT:
            print(json.dumps(error_msg))
            exit(1)
        exit(0)
    else:
        RESULT, driver_nodes = get_ebackup_driver_node2(supported_template, cps_user_name, password)
        if 0 == len(driver_nodes):
            logger.error("Get ebackup driver node failed.")
            print("ERROR: Get ebackup driver node failed.")
            exit(1)
        if operation == "upgrade":
            RESULT, cps_server = get_cps_server_node(cps_user_name, password)
            if not RESULT:
                logger.error("Get cps server failed.")
                print("ERROR: Get cps server failed.")
                exit(1)

            print(json.dumps({"cpsnode": cps_server}))
            print(json.dumps({"eBackupDriver": driver_nodes}))
            exit(0)
