# -*- coding:utf-8 -*-

import utils.common.log as logger
from utils.common.exception import HCCIException

from plugins.eReplication.common.constant import Capacity
from plugins.eReplication.common.constant import VMName
from plugins.eReplication.common.constant import VMScale
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.scale_api import API as SCALE_API

logger.init("eReplication_calc")


def resource_calc(project_id, pod_id):
    """
    预计扩容资源容量
    :param project_id: 工程ID
    :param pod_id:
    :return: 所需资源信息
    """
    try:
        return _do_create_resource_list(pod_id, project_id)
    except HCCIException as e:
        logger.error(f"Calculate eReplication expand resource failed: {e}.")
        raise e
    except Exception as e:
        logger.error(f"Calculate eReplication expand resource failed: {e}.")
        raise HCCIException('663600', e)


def _do_create_resource_list(pod_id, project_id):
    logger.info("Begin to calculate eReplication expand resource.")
    # eReplication扩容的虚拟机有2个，一共只有两个档位，部署时默认是4U8G。
    # 扩容规模固定从4U8G扩到8U16G，系统盘不变，如果已经是目标档位，不返回资源。
    expand_resource = {
        "type": "vm", "host_id": None, "cpu": VMScale.CPU_4,
        "mem": VMScale.RAM_8, "sys_disk": VMScale.DISK_0,
        "sys_disk_type": "remote", "data_disk_total": VMScale.DISK_0,
        "data_disk_type": "remote"
    }
    params = Params(project_id, pod_id)
    vm_mem = int(params.mem_num)
    vm_cpu = int(params.cpu_num)
    # 低档位扩容时只需要刷新CMDB，不需要真实扩规格，不需要返回资源
    vms_detail = SCALE_API.query_cpu_mem(pod_id, VMName.VM_NAME_PREFIX)
    resource_lst = list()
    for vm_info in vms_detail:
        origin_vcpus = int(vm_info.get("vcpus"))
        try:
            origin_ram = int(vm_info.get("ram")) / Capacity.ONE_THOUSAND
        except ZeroDivisionError as e:
            logger.error(
                f"{Capacity.ONE_THOUSAND} cannot be used as a divisor.")
            raise e
        if (origin_vcpus == VMScale.CPU_4 and
            origin_ram == VMScale.RAM_8) and (
                vm_cpu == VMScale.CPU_8 and vm_mem == VMScale.RAM_16):
            resource_lst.append(expand_resource)
    return resource_lst
