import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.ssh_util import Ssh

from plugins.CSBS.common.model import Account
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.util import check_az_param
from plugins.CSBS.common.util import check_ip


class EBackupUtil(object):
    sshclient_list = dict()

    def __init__(self, project_id, pod_id, regionid_list=None):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.param_util = ParamUtil()
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.ebackup = "eBackup"

    def _get_ebackup_az_when_expand_ebackup(self):
        az = self.param_util.get_value_from_cloudparam(self.pod_id, "CSBS-VBS", "eBackup_driver_az")
        if not az:
            raise Exception("Can't get ebackup az because az is None.")
        check_az_param(az)
        return az

    def _get_ebackup_az_when_expand_az(self):
        az = self.param_util.get_param_value(self.pod_id, "OpenStack", "openstack_expansion_az", '')
        if not az:
            raise Exception("Can't get ebackup az because az is None.")
        check_az_param(az)
        return az

    def get_ebackup_az(self):
        if self.param_tool.is_expand_ebackup_server(self.project_id) or \
                self.param_tool.is_install_csbs(self.project_id):
            return self._get_ebackup_az_when_expand_ebackup()
        elif self.is_ex_az_reuse_ebackup():
            return self._get_ebackup_az_when_expand_az()
        else:
            raise Exception("Can't get ebackup az because no scenes match.")

    def get_ebackup_management_ip(self):
        if self.param_tool.is_expand_ebackup_server(self.project_id):
            datamover_ip_list = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.ebackup, 'datamover_externalom_iplist').split(',')
            return self._get_ebackup_management_ip(
                datamover_ip_list, self.pod_id, self.ebackup, 'datamover_management_float_ip')
        elif self.is_ex_az_reuse_ebackup():
            datamover_ip_list = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.ebackup, 'existed_datamover_externalom_iplist').split(',')
            return self._get_ebackup_management_ip(
                datamover_ip_list, self.pod_id, self.ebackup, 'existed_datamover_mg_float_ip')
        else:
            raise Exception("Can't get ebackup management ip because no scenes match.")

    def get_ebackup_userpwd(self):
        user = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS', 'eBackup_azconfig_user')
        if not user:
            raise Exception("Ebackup user is None.")
        pwd = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS', 'eBackup_azconfig_pwd')
        if not pwd:
            raise Exception("Ebackup pwd is None.")
        return Account(user, pwd)

    def is_ex_az_reuse_ebackup(self):
        return self.param_tool.is_expand_az(self.project_id) and \
               self.param_tool.is_installed_csbs(self.project_id)

    def _get_ebackup_management_ip(self, datamover_ip_list, pod_id, srv_name, key):
        if not datamover_ip_list:
            raise Exception("Datamover ip list is None.")

        management_float_ip = datamover_ip_list[0] if len(datamover_ip_list) == 1 else \
            self.param_util.get_value_from_cloudparam(pod_id, srv_name, key)

        if not check_ip(management_float_ip):
            raise Exception("Management float ip is illegal.")
        return management_float_ip

    def get_workflow_ip_list(self):
        params_dict = self.param_util.get_service_cloud_param(self.project_id, "CSBS-VBS", self.regionid_list[0])

        workflow_ip_list = params_dict["eBackup_workflow_node"].replace(",", ";").lower()
        workflow_ip_list = workflow_ip_list.replace(" ", "").replace("|", ";").split(";")

        manager_ip_list = params_dict["eBackup_manager_node"].replace(",", ";").lower()
        manager_ip_list = manager_ip_list.replace(" ", "").replace("|", ";").split(";")

        workflow_and_manager_ip_list = workflow_ip_list + manager_ip_list
        return workflow_and_manager_ip_list

    @classmethod
    def close_ssh_clinet(cls, ssh_client):
        Ssh().ssh_close(ssh_client)
        for key, client in cls.sshclient_list.items():
            if client == ssh_client:
                cls.sshclient_list.pop(key)
                logger.info("remove ssh client from sshclient_list")
                break

    def find_float_ip(self, host_list, hcp_password, root_password):
        try:
            return self._find_float_ip(host_list, hcp_password, root_password)
        except Exception as e:
            logger.info("Find float ip failed,the reason is %s" % str(e))
            return ""

    def _find_float_ip(self, host_list, hcp_password, root_password):
        ssh = Ssh()
        find_float_ip_cmd = 'cat /opt/huawei-data-protection/ebackup/conf/floatIpBak || echo "isSignleMode"'
        float_ip = ""
        result = None
        for ip in host_list:
            ssh_client = ssh.ssh_create_client(ip, "hcp", hcp_password)
            ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
            ssh.ssh_send_command(ssh_client, root_password, '#', 100)
            result = ssh.ssh_exec_command_return_list(ssh_client,
                                                      find_float_ip_cmd)
            self.close_ssh_clinet(ssh_client)
            logger.info("The result is %s." % str(result))
            if -1 != str(result).find("isSignleMode"):
                logger.info("It is in SignleMode or this node is a proxy node.")
                continue
            float_ip = result[0].strip().replace("\n", "")
            if float_ip != "":
                logger.info("The float ip of host(%s) is %s." % (str(host_list), float_ip))
                return float_ip

        if -1 != str(result).find("isSignleMode"):
            logger.info("It is in SignleMode,use admin node ip as float ip.")
            float_ip = self.find_admin_ip(host_list, hcp_password, root_password)
            if float_ip != "":
                logger.info("It is in SignleMode,The admin node ip is %s." % float_ip)
                return float_ip
        logger.error("Find float ip of host(%s) failed." % str(host_list))
        return float_ip

    def find_admin_ip(self, host_list, hcp_password, root_password):
        try:
            return self._find_admin_ip(host_list, hcp_password, root_password)
        except Exception as e:
            logger.info("Find admin node ip failed,the reason is %s" % str(e))
            return ""

    def _find_admin_ip(self, host_list, hcp_password, root_password):
        ssh = Ssh()
        is_admin_node = 'ps -ef | grep -v "grep" | grep "AdminNode" > /dev/null 2>&1 && echo "isAdminNode"'
        ret = ""
        for ip in host_list:
            ssh_client = ssh.ssh_create_client(ip, "hcp", hcp_password)
            ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 100)
            ssh.ssh_send_command(ssh_client, root_password, '#', 100)
            result = str(ssh.ssh_exec_command_return(ssh_client, is_admin_node))
            self.close_ssh_clinet(ssh_client)
            if -1 != result.find("isAdminNode"):
                ret = str(ip)
                logger.info("The admin node of hosts(%s) is %s" % (str(host_list), ip))
                return ret
        logger.info("Find admin node of host(%s) failed." % str(host_list))
        return ret
