# coding=utf-8
import json
import os
from time import sleep

import requests

from utils.common import log as common_logger


class HttpClient(object):
    """This is an HTTP client that determines whether SSL authentication

    is supported by instantiating parameters.
    """

    def __init__(self, insecure=False, timeout=60, ca_file=None,
                 cert_file=None, key_file=None, logger=None,
                 retries=3, retry_login=True, login_fun=None):
        self.insecure = insecure
        self.timeout = timeout
        self.ca_file = ca_file
        self.cert_file = cert_file
        self.key_file = key_file
        self._use_ssl = self._is_ssl_enabled()
        self._verify = self._get_verify()
        self.logger = logger or common_logger
        self._retries = retries
        self._retry_login = retry_login
        self._login_fun = login_fun
        self._headers = dict()

    @property
    def retries(self):
        return self._retries

    @retries.setter
    def retries(self, retries):
        self._retries = retries

    @property
    def retry_login(self):
        return self._retry_login

    @retry_login.setter
    def retry_login(self, retry_login):
        self._retry_login = retry_login

    @property
    def login_fun(self):
        return self._login_fun

    @login_fun.setter
    def login_fun(self, login_fun):
        self._login_fun = login_fun

    @property
    def headers(self):
        return self._headers

    @headers.setter
    def headers(self, headers):
        self._headers = headers

    def headers_update(self, _headers):
        self.headers.update(_headers)

    def get(self, url, **kwargs):
        return self._send(url, 'GET', **kwargs)

    def post(self, url, **kwargs):
        return self._send(url, 'POST', **kwargs)

    def put(self, url, **kwargs):
        return self._send(url, 'PUT', **kwargs)

    def delete(self, url, **kwargs):
        return self._send(url, 'DELETE', **kwargs)

    def request(self, url, method, **kwargs):
        """Send https request main function"""
        kwargs.setdefault('headers', kwargs.get('headers', self.headers))
        kwargs['headers'].setdefault('Accept', 'application/json')

        if 'body' in kwargs and kwargs['body']:
            kwargs['headers']['Content-Type'] = 'application/json'
            body = kwargs.pop('body')
            if isinstance(body, dict):
                body = json.dumps(body)
            kwargs['data'] = body
        elif 'body' in kwargs:
            kwargs.pop('body')

        kwargs.setdefault('timeout', self.timeout)

        kwargs["verify"] = self._verify

        if self._use_ssl:
            kwargs["cert"] = (self.cert_file, self.key_file)

        resp = requests.request(method, url, **kwargs)

        body = dict()
        if resp.text:
            try:
                body = json.loads(resp.text)
            except Exception as err:
                body = resp.text
                self.logger.warn(
                    "Load http response text error: {}.".format(err))

        return resp, body

    def _get_verify(self):
        """Determine whether ca certificate verification is

        enabled and check if the file exists.
        """
        if self.insecure or not self.ca_file:
            verify = False
        else:
            if not os.path.exists(self.ca_file):
                raise Exception(
                    "Unable to find ca_file: {}.".format(self.ca_file))
            verify = self.ca_file
        return verify

    def _is_ssl_enabled(self):
        """Check if the provided certificate files exists"""
        use_ssl = self.cert_file or self.key_file

        if self.cert_file and not os.path.exists(self.cert_file):
            raise Exception(
                "Unable to find cert_file: {}.".format(self.cert_file))

        if self.key_file and not os.path.exists(self.key_file):
            raise Exception(
                "Unable to find key_file: {}.".format(self.key_file))

        if use_ssl and (not self.cert_file or not self.key_file):
            raise Exception(
                "When client verify for https req, "
                "you must specify both a cert_file and key_file option value.")

        return use_ssl

    def _send(self, url, method, **kwargs):
        """HTTP request method with retry, the retry interval is an

        exponential backoff algorithm.
        """
        attempts = 1
        backoff = 1
        retry_login = self.retry_login
        while True:
            attempts += 1
            try:
                rsp, body = self.request(url, method, **kwargs)
            except Exception as err:
                rsp = None
                body = ""
                self.logger.trace()
                self.logger.error(err)

            if rsp and rsp.status_code < 400:
                return rsp, body

            if attempts <= self.retries:
                # The failure may be caused by the expiration of the
                # console session, so need to retry login again.
                if retry_login is True and callable(self.login_fun):
                    self.login_fun()
                    retry_login = False
                sleep(backoff)
                backoff *= 2
                attempts += 1
            else:
                raise Exception(body)
