import json
import time

import utils.common.log as logger
from utils.business.iam_util import IamApi
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean

from plugins.CSBS.common import constant
from plugins.CSBS.common.model import Account
from plugins.CSBS.common.params_tool import ParamTool


class IamUtil(object):
    class IAMAccount(Account):
        def __init__(self, account_name, account_pwd, domain_name, is_default=True):
            super().__init__(account_name, account_pwd)
            self.domain_name = domain_name
            self.is_default = is_default

    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.param_util = ParamUtil()
        self.service_name = constant.CSBS_SERVICE_NAME
        self.param_tool = ParamTool(self.project_id, self.pod_id)

    def create_op_service_account(self):
        account = self._get_op_service_default_account()
        if self.can_authed(account):
            return True
        account_msg = {
            "user": {
                "name": account.account_name,
                "mobile": "",
                "email": "",
                "password": account.account_pwd,
                "domain_name": "op_service",
                "bind_group": "services"
            }}
        logger.info(f"Start to create karbor iam account, "
                    f"account name:{account.account_name}.")
        result = IamApi.create_account(account_msg, self.pod_id)
        if not result:
            err_msg = f"Failed to create {account.account_name} iam account."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Succeed to create {account.account_name} iam account.")

    def create_op_svc_account(self):
        account = self._get_op_svc_default_account()
        if self.can_authed(account):
            return True
        logger.info(f"Start to create IAM {account.account_name} account.")
        account_list = [
            {
                "domain_name": account.account_name,
                "password": account.account_pwd,
                "acl": {}
            }
        ]
        result = IamApi.create_op_svc_account(account_list, self.pod_id)
        if not result:
            err_msg = f"Failed to create {account.account_name} iam account."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info(f"Succeed to create {account.account_name} iam account.")

    def create_role(self):
        logger.info('Start to create iam roles.')
        role_msgs = [
            {
                "name": "csbs_adm",
                "display_name": "CSBS Administrator",
                "description": "Cloud Server Backup Service Administrator",
                "catalog": "CSBS",
                "policy":
                    {"Version": "1.0",
                     "Statement": [{"Effect": "Allow", "Action": ["CSBS:backup:*"]}],
                     "Depends": [
                         {"catalog": "BASE", "display_name": "Server Administrator"},
                         {"catalog": "BASE", "display_name": "Tenant Guest"}
                     ]}
            },
            {
                "name": "vbs_adm",
                "display_name": "VBS Administrator",
                "description": "VBS Administrator",
                "catalog": "VBS",
                "policy":
                    {"Version": "1.0",
                     "Statement": [{"Effect": "Allow", "Action": ["VBS:backup:*"]}],
                     "Depends": [
                         {"catalog": "BASE", "display_name": "Server Administrator"},
                         {"catalog": "BASE", "display_name": "Tenant Guest"}
                     ]}
            }]
        for role_msg in role_msgs:
            logger.info(f"Create role info:{role_msg}.")
            result = IamApi.create_role(role_msg, self.pod_id)
            if not result:
                err_msg = f"Failed to create role, role name:{role_msg['name']}."
                logger.error(err_msg)
                raise Exception(err_msg)
        logger.info("Succeed to create iam roles.")

    def _get_op_service_default_account(self):
        account_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, 'iam_karbor_admin')
        if not account_name:
            raise Exception("Failed to obtain the op_service account name.")
        account_pwd = self.param_util.get_param_value(
            self.pod_id, self.service_name, 'iam_karbor_admin_pwd')
        if not account_pwd:
            raise Exception("Failed to obtain the op_service account pwd.")
        return self.IAMAccount(account_name, account_pwd, domain_name="op_service")

    def _get_op_svc_default_account(self):
        account_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, 'iam_op_svc_csbs_account')
        if not account_name:
            raise Exception("Failed to obtain the op_svc account name.")
        account_pwd = self.param_util.get_param_value(
            self.pod_id, self.service_name, 'iam_op_svc_csbs_pwd')
        if not account_pwd:
            raise Exception("Failed to obtain the op_svc account pwd.")
        return self.IAMAccount(account_name, account_pwd, domain_name="op_svc_csbs")

    def get_iam_account_info(self, account_type="op_service"):
        if account_type not in ("op_service", "op_svc_csbs"):
            raise Exception("Unsupported account_type, the input "
                            f"account_type parameter is {account_type}.")
        if account_type == "op_service":
            account = self._get_op_service_default_account()
        else:
            account = self._get_op_svc_default_account()

        # 首region且部署场景， 直接返回默认账号密码
        if get_project_condition_boolean(self.project_id, "PrimaryRegion & ProjectDeploy"):
            logger.info("The current scene is the primary region deployment scene, "
                        "use the default account information.")
            return account

        if self.can_authed(account):
            logger.info("The default account is used for authentication.")
            return account

        try:
            region_id_list = self.param_tool.get_all_region_code()
        except Exception as e:
            logger.warning(f"Failed to obtain the region id list from CMDB, err_msg:{str(e)}.")
            return account
        logger.info(f"Succeeded to obtain all region codes, region codes are: {region_id_list}.")

        cur_account = self._get_modified_pwd_account(region_id_list, account)
        if not cur_account:
            raise Exception(f"Failed to obtain the {account.account_name} pwd from the unify_pwd "
                            f"and the {account.account_name} pwd is not the default password.")
        return cur_account

    def _get_modified_pwd_account(self, region_id_list, tar_account):
        iam_pwd_list = []

        for region_id in region_id_list:
            try:
                account_info = self.param_tool.get_account_pwd_from_unify_pwd(
                    region_id, [(tar_account.account_name, 5)])
            except Exception as e:
                logger.warning(f"Failed to obtain the account information of {region_id} "
                               f"from the unify_pwd, err_msg:{str(e)}.")
                continue
            for item in account_info:
                if item.get('accountName') != tar_account.account_name or \
                        item.get('newPasswd') == tar_account.account_pwd:
                    continue
                iam_pwd_list.append(item.get('newPasswd'))
        authed_count = 1
        for pwd in set(iam_pwd_list):
            tar_account.account_pwd = pwd
            if self.can_authed(tar_account):
                tar_account.is_default = False
                return tar_account
            # 15分钟内鉴权失败5次，账号会被锁15分钟
            authed_count += 1
            if authed_count >= 5:
                time.sleep(15 * 60+10)
                authed_count = 0

    def get_iam_token(self, account):
        """

        :param account: IAMAccount() object instance
        :return: iam token, type:str
        """
        if not hasattr(account, "domain_name") or account.domain_name not in ("op_service", "op_svc_csbs"):
            raise Exception("The domain_name of the account object is not expected, "
                            f"the domain_name is {getattr(account, 'domain_name', None)}.")
        account_info = {"domain_name": account.domain_name,
                        "name": account.account_name}

        # 修改get_iam_token获取iam的token，兼容HCSD与HCSU
        body = {
            "auth": {
                "identity": {
                    "methods": ["password"],
                    "password": {
                        "user": {"name": account_info.get("name"),
                                 "password": account.account_pwd,
                                 "domain": {"name": account_info.get("domain_name")}}
                    }
                },
                "scope": {"domain": {"name": account_info.get("domain_name")}}
            }
        }
        return IamApi.get_iam_token(self.pod_id, json.dumps(body))

    def can_authed(self, account):
        """

        :param account: IAMAccount() object instance
        :return: bool
        """
        try:
            if self.get_iam_token(account):
                logger.info("Success to use the current account for authentication, "
                            f"account name:{account.account_name}")
                return True
        except Exception as e:
            logger.warning(f"Failed to use the current account for authentication, "
                           f"account name:{account.account_name}, err_msg:{str(e)}")
        return False
