import utils.common.log as logger

from plugins.CSBS.common import node_installer
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.scripts.deploy.karbor.cbs_installer.karbor_cpshelper import CPSInfo


class KarborCerts(node_installer.Installer):
    cps_nova_cert_old_path = "/etc/FSSecurity/server-cert/nova_ca.crt"
    cps_nova_cert_new_path = "/home/fsp/nova_ca.crt"
    djmanager_path = "/home/djmanager/"
    openstack_path = "/home/openstack/"
    nova_cert = "nova_ca.crt"
    iam_cert = "trust_apimlb_iam.cer"
    sc_cert = "trust_apimlb_sc.cer"
    oc_cert = "trust_apimlb_oc.cer"

    def __init__(self, project_id, pod_id):
        super(KarborCerts, self).__init__()
        self.param_tool = ParamTool(project_id, pod_id)
        self.karbor_util = KarborUtil(project_id, pod_id)
        self.pod_id = pod_id
        self.ssh_client = SshClient()

    def _make_cert_file(self):
        logger.info('Start copying nova_ca.crt and change owners.')
        cps_node = CPSInfo(self.pod_id).get_cps_node()
        logger.info(f'Ssh to scp node:{cps_node.ip}.')
        cps_client = self.ssh_client.get_ssh_client(cps_node, sudo_type="su")
        cmd = f"cp {self.cps_nova_cert_old_path} {self.cps_nova_cert_new_path}"
        logger.info(f'Cmd:{cmd}.')
        self.ssh_client.ssh_exec_command_return(cps_client, cmd)
        cmd = f"chown fsp:fsp {self.cps_nova_cert_new_path}"
        logger.info(f'Cmd:{cmd}.')
        result = self.ssh_client.ssh_exec_command_return(cps_client, cmd)
        self.ssh_client.ssh_close(cps_client)
        return self.ssh_client.is_ssh_cmd_executed(result)

    def _get_nova_cert_from_cps_node(self, karbor_node):
        if not self._make_cert_file():
            logger.error('Failed copying nova cert and change owners.')
            return False
        logger.info(f'Ssh to karbor node:{karbor_node.ip}.')
        karbor_client = self.ssh_client.get_ssh_client(karbor_node)
        work_dir = self.karbor_util.get_installer_workdir()
        cd_cmd = f"cd {work_dir}/CSBS/install_scripts"
        logger.info(f'Cmd:{cd_cmd}.')
        self.ssh_client.ssh_exec_command_return(karbor_client, cd_cmd)
        cps_node = CPSInfo(self.pod_id).get_cps_node()
        cps_nova_cert = f"{cps_node.user}@[{cps_node.ip}]:" \
                        f"{self.cps_nova_cert_new_path}"
        logger.info('Uploading cert to karbor node from '
                    'cps, Cmd:{}'.format(cps_nova_cert))
        ret = self.ssh_client.scp_file(karbor_client, cps_nova_cert, f"{self.djmanager_path}{self.nova_cert}",
                                       cps_node.user_pwd)
        if not ret:
            logger.error('Failed calling ssh_client.scp_file() '
                         'to upload cert.')
            self.ssh_client.ssh_close(karbor_client)
            return False
        self.ssh_client.ssh_close(karbor_client)
        return True

    def _config_cert(self, karbor_node):
        auth_type = self.param_tool.get_auth_type()
        logger.info('Current auth_type:{}'.format(auth_type))
        if auth_type == "iam":
            auth_cmd = f"source /opt/huawei/dj/inst/utils.sh;" \
                       f"set_auth_endpoint " \
                       f"--auth_ca_cert {self.openstack_path}{self.iam_cert}"
        elif auth_type == "keystone":
            auth_cmd = f"source /opt/huawei/dj/inst/utils.sh;" \
                       f"set_auth_endpoint " \
                       f"--auth_ca_cert " \
                       f"{self.openstack_path}{self.nova_cert} " \
                       f"--auth_type keystone"
        else:
            raise Exception(f"An unexpected auth_type is obtained, "
                            f"auth_type:{auth_type}.")

        cmds = [
            "source /opt/huawei/dj/inst/utils.sh",
            f"mv -f {self.djmanager_path}*.crt {self.djmanager_path}*.cer "
            f"{self.openstack_path}",
            f"chown openstack:openstack {self.openstack_path}*.crt "
            f"{self.openstack_path}*.cer",
            auth_cmd,
            f"set_auth_endpoint "
            f"--sc_ca_cert {self.openstack_path}{self.sc_cert}",
            f"set_auth_endpoint "
            f"--oc_ca_cert {self.openstack_path}{self.oc_cert}",
            f"sync_karbor_certs "
            f"--openstack {self.openstack_path}{self.nova_cert}"
        ]
        karbor_client = self.ssh_client.get_ssh_client(karbor_node)
        for cmd in cmds:
            logger.info(f'Cmd:{cmd}.')
            result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                self.ssh_client.ssh_close(karbor_client)
                logger.error(f'Failed to execute: {cmd}, '
                             f'returned message:{result[0]}.')
                return False
        if karbor_client:
            self.ssh_client.ssh_close(karbor_client)
        return True

    def install(self):
        karbor_node = self.karbor_util.get_karbor_node_list()[0]
        if not self._get_nova_cert_from_cps_node(karbor_node):
            return False

        if not self._config_cert(karbor_node):
            return False
        logger.info("Succeed to config certificate for the Karbor node.")
        return True

    def check(self):
        return True

    def rollback(self):
        logger.info('Executing rollback cert.')
        karbor_node_list = self.karbor_util.get_karbor_node_list()
        logger.info(f'Ssh to karbor node:{karbor_node_list[0].ip}.')
        karbor_client = self.ssh_client.get_ssh_client(karbor_node_list[0])
        cmd = f"rm -f {self.djmanager_path}*.crt {self.djmanager_path}*.cer"
        logger.info(f'Cmd:{cmd}.')
        self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
        self.ssh_client.ssh_close(karbor_client)
        return True
