import os
import time

import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.business.project_util import ProjectApi
from utils.business.vm_util import PublicServiceOM
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.util import unzip_file
from plugins.CSBS.scripts.deploy.karbor.api_adapter import API

logger.init("CSBS")


class CreateVM(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(CreateVM, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.om = PublicServiceOM()
        self.params = ParamUtil()
        self.param_tool = ParamTool(project_id, pod_id)
        self.network_name = "external_om"
        project_conditions = get_project_conditions(self.project_id)
        if project_conditions.get('UpgradedToHCS80') == 1:
            self.network_name = "Public_Service"

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            csha = self.__get_project_info(project_id)
        except HCCIException as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, e)
        except Exception as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, HCCIException(640001, str(e)))
        try:
            self.__create_vm(pod_id, csha)
            return Message(200)
        except HCCIException as e:
            logger.error("execute error:%s" % str(e))
            error = str(e)
        except Exception as e:
            logger.error("execute error:%s" % str(e))
            error = str(e)
        return Message(500, HCCIException(640001, error))

    def rollback(self, project_id, pod_id, regionid_list=None):
        try:
            csha = self.__get_project_info(project_id)
        except HCCIException as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, e)
        except Exception as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, HCCIException(640001, str(e)))
        try:
            self.__delete_vm(pod_id, csha)
        except HCCIException as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, e)
        except Exception as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, HCCIException(640001, str(e)))
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        try:
            csha = self.__get_project_info(project_id)
        except HCCIException as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, e)
        except Exception as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, HCCIException(640001, str(e)))
        try:
            self.__delete_vm(pod_id, csha)
        except HCCIException as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, e)
        except Exception as e:
            logger.error("execute error:%s" % str(e))
            return Message(500, HCCIException(640001, str(e)))
        return self.execute(project_id, pod_id)

    def __get_project_info(self, project_id):
        project_api = ProjectApi()
        replication_dict = project_api.get_project_ha_dr_status(project_id)
        csha = replication_dict['CSHA']
        if replication_dict['RegionConHA'] == 1:
            csha = 1
        if not isinstance(csha, int):
            raise HCCIException(640067)
        return csha

    def __delete_vm(self, pod_id, csha):
        csbs_service_ip01 = \
            self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS",
                                                  "CSBS_Service1_ip")
        csbs_service_ip02 = \
            self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS",
                                                  "CSBS_Service2_ip")
        if csha == 0:
            csbs_service_ip03 = \
                self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS",
                                                      "CSBS_Service3_ip")
        csbs_float_ip01 = \
            self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS",
                                                  "CSBS_Float_IP1")
        csbs_float_ip02 = \
            self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS",
                                                  "CSBS_Float_IP2")

        if csha == 1:
            self.om.delete_servers_with_group(pod_id, "Service", "CSBS",
                                              [csbs_service_ip01],
                                              self.network_name)
            self.om.delete_servers_with_group(pod_id, "Service", "CSBSdr",
                                              [csbs_service_ip02],
                                              self.network_name)
        else:
            self.om.delete_servers_with_group(pod_id, "Service", "CSBS",
                                              [csbs_service_ip01],
                                              self.network_name)
            self.om.delete_servers_with_group(pod_id, "Service", "CSBS2",
                                              [csbs_service_ip02],
                                              self.network_name)
            self.om.delete_servers_with_group(pod_id, "Service", "CSBS3",
                                              [csbs_service_ip03],
                                              self.network_name)
        # 创建浮动ip对应的port，把ip占用删除
        self.om.delete_port_by_name(pod_id, "CSBS_float_ip01", csbs_float_ip01,
                                    self.network_name)
        self.om.delete_port_by_name(pod_id, "CSBS_float_ip02", csbs_float_ip02,
                                    self.network_name)

    def __create_vm(self, pod_id, csha=0, scale=10000):
        file_name = self._get_file_name()
        cpu = self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS", "CSBS_cpu")
        mem = self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS", "CSBS_mem")
        csbs_service_ip, csbs_float_ip = self._get_csbs_ip(pod_id, csha)
        if csha == 1:
            self.om.create_servers_with_group(pod_id, "Service", "CSBS", cpu, mem, 0, [csbs_service_ip[0]],
                                              self.network_name, vm_image_name="CSBS-VBS-image", sys_disk=150,
                                              image_file_name=file_name, availability_zone="manage-az",
                                              image_template_type=0, volume_type="manager.fsa.csg",
                                              ip_list_ex=None, network_name_ex=None)
            self.om.create_servers_with_group(pod_id, "Service", "CSBSdr", cpu, mem, 0, [csbs_service_ip[1]],
                                              self.network_name, vm_image_name="CSBS-VBS-image", sys_disk=150,
                                              image_file_name=file_name, availability_zone="dr-manage-az",
                                              image_template_type=0, volume_type="dr.manager.fsa.csg", ip_list_ex=None,
                                              network_name_ex=None)
        else:
            self.om.create_servers_with_group(pod_id, "Service", "CSBS", cpu, mem, 0, [csbs_service_ip[0]],
                                              self.network_name, vm_image_name="CSBS-VBS-image", sys_disk=150,
                                              image_file_name=file_name)

            self.om.create_servers_with_group(pod_id, "Service", "CSBS2", cpu, mem, 0, [csbs_service_ip[1]],
                                              self.network_name, vm_image_name="CSBS-VBS-image", sys_disk=150,
                                              image_file_name=file_name)

            self.om.create_servers_with_group(pod_id, "Service", "CSBS3", cpu, mem, 0, [csbs_service_ip[2]],
                                              self.network_name, vm_image_name="CSBS-VBS-image", sys_disk=150,
                                              image_file_name=file_name)
        # 创建浮动ip对应的port，把ip占用
        self.om.create_port(pod_id, "CSBS_float_ip01", csbs_float_ip[0], self.network_name)
        self.om.create_port(pod_id, "CSBS_float_ip02", csbs_float_ip[1], self.network_name)

    def _get_csbs_ip(self, pod_id, csha):
        csbs_service_ip = []
        csbs_service_ip01 = self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS", "CSBS_Service1_ip")
        if not csbs_service_ip01:
            raise HCCIException("640003", "CSBS_Service1")
        csbs_service_ip.append(csbs_service_ip01)
        csbs_service_ip02 = self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS", "CSBS_Service2_ip")
        if not csbs_service_ip02:
            raise HCCIException("640003", "CSBS_Service2")
        csbs_service_ip.append(csbs_service_ip02)
        if csha == 0:
            csbs_service_ip03 = \
                self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS",
                                                      "CSBS_Service3_ip")
            if not csbs_service_ip03:
                raise HCCIException("640003", "CSBS_Service3")
            csbs_service_ip.append(csbs_service_ip03)

        csbs_float_ip01 = self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS", "CSBS_Float_IP1")
        if not csbs_float_ip01:
            raise HCCIException("640003", "CSBS_Float_IP1")

        csbs_float_ip02 = self.params.get_value_from_cloudparam(pod_id, "CSBS-VBS", "CSBS_Float_IP2")
        if not csbs_float_ip02:
            raise HCCIException("640003", "CSBS_Float_IP2")
        csbs_float_ip = [csbs_float_ip01, csbs_float_ip02]
        return csbs_service_ip, csbs_float_ip

    def _get_file_name(self):
        is_cpu_arm = get_project_condition_boolean(self.project_id, 'manageARM')
        if is_cpu_arm:
            os_template = self.get_os_file("OceanStor_BCManager_Euler_ARM", "KVMtemplate_Euler_ARM.zip")
        else:
            os_template = self.get_os_file("OceanStor_BCManager_Euler_X86", "KVMtemplate_Euler_X86.zip")

        file_name = os_template.replace(".qcow2", "")
        return file_name

    def get_os_file(self, os_file_name, os_zip_file_name):
        default_hcci_pkg_path = self.param_tool.get_default_hcci_pkg_path()
        pkg_path, pkg_name = API.find_file(os_file_name, ".qcow2")

        if (pkg_path != default_hcci_pkg_path and pkg_name) or (not pkg_name):
            zip_file_path, zip_file_name = API.find_file("OceanStor BCManager", os_zip_file_name)
            if not zip_file_name or not zip_file_path:
                raise HCCIException(640002, f"OceanStor BCManager***{os_zip_file_name}")
            file_path = os.path.join(zip_file_path, zip_file_name)
            logger.info(f'Current os image zip:{file_path}.')

            unzip_file(file_path, default_hcci_pkg_path, unzip_size_limit_mb=5120, unzip_file_count_limit_kilo=10)
            _, pkg_name = API.find_file(os_file_name, ".qcow2")
            if not pkg_name or not os.path.exists(os.path.join(default_hcci_pkg_path, pkg_name)):
                raise HCCIException(640002, f"{os_file_name}.qcow2")
            logger.info(f'Current os image qcow2:{pkg_name}.')

        # 有可能文件已经生成，但是还未完全解压，做个延时
        time.sleep(10)

        return pkg_name
