import os
import shutil
import threading

import utils.common.log as logger
from utils.business.manageone_util import ManageOneUtil
from utils.business.manageone_util import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.CSBS.common.params_tool import ParamTool, Params
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.util import unzip_file
from plugins.CSBS.scripts.deploy.karbor.api_adapter import API

logger.init("CSBS")


class DeployCBSConsole(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DeployCBSConsole, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.mo_util = ManageOneUtil()
        self.params = ParamUtil()
        self.region_id = \
            self.params.get_param_value(self.pod_id, "public",
                                        "region_id", "region0_id")
        self.param_tool = ParamTool(project_id, pod_id)
        self.db_param_dict = Params(project_id, pod_id).get_params()
        self.fcd_pkg_temp_dir = "/var/tmp/cbs_console"
        self.r_lock = threading.RLock()

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self._deploy_concle()
        except HCCIException as e:
            logger.error("Execute error:{}.".format(str(e)))
            return Message(500, e)
        except Exception as e:
            logger.error("Execute error:{}.".format(str(e)))
            return Message(500, HCCIException(640082))
        finally:
            if os.path.exists(self.fcd_pkg_temp_dir):
                shutil.rmtree(self.fcd_pkg_temp_dir)

    def _deploy_concle(self):
        pkg_path, file_name = API.find_file("OceanStor BCManager", "_CBS_Console.zip")
        if not file_name:
            raise HCCIException("640010")
        pkg = os.path.join(pkg_path, file_name)
        console_version = self.param_tool.get_pkg_version(pkg)
        if not console_version:
            raise Exception("Failed getting version from console's package.")
        if os.path.exists(self.fcd_pkg_temp_dir):
            shutil.rmtree(self.fcd_pkg_temp_dir)
        unzip_file(pkg, self.fcd_pkg_temp_dir, unzip_size_limit_mb=100, unzip_file_count_limit_kilo=10)
        pkg_name = "cbs-{version}.tar.gz".format(version=console_version)
        package_dir = "{}/{}".format(self.fcd_pkg_temp_dir, pkg_name)
        self._upload_console_pkg_to_tenant_node(package_dir)

        # 安装CBS Console
        logger.info('Begin to deploy CSBS-VBS console, package:{}.'.format(package_dir))
        # Parameter console_version is used to specify the prefix
        # of the static file on the frontend and distinguish
        # different region file.
        self.mo_util.deploy_console(self.pod_id, "", "", "", "", cloud_service_type="cbs", region_id=self.region_id,
                                    console_version=self.region_id, file_name=pkg_name)
        logger.info("Deploy CSBS-VBS console successfully.")
        return Message(200)

    def _upload_console_pkg_to_tenant_node(self, package_dir):
        tenant_ip_list = \
            self.mo_util.get_console_static_server_ips(self.pod_id,
                                                       self.region_id)
        logger.info("The obtained IP addresses of the tenant "
                    "node are:{}.".format(tenant_ip_list))
        sopuser_pwd = ManageOneUtil2().getSopuserPassWdInfo(self.pod_id)
        for ip in tenant_ip_list:
            logger.info("Start to upload [{}] pkg "
                        "to {} node.".format(package_dir, ip))
            with self.r_lock:
                Ssh.put_file(host=ip, user="sopuser", passwd=sopuser_pwd,
                             source=package_dir, destination="/tmp")
            logger.info("Succeed to upload pkd to {} node.".format(ip))
