import utils.common.log as logger
from utils.common.message import Message

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.ssh_client import SshClient

logger.init("CSBS")


class DockingKMS(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DockingKMS, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self._docking_kms()
        except Exception as e:
            logger.error(f"Execute error:{e}")
            return Message(500, f"Failed to dock KMS, err_msg:{e}.")

        logger.info("Succeed to dock KMS.")
        return Message(200)

    def _docking_kms(self):
        cipher_type = ParamTool(self.project_id, self.pod_id).get_cipher_type()
        if not cipher_type:
            raise Exception("Failed to obtain the cipher type.")
        self._switch_cipher_type(cipher_type)
        self._switch_os_cipher(cipher_type)

    def _switch_cipher_type(self, cipher_type):
        result = self.karbor_util.change_cipher_type(
            cipher_type=cipher_type)
        if not result:
            raise Exception("Failed to change karbor cipher type.")
        logger.info("The cipher algorithm switched successfully.")

    def _switch_os_cipher(self, cipher_type):
        karbor_client = SshClient()
        if cipher_type not in ("generalCipher", "SMCompatible", "SMOnly"):
            raise Exception("Unsupported cipher type.")

        logger.info("Ssh to karbor node and switch OS cipher.")
        for node in self.karbor_util.get_karbor_node_list():
            ssh_client = karbor_client.get_ssh_client(node)
            cmd = "bash /opt/huawei/root_tools/base/changeOsCipher.sh " \
                  f"{cipher_type}"
            result = karbor_client.ssh_exec_command_return(ssh_client, cmd)
            if not karbor_client.is_ssh_cmd_executed(result):
                logger.error("Failed to switch OS cipher type on "
                             f"karbor node, node ip:{node.ip}.")
                karbor_client.ssh_close(ssh_client)
                raise Exception("Failed to witch OS cipher type on "
                                f"karbor node, node ip:{node.ip}.")
            logger.info("Succeed to switch OS cipher type on "
                        f"karbor node, node ip:{node.ip}.")
            karbor_client.ssh_close(ssh_client)
        logger.info("Succeed to switch all karbor node OS cipher.")
        return True
