import utils.common.log as logger
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.service_monitor.service_monitor import AuthProvider
from utils.business.service_monitor.service_monitor import Host
from utils.business.service_monitor.service_monitor import deploy_agent
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.dmk_tool import DmkTool
from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool, Params
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.ssh_client import SshClient

logger.init("CSBS")


class ICAgentInstall(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(ICAgentInstall, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.param_tool = ParamTool(project_id, pod_id)
        self.karbor_util = KarborUtil(project_id, pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self._install_moicagent_with_dmk(project_id, pod_id)
        except HCCIException as e:
            logger.error(f"Execute error:{e}")
            return Message(500, e)
        except Exception as e:
            logger.error(f"Execute error:{e}")
            return Message(500, HCCIException("640050", str(e)))

    def _install_moicagent_with_dmk(self, project_id, pod_id):
        dmk_tool = DmkTool(project_id, pod_id)
        params = Params(project_id, pod_id)
        param_dict = params.get_params()
        dmk_info = AuthProvider(param_dict.get('dmk_floatIp'), "8443", dmk_tool.dmk_user, dmk_tool.user_pwd)
        logger.info('Get DMK info : [ip:{}, port:{}, username:{}, '
                    'password:******]'.format(dmk_info.dmk_ip, dmk_info.port, dmk_info.user_name))
        karbor_iplist = self.karbor_util.get_karbor_ip_list()
        logger.info('Karbor ip list: {}'.format(karbor_iplist))
        hosts = []
        for karbor in karbor_iplist:
            karbor_node = Host(karbor, 22, karbor, karbor)
            hosts.append(karbor_node)
        # arm场景不用适配python
        if not get_project_condition_boolean(project_id, 'manageARM'):
            if not self.icagent_adapt_python():
                return Message(500, HCCIException("640050"))

        logger.info('Start installing MOICAgent with DMK')
        result = deploy_agent(pod_id, dmk_info, hosts, self.karbor_util.get_karbor_user_info().user_name, "sudo")
        return result

    def icagent_adapt_python(self):
        karbor_client = SshClient()
        """适配切换python3后icagent的安装，修改适配python的链接指向"""
        karbor_node_list = self.karbor_util.get_karbor_node_list()
        logger.info(f"Adapt python on nodes:{karbor_node_list}.")
        ln_cmd = "ln -sf /usr/bin/python3 /usr/bin/python"
        for node in karbor_node_list:
            ssh_client = karbor_client.get_ssh_client(node)
            if not ssh_client:
                return False
            try:
                self._exec_cmd_to_karbor(node, ssh_client, ln_cmd, karbor_client)
            except Exception as e:
                logger.error('Execute error: {}'.format(e))
                return False
            finally:
                karbor_client.ssh_close(ssh_client)
        return True

    def _exec_cmd_to_karbor(self, node, ssh_client, ln_cmd, karbor_client):
        logger.info(f'Ssh to karbor node:{node.ip} execute cms:{ln_cmd}.')
        result = karbor_client.ssh_exec_command_return(ssh_client, ln_cmd)
        if not karbor_client.is_ssh_cmd_executed(result):
            karbor_client.ssh_close(ssh_client)
            raise Exception(f'Exec cmd to karbor fail ：{ln_cmd}')
