import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.message import RESULT_CODE

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.param_check.os_param_check import check_pwd_will_expired
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.ssh_client import SshClient

logger.init("CSBS")


class PreCheckNodeParams(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(PreCheckNodeParams, self).__init__(project_id, pod_id, regionid_list)
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)
        self.karbor_ip_list = self.karbor_util.get_karbor_ip_list()
        self.karbor_user_info = self.karbor_util.get_karbor_user_info()
        self.ssh_client = SshClient()

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self._check_karbor_login_ability()
        except Exception as e:
            logger.error(f"Execute error:{str(e)}")
            return Message(RESULT_CODE['ERROR'], f"Failed to check karbor params, err_msg:{str(e)}.")

    def _check_karbor_login_ability(self):
        result = self._check_karbor_os_user_login_ability()
        if result != RESULT_CODE['SUCCESS']:
            return Message(RESULT_CODE['ERROR'], HCCIException(result, self.karbor_user_info.user_name))

        result = self._check_karbor_root_login_ability()
        if result != RESULT_CODE['SUCCESS']:
            return Message(RESULT_CODE['ERROR'], HCCIException(result, self.karbor_user_info.root_name))

        logger.info("Succeed to check karbor params.")
        return Message(RESULT_CODE['SUCCESS'])

    def _check_karbor_os_user_login(self, ip, user, pwd):
        is_expired = False
        will_expire = False
        failed_login_with_other_cause = False
        karbor_client = None

        try:
            karbor_client = self.ssh_client.create_ssh_client(ip, user, pwd)
        except Exception as e:
            # 密码过期或异常
            if str(e).lower().find("authentication") != -1:
                logger.warning(f"The {user} account password has expired, please check.")
                is_expired = True
            else:
                logger.error("Failed to login to Karbor node: IP {ip}, User {user}, error: {str(e)}")
                failed_login_with_other_cause = True

        if not karbor_client and not is_expired:
            failed_login_with_other_cause = True

        if karbor_client and "Current password".lower() in karbor_client.get("prompt", "").lower():
            is_expired = True
            logger.warning(f"The {user} account password has expired, please check.")
        elif karbor_client and not is_expired:
            will_expire = check_pwd_will_expired(self.ssh_client, karbor_client, user)
            logger.warning(f"Node {ip}, User {user} password will expire, please change and retry.")

        if karbor_client:
            try:
                self.ssh_client.ssh_close(karbor_client)
            except Exception as e:
                logger.error(f"Failed to close ssh client: IP {ip}, "
                             f"error: {str(e)}.")

        return is_expired, will_expire, failed_login_with_other_cause

    def _check_karbor_os_user_login_ability(self):
        user = self.karbor_user_info.user_name
        pwd = self.karbor_user_info.user_pwd
        djmanager_pwd_expired = []
        djmanager_pwd_will_expired = []
        check_success_count = 0
        djmanager_pwd_no_ssh_ips = []

        for ip in self.karbor_ip_list:
            is_expired, will_expire, failed_login_with_other_cause = self._check_karbor_os_user_login(ip, user, pwd)
            if is_expired:
                djmanager_pwd_expired.append(ip)
            elif will_expire:
                djmanager_pwd_will_expired.append(ip)
            elif failed_login_with_other_cause:
                djmanager_pwd_no_ssh_ips.append(ip)
            else:
                check_success_count += 1

        if check_success_count == len(self.karbor_ip_list):
            logger.info("All users and passwords are OK.")
            return 200
        if djmanager_pwd_expired:
            logger.error(f"Failed to login to Karbor node: IP {', '.join(djmanager_pwd_expired)},"
                         f" User {user} passwd is expired.")
            return 640128
        elif djmanager_pwd_will_expired:
            logger.error(f"Node IP {', '.join(djmanager_pwd_will_expired)}, User {user} password will expire.")
            return 640129
        else:
            logger.error(f"Failed to login to Karbor node: IP {', '.join(djmanager_pwd_no_ssh_ips)}, User {user}")
            return 640130

    def _check_karbor_root_login(self, ip):
        def login_karbor_for_root(karbor_ip, user_name, user_pwd, root_pwd):
            karbor_client = self.ssh_client.create_ssh_client(karbor_ip, user_name, user_pwd)
            self.ssh_client.ssh_send_command_expect(karbor_client, 'sudo su root', 'password for root:', 10)
            self.ssh_client.ssh_send_command_expect(karbor_client, root_pwd, '#', 10)
            return karbor_client

        is_login_succeed = True
        ssh_client = None
        try:
            ssh_client = login_karbor_for_root(ip, self.karbor_user_info.user_name, self.karbor_user_info.user_pwd,
                                               self.karbor_user_info.root_pwd)
        except Exception as e:
            logger.error(f"Failed to login to Karbor node: IP {ip}, User root, error: {str(e)}.")
            is_login_succeed = False
        finally:
            if ssh_client:
                self.ssh_client.ssh_close(ssh_client)
        return is_login_succeed

    def _check_karbor_root_login_ability(self):
        karbor_root_no_ssh_ips = []
        for ip in self.karbor_ip_list:
            is_login_succeed = False
            try:
                is_login_succeed = self._check_karbor_root_login(ip)
            except Exception as e:
                logger.error(f"Failed to login to Karbor node: IP {ip}, User root, error: {str(e)}.")
            if not is_login_succeed:
                karbor_root_no_ssh_ips.append(ip)

        if len(karbor_root_no_ssh_ips) != 0:
            logger.error(f"Failed to login to Karbor node: IP {karbor_root_no_ssh_ips}, User root.")
            return 640130
        return 200
