import os

import utils.common.log as logger
from utils.business.apig_util import APIGApi
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.util import open_file

logger.init("RegisterAPIGateway")


def get_config_content(file_path):
    file_object = open_file(file_path)
    content = ""
    try:
        for line in file_object:
            content = "{0}{1}".format(content, line)
    except Exception:
        logger.error(f'get file {file_path} content failed')
    finally:
        file_object.close()
    return content


class RegisterApigateway(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(RegisterApigateway, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.service_name = 'CSBS-VBS'
        self.param_util = ParamUtil()
        self.params_dict = self.param_util.get_service_cloud_param(
            self.pod_id, self.service_name)
        self.param_tool = ParamTool(project_id, pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self._register_api_gateway(pod_id)
        except HCCIException as e:
            logger.error('Execute error: {}'.format(str(e)))
            return Message(500, HCCIException(753180, e))
        except Exception as e:
            logger.error('Execute error: {}'.format(str(e)))
            return Message(500, HCCIException(753180, e))

    def _register_api_gateway(self, pod_id):
        service_domain_name = f"csbs-vbs-api.{self.params_dict['region_id']}." \
                              f"{self.params_dict['global_domain_name']}:8799"
        logger.info('service_domain_name: {}'.format(service_domain_name))
        api_gateway_domain_name = f"csbs-vbs.{self.params_dict['region_id']}.{self.params_dict['global_domain_name']}"
        logger.info('api_gateway_domain_name: {}'.format(api_gateway_domain_name))
        as_api_config = get_config_content(os.path.join(self.param_tool.get_project_plugins_path(),
                                                        "plugins/CSBS/conf/deploy/karbor/api_config.yaml"))
        as_api_config = as_api_config % (api_gateway_domain_name, service_domain_name)

        logger.info("Start deleting old API info.")
        # 注册前先执行一次删除, 确保信息清空
        try:
            APIGApi().register_api_by_content(pod_id=pod_id,
                                              service_name=self.service_name,
                                              vars_content=as_api_config,
                                              action='Remove Api')
        except Exception as e:
            # 移除出错不抛异常
            logger.error(f"Remove API failed, errMsg={str(e)}")

        # 开始注册API
        logger.info("Start registering API gateway.")
        result = APIGApi().register_api_by_content(pod_id=pod_id,
                                                   service_name=self.service_name,
                                                   vars_content=as_api_config,
                                                   action='Register Api')
        if result:
            logger.info("Succeed registering API gateway.")
            return Message(200)
        else:
            logger.info("Failed registering API gateway.")
            return Message(500, HCCIException(753180, ""))
