import utils.common.log as logger
from utils.common.message import Message

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.step_base import DeployBaseSubJob

logger.init("CSBS")


class SecurityHardening(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(SecurityHardening, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self._secure_harden()
        except Exception as e:
            logger.error(f"Execute error:{e}")
            return Message(500, f"Failed to execute security hardening, "
                                f"err_msg:{e}.")

        logger.info("Succeed to execute karbor node security hardening.")
        return Message(200)

    def _set_os_om_user_first_login(self):
        karbor_client = SshClient()
        user_info = self.karbor_util.get_karbor_user_info()
        logger.info("Login to karbor node and "
                    "setting OS O&M user first login.")
        cmd = f"chage -d0 {user_info.user_name}"
        for node in self.karbor_util.get_karbor_node_list():
            try:
                # 判断节点密码是否已经过期，如果已经过期则已经设置了过期时间
                if self._check_password_expires(node, karbor_client):
                    logger.info(f"The node ip of {node.ip} has been "
                                "set first login.")
                    continue
            except Exception as e:
                if str(e).find("Bad authentication") != -1:
                    logger.info(f"The karbor node of ip {node.ip} "
                                "has already set first login.")
                    continue
                raise Exception(e) from e

            ssh_client = karbor_client.get_ssh_client(node)
            result = karbor_client.ssh_exec_command_return(ssh_client, cmd)
            if not karbor_client.is_ssh_cmd_executed(result):
                logger.error("Failed to set password expiration on "
                             f"karbor node, node ip:{node.ip}.")
                karbor_client.ssh_close(ssh_client)
                raise Exception("Failed to set password expiration on "
                                f"karbor node, node ip:{node.ip}.")
            logger.info("Succeed to set password expiration on "
                        f"karbor node, node ip:{node.ip}.")
            karbor_client.ssh_close(ssh_client)
        logger.info("Succeed to set password expiration "
                    "for all karbor node.")

    @staticmethod
    def _check_password_expires(node, karbor_client):
        ssh_client = karbor_client.create_ssh_client(
            node.ip, node.user, node.user_pwd)
        return "Current password".lower() in \
               ssh_client.get("prompt", "").lower()

    def _del_karbor_dirs(self):
        logger.info("Start to delete karbor dirs.")
        cmd = "rm -rf /karbor/CSBS;find /karbor -maxdepth 1 -type d -name CSBS|wc -l"
        ssh_client = SshClient()
        fail_to_del_karbor_node_list = list()
        for karbor_node in self.karbor_util.get_karbor_node_list():
            karbor_client = ssh_client.get_ssh_client(karbor_node)
            result = ssh_client.ssh_send_command_expect(karbor_client, cmd, "0", 30)
            ssh_client.ssh_close(karbor_client)
            if ssh_client.failed_to_return(result, "0", karbor_client):
                fail_to_del_karbor_node_list.append(karbor_node)
        if fail_to_del_karbor_node_list:
            raise Exception(f"Failed to del karbor dirs in {fail_to_del_karbor_node_list}.")
        logger.info('Delete karbor dirs success.')

    def _secure_harden(self):
        self._del_karbor_dirs()
        self._set_os_om_user_first_login()
