import os
import re
import subprocess
import tarfile
import zipfile

from utils.common.exception import HCCIException
import utils.common.log as logger
from utils.common.param_check import check_param_ip, check_param_integer

from plugins.CSBS.common.util import check_shell_inject_args
from plugins.CSBS.scripts.deploy.karborproxy.libs.api_adapter import API

MB_SIZE = 1024 * 1024 * 1024 * 5
# 最大文件数量限定10万个
MAX_FILE_COUNT = 1 * 1000 * 100


def exe_shell_get_result(cmd):
    check_shell_inject_args(cmd)
    exc_result = subprocess.Popen(
        args=["/bin/bash", "-c", cmd],
        shell=False,
        stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT
    )
    out, _ = exc_result.communicate(timeout=300)
    result = out.decode().strip()
    return result


def check_zipfile(file_list):
    """
    Check the size of the zip and tar.gz files.
    :param file_list:
    :return:
    """
    for file in file_list:
        if file.endswith(".zip"):
            zip_file = zipfile.ZipFile(file, "r")
            zip_file_name_list = zip_file.namelist()
            if len(zip_file_name_list) > MAX_FILE_COUNT:
                raise Exception(f"zipfile: {file} is too many.")
            file_size = 0
            for zip_file_name in zip_file_name_list:
                file_info = zip_file.getinfo(zip_file_name)
                file_size += file_info.file_size
            if file_size > MB_SIZE:
                raise Exception("zipfile: %s is too large." % file)
        elif file.endswith(".tar.gz"):
            tar_files = tarfile.open(file, "r")
            try:
                check_tar_file(tar_files)
            except Exception as err:
                raise Exception(f'tarfile {file} check failed: {str(err)}.') from err
            finally:
                tar_files.close()


def check_tar_file(tar_files):
    file_size = 0
    file_count = 0
    for tar_file in tar_files.getmembers():
        file_count += 1
        file_size += tar_file.size
    if file_count > MAX_FILE_COUNT:
        raise Exception(f"zipfile: {tar_files} is too many.")
    if file_size > MB_SIZE:
        raise Exception(f"tarfile: {tar_files} is too large.")


def get_os_file(os_file_name, os_zip_file_name):
    _, os_template = API.find_file(
        os_file_name, ".qcow2", scan_subdir=False)
    if not os_template:
        zippath, zipname = API.find_file(
            "OceanStor BCManager", os_zip_file_name, scan_subdir=False)
        if not zipname:
            raise HCCIException(
                640091, "OceanStor BCManager..." + os_zip_file_name)
        file_path = os.path.join(zippath, zipname)
        check_zipfile([file_path])
        API.extract_file(file_path, "/home/pkg/")
        _, os_template = API.find_file(
            os_file_name, ".qcow2", scan_subdir=False)
    logger.info("find " + os_file_name + " succ.")
    return os_template


def check_url_list(value, ip_version):
    ip_version = int(ip_version[-1])
    for sub_url in value.split(','):
        if ip_version == 4:
            urlinfo = sub_url.split(':')
        else:
            urlinfo = []
            start_index = sub_url.find('[')
            if start_index != 0:
                return False
            end_index = sub_url.find(']')
            if end_index == -1:
                return False
            urlinfo.append(sub_url[start_index + 1:end_index])
            if len(sub_url) < (end_index + 3):
                return False
            if sub_url[end_index + 1] != ':':
                return False
            urlinfo.append(sub_url[(end_index + 2):])

        if len(urlinfo) != 2:
            return False
        if not check_param_ip(urlinfo[0], ip_version):
            return False
        if not check_param_integer(urlinfo[1]):
            return False
    return True


def check_param_string(param_name):
    pattern = r"^[-a-zA-Z0-9:/_\.]{1,}$"
    match_obj = re.match(pattern, param_name)
    return bool(match_obj)


def check_openstack_region_sub(param_name):
    pattern = r"^[a-z0-9][a-z0-9-]{0,48}[a-z0-9]" \
              r"\.[a-z0-9][a-z0-9-]{0,48}[a-z0-9]$"

    match_obj = re.match(pattern, param_name, re.I)
    return bool(match_obj)
