import re

import IPy
import utils.common.log as logger
from utils.business.project_condition_utils import get_project_condition_boolean

from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karborproxy.config.agent_proxy_param import AgentProxyDPAInfos
from plugins.CSBS.scripts.deploy.karborproxy.config.project_set import project_preset
from plugins.CSBS.scripts.deploy.karborproxy.config.ssh_cmd import SShCmd

logger.init("KarborProxy ParamCheck")
DEFAULT_REQUEST_TIMEOUT = 120


class ProxyCheck(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super().__init__(project_id, pod_id, regionid_list)
        self.ssh = SShCmd()
        self.agent_proxy = AgentProxyDPAInfos(project_id, pod_id)

    @project_preset
    def execute(self, project_id, pod_id, regionid_list=None):
        self.check_dpa_cidr()
        self.check_karproxy_expired()
        self.check_dpa_az_available()

    def check_karproxy_expired(self):
        logger.info('[KarborProxy]: KarborProxy Expiration Check Start')
        if get_project_condition_boolean(self.agent_proxy.project_id,
                                         "InstalledCSBS-VBS-Proxy"):
            for node in self.agent_proxy.get_proxy_node_list():
                self.ssh.check_node_password_expired(node)

    def check_dpa_az_available(self):
        if get_project_condition_boolean(
                self.agent_proxy.project_id, 'ExpansionAdCloudService'):
            logger.info('[KarborProxy]: AZ Check Start')
            if self.agent_proxy.az_related_to_dpa:
                self.ssh.check_az_available(self.agent_proxy)

    def check_dpa_cidr(self):
        """
        检查内大网信息：
        Returns:

        """
        if get_project_condition_boolean(self.agent_proxy.project_id, "InstallCSBS-VBS-Proxy"):
            math_result = self.check_network_segment(self.agent_proxy.dpa_cidr)
            if not math_result:
                raise Exception("DPA DPA_Service_Cidr is not a network segment.")
            for tmp_url in self.agent_proxy.dpa_business_address_list:
                ip = tmp_url[0].split(':')[0] if isinstance(tmp_url, tuple) else tmp_url.split(':')[0]
                try:
                    IPy.IP(self.agent_proxy.dpa_cidr)
                except Exception as e:
                    raise Exception("DPA DPA_Service_Cidr is not a network segment.") from e
                if ip not in IPy.IP(self.agent_proxy.dpa_cidr):
                    raise Exception("The DPA_Service_Cidr parameter is incorrect.")

    @staticmethod
    def check_network_segment(net_work):
        pattern = r'^(?=(\b|\D))(((\d{1,2})|(1\d{1,2})|(2[0-4]\d)|(25[0-5]))' \
                  r'\.){3}((\d{1,2})|(1\d{1,2})|(2[0-4]\d)|(25[0-5]))(?=(\b|\
                  D))/([1-2][0-9]|3[0-2]|[1-9])$'
        match_obj = re.match(pattern, net_work)
        return match_obj
