import datetime
import json
import time

import requests
import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.ssh_util import Ssh

from plugins.CSBS.common.iam_util import IamUtil
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karborproxy.config.agent_proxy_param import AgentProxy
from plugins.CSBS.scripts.deploy.karborproxy.config.project_set import project_preset
from plugins.CSBS.scripts.deploy.karborproxy.config.shell_cmd import ShellCmd
from plugins.CSBS.scripts.deploy.karborproxy.config.ssh_cmd import SShCmd

logger.init("KarborProxy")

DPA_ACCOUNT_DESC = '{"zh-cn":"使用该帐户登录KarborProxy虚拟机节点' \
                   '进行运维操作，查看节点配置，查看日志，' \
                   '巡检和信息收集等。","en-us":"Use this account to log in to the ' \
                   'KarborProxy VM node to perform O&M operations, view node ' \
                   'configurations, logs, inspection, and information ' \
                   'collection."} '
ROOT_ACCOUNT_DESC = '{"zh-cn":"考虑到对系统安全的影响，在登录节点时，' \
                    '不能使用root用户直接登录，需要由dpamanager' \
                    '用户登录后通过执行命令 su - ' \
                    'root切换为root用户。切换到root' \
                    '帐号后，可对服务进行日常操作和维护，例如执行进程查看、' \
                    '日志查看、密码修改、配置文件修改等操作。",' \
                    '"en-us":"To ensure system security, you cannot log in ' \
                    'to the node as user root. You need to log in to the ' \
                    'node as user dpamanager and run the su - root command to ' \
                    'switch to user root. After switching to user root, ' \
                    'you can perform routine operations and maintenance on ' \
                    'services, such as viewing processes and logs, changing ' \
                    'passwords and configuration files."} '

SYSTEM_TYPE = "CSBS-VBS"
PASSWD_CHANGE_MAX_BETWEEN = 90
INSECURE = False

RETRY_TIME_LIMIT = 5
# 命令返回的标准长度
SSH_RET_LENGTH = 7
EXCEPT_CODE = -3
ERROR_CODE = -2
COMMON_CODE = 0
# 最大天数
MAX_DAY_NUM = 999


class RegisterUniPassword(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super().__init__(project_id, pod_id, regionid_list)
        self.params = ParamUtil()
        self.project_id = project_id
        self.pod_id = pod_id
        self.ssh_obj = Ssh()
        self.headers = None
        self.auth_url = ""
        self.agent_proxy = AgentProxy(project_id, pod_id)
        self.ssh = SShCmd()
        self.iam_util = IamUtil(self.project_id, self.pod_id)
        self.agent_proxy.op_service_account = self.iam_util.get_iam_account_info()

    @project_preset
    def execute(self, project_id, pod_id, regionid_list=None):
        self.agentproxy_register_unify_pwd()
        self.update_config()
        return Message(200)

    def agentproxy_register_unify_pwd(self):
        """
        注册统一密码，注册两个IP，两个账号。
        1、拿到IP
        2、构建请求体，
        3、发送请求体。
        :return:
        """
        # 两个需要注册的IP
        iplist = list((self.agent_proxy.karbor_proxy_ip0,
                       self.agent_proxy.karbor_proxy_ip1))
        need_register_accounts = ['dpamanager', 'root']
        for node_ip in iplist:
            for account in need_register_accounts:
                last_time, expire_time, quality_item = \
                    self.get_account_time_about(account, node_ip)
                data = self.build_unify_pwd_body(last_time,
                                                 expire_time, quality_item,
                                                 node_ip, account)
                result = self.send_unify_pwd_req(data)
                if not result:
                    raise HCCIException(640103)

    def get_account_time_about(self, user_name, node_ip):
        """
        获得账号的时间相关信息
        Args:
            node_ip: 节点ip
            user_name: 账户信息

        Returns:

        """
        user_info = {}
        if user_name == "dpamanager":
            user_info = {"name": user_name,
                         "passwd": self.agent_proxy.dpamanager_pwd}
        elif user_name == "root":
            user_info = {"name": user_name,
                         "passwd": self.agent_proxy.root_pwd}
        last_time, expire_time, _ = self.get_password_about_time(
            node_ip, user_info)
        quality_item = self.build_unify_pwd_quality(user_name)
        return last_time, expire_time, quality_item

    def build_unify_pwd_body(self, last_time, expire_time,
                             quality_item, node_ip, account,
                             operation_type=0,
                             user_pwd=""):
        """构建注册请求体。

        :param last_time:
        :param expire_time:
        :param quality_item:
        :param node_ip:
        :param account:
        :param operation_type:
        :param user_pwd:
        :return: data [dict] 注册需要发送的请求体。
        """

        account_decs = ""
        account_type = 0
        sub_com_name = "KarborProxy"

        data = {
            "componentName": SYSTEM_TYPE,
            "subComponents": [
                {
                    "subComponentName": sub_com_name,
                    "createdAccountList": []
                }
            ]
        }
        account_pwd = None
        modify_type = None
        if account == "dpamanager":
            account_decs = DPA_ACCOUNT_DESC
            modify_type = 1
            if user_pwd:
                account_pwd = user_pwd
            else:
                account_pwd = self.agent_proxy.dpamanager_pwd
            account_type = 1

        elif account == "root":
            account_decs = ROOT_ACCOUNT_DESC
            modify_type = 2
            if user_pwd:
                account_pwd = user_pwd
            else:
                account_pwd = self.agent_proxy.root_pwd
            account_type = 1
        account_old_pwd = account_pwd
        expire_time = -1 if account == "root" else expire_time
        account_dic = dict(accountName=account,
                           region=self.agent_proxy.region_id,
                           accountType=account_type,
                           accountDescription=account_decs, passwd=account_pwd,
                           oldPasswd=account_old_pwd,
                           lastPasswdChange=last_time, modifyType=modify_type,
                           passwdExpires=expire_time,
                           riskMessage='{"zh-cn":"","en-us":""}',
                           passwdComplexity=quality_item, ip=node_ip,
                           usedScene=node_ip, operationType=operation_type)
        data["subComponents"][0]["createdAccountList"].append(account_dic)
        return data

    def send_unify_pwd_req(self, data):
        """
        发送统一密码请求
        Parameters
        ----------
        data

        Returns
        -------

        """
        iam_token = self.iam_util.get_iam_token(self.agent_proxy.op_service_account)
        register_url = "https://%s/rest/mounpwdservice/v1/account" % \
                       self.agent_proxy.om_url
        header = {
            "Content-Type": "application/json",
            "X-Auth-Token": iam_token
        }
        register_retry_times = 0
        while True:
            try:
                rsp = requests.post(register_url, headers=header, data=json.dumps(data), verify=False)
            except Exception as err:
                logger.error("Register service to mo unify pass_word occur exc, info: %s" % str(err))
            else:
                logger.info("Register unify pass_word result code:%s" % rsp.status_code)
                if rsp.status_code <= 300 and rsp.content:
                    rsp_content = json.loads(rsp.content)
                else:
                    return False
                if rsp_content.get("code") == "00000000":
                    logger.info(
                        "Register service to mo unify pass_word "
                        "success, response code:%s"
                        % rsp_content.get("code"))
                    return True
                logger.error("Register service to mo unify pass_word failed, msg: %s" % rsp.content)

            register_retry_times += 1
            if register_retry_times > RETRY_TIME_LIMIT:
                logger.error("Unify pass_word reach the maximum number of retries, so do not try again")
                return False
            time.sleep(30)

    def get_password_about_time(self, node_ip, user_info):
        """
        获取密码相关的时间信息。
        :param node_ip:
        :param user_info:
        :return:
        """
        user_name = user_info.get('name')
        passwd = user_info.get('passwd')
        output_list = []
        if user_name == "dpamanager":
            sign = "$"
            ssh_client = self.ssh_obj.ssh_create_client(node_ip, user_name, passwd)
            cmd = "chage -l {}".format(user_name)
            output_list = Ssh.ssh_send_command(ssh_client, cmd, sign, 20)
            output_list = output_list[1:]
        if user_name == "root":
            cmd = "chage -l {}".format(user_name)
            output_list = self.ssh_obj.ssh_cmds(node_ip, cmd, 'dpamanager', self.agent_proxy.dpamanager_pwd,
                                                self.agent_proxy.root_pwd, "", "")
        expired_time = ""
        change_time = ""
        inactive_time = ""
        expired_str = ""
        # 只要命令返回的结果
        if len(output_list) < SSH_RET_LENGTH:
            raise Exception(f"Insufficient length of ssh cmd return val, error code:{ERROR_CODE}")
        try:
            for out_put_line in output_list:
                out_put_list = out_put_line.split(":")
                if len(out_put_list) < 2:
                    continue
                temp_name = out_put_list[0].strip()
                temp_value = out_put_list[1].strip()
                if temp_name == 'Password expires':
                    expired_str = temp_value
                    expired_time = datetime.datetime.utcnow() + datetime.timedelta(days=MAX_DAY_NUM) \
                        if expired_str == 'never' else datetime.datetime.strptime(expired_str, '%b %d, %Y')
                elif temp_name == 'Last password change':
                    change_time = datetime.datetime.strptime(temp_value, '%b %d, %Y')
                elif temp_name == 'Password inactive':
                    inactive_str = temp_value
                    inactive_time = datetime.datetime.utcnow() + datetime.timedelta(days=MAX_DAY_NUM) \
                        if inactive_str == 'never' else datetime.datetime.strptime(expired_str, '%b %d, %Y')
        except Exception as error:
            # 执行命令结果解析，错误
            logger.error(f"Failed to parse ssh command results. Error Message: {error}")
            raise Exception(
                f"Failed to parse ssh command results, except code: {EXCEPT_CODE}. Error Message: {error}") from error
        change_time = int(time.mktime(change_time.timetuple())) * 1000
        expired_time = int(time.mktime(expired_time.timetuple())) * 1000
        inactive_time = int(time.mktime(inactive_time.timetuple())) * 1000
        return change_time, expired_time, inactive_time

    @staticmethod
    def build_unify_pwd_quality(user_name):
        """
        构建统一密码复杂度
        Parameters
        ----------
        user_name

        Returns
        -------

        """
        account_min_time = 0
        account_max_time = 0
        if user_name == "dpamanager":
            account_min_time = 0
            account_max_time = 90
        elif user_name == "root":
            account_min_time = 0
            account_max_time = 99999
        quality_item = {
            "difok": 3,
            "minlen": 8,
            "passwdChangeMinBetweenTime": account_min_time,
            "passwdChangeMaxBetweenTime": account_max_time
        }
        return quality_item

    def update_config(self):
        """
        修改KarborProxy节点配置，防止密码校验不通过。
        """
        self.ssh.ssh_cmds_to_all_proxys(self.agent_proxy, ShellCmd.update_group)
        self.ssh.ssh_cmds_to_all_proxys(self.agent_proxy, ShellCmd.change_password_complexity)
