import json
import os.path

import utils.common.log as logger
from utils.common.ssh_util import Ssh

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karborproxy.common.funtions import exe_shell_get_result, check_zipfile
from plugins.CSBS.scripts.deploy.karborproxy.config.agent_proxy_param import AgentProxy
from plugins.CSBS.scripts.deploy.karborproxy.config.pkg_param import PkgParam
from plugins.CSBS.scripts.deploy.karborproxy.config.project_set import project_preset
from plugins.CSBS.scripts.deploy.karborproxy.config.shell_cmd import ShellCmd
from plugins.CSBS.scripts.deploy.karborproxy.config.ssh_cmd import SShCmd

logger.init("KarborProxy Upload The AgentAssistant Package")

agent_pkgs = [
    "ls /home/pkg/CDM_*_Client_Linux_5_x64.tar.gz",
    "ls /home/pkg/CDM_*_Client_Linux_el7_aarch64.tar.gz",
    "ls /home/pkg/CDM_*_Client_Linux_el7_x64.tar.gz",
    "ls /home/pkg/CDM_*_Client_Linux_el8_x64.tar.gz",
    "ls /home/pkg/CDM_*_Client_NeoKylin_7_mips64el.tar.gz",
    "ls /home/pkg/CDM_*_Client_Windows_All_i386.zip",
    "ls /home/pkg/CDM_*_Client_Windows_All_x64.zip",
    "ls /home/pkg/plugins-oracle-*-Linux_5_x64.tar.gz",
    "ls /home/pkg/plugins-oracle-*-Linux_el7_x64.tar.gz",
    "ls /home/pkg/plugins-oracle-*-Linux_el8_x64.tar.gz",
    "ls /home/pkg/plugins-oracle-*-Windows_All_x64.tar.gz",
]
assistant_pkgs = {
    "ls /home/pkg/OceanStor_BCManager*AgentAssist_linux_x86_64.tar.gz": "linuxx86",
    "ls /home/pkg/OceanStor_BCManager*AgentAssist_linux_aarch64.tar.gz": "linuxarm",
    "ls /home/pkg/OceanStor_BCManager_*_AgentAssist_windows.zip": "windows"
}


class UploadAgentAssistantPackage(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super().__init__(project_id, pod_id, regionid_list)
        self.agent_proxy = AgentProxy(project_id, pod_id)
        self.ssh = SShCmd()
        self.ssh_obj = Ssh()
        self.karbor_info = KarborUtil(project_id, pod_id)
        self.sha256_file_list = []

    @project_preset
    def execute(self, project_id, pod_id, regionid_list=None):
        self._upload_agent_assistant_pkg_to_karborproxy()
        self._update_karborproxy_agent_pkg_chown()
        self._update_karborproxy_agent_pkg_auth()
        self._upload_agent_assistant_pkg_to_karbor()
        self._agent_generated_sha256()

    def _upload_agent_assistant_pkg_to_karbor(self):
        """
        1. Upload the assistant package to all Karbor nodes.
        2. Change the directory permission to 750,
         the file permission to 400, and the file owner group to openstack.
        :return:
        """
        logger.info("[Karbor] Uploading the Assistant Package to Kabor Starts")
        karbor_node_list = self.karbor_info.get_karbor_node_list()
        for karbor_node in karbor_node_list:
            self.ssh.ssh_cmd_to_karbor(karbor_node, ShellCmd.pkg_mkdir)
            self.ssh.ssh_cmd_to_karbor(karbor_node, ShellCmd.removes_i_attr)
            for cmd, os_type in assistant_pkgs.items():
                file_path = PkgParam().find_file(cmd)
                check_zipfile([file_path])
                self._upload_file_to_karbor(karbor_node, file_path)
                p7s_file_path = PkgParam().is_exists_p7s_file(file_path)
                if p7s_file_path:
                    logger.info('%s verification file' % file_path + '.p7s')
                    self._upload_file_to_karbor(karbor_node,
                                                file_path + '.p7s')
                    # 清理告警
                    alarm_data = {"OSType": os_type}
                    self.ssh.ssh_cmd_to_karbor(
                        karbor_node,
                        ShellCmd.clear_alarm % json.dumps(alarm_data))
                else:
                    result = exe_shell_get_result('sha256sum %s' % file_path)
                    sha256, _ = result.split('  ')
                    cms_key = os_type + '.sha256'
                    # 上传sha256值，写入数据库
                    self.ssh.ssh_cmd_to_karbor(
                        karbor_node,
                        ShellCmd.put_assist_pkg_sha256 % (cms_key, sha256))
            self.ssh.ssh_cmd_to_karbor(karbor_node, ShellCmd.update_pkg_chown)
            self.ssh.ssh_cmd_to_karbor(karbor_node, ShellCmd.update_pkg_chmod)
        logger.info("[Karbor] Succeeded "
                    "in uploading the assistant package to Kabor.")

    def _upload_file_to_karbor(self, karbor_node, file_path):
        self.ssh.put_file_to_kabor(karbor_node, file_path)
        file_name = os.path.basename(file_path)
        mv_pkg = ShellCmd.mv_pkg % file_name
        self.ssh.ssh_cmd_to_karbor(karbor_node, mv_pkg)

    def _upload_agent_assistant_pkg_to_karborproxy(self):
        root_path = '/opt/huawei-data-protection/agentproxy/pkg/agent'
        self.ssh.ssh_cmds_to_all_proxys(self.agent_proxy,
                                        ShellCmd.proxy_removes_i_attr)
        for cmd in agent_pkgs:
            file_path = PkgParam().find_file(cmd)
            check_zipfile([file_path])
            self.ssh.put_file_to_all_roots(
                self.agent_proxy, file_path, root_path)
            p7s_file_path = PkgParam().is_exists_p7s_file(file_path)
            if p7s_file_path:
                logger.info('%s verification file' % file_path + '.p7s')
                self.ssh.put_file_to_all_roots(
                    self.agent_proxy, file_path + '.p7s', root_path)
            else:
                self.sha256_file_list.append(file_path)
        root_path = \
            '/opt/huawei-data-protection/agentproxy/pkg/agent_assistant'
        for cmd in assistant_pkgs:
            file_path = PkgParam().find_file(cmd)
            self.ssh.put_file_to_all_roots(
                self.agent_proxy, file_path, root_path)
            p7s_file_path = PkgParam().is_exists_p7s_file(file_path)
            if p7s_file_path:
                logger.info('%s verification file' % file_path + '.p7s')
                self.ssh.put_file_to_all_roots(
                    self.agent_proxy, file_path + '.p7s', root_path)
            else:
                self.sha256_file_list.append(file_path)
        logger.info("[KarborProxy] find agent package succ.")

    def _update_karborproxy_agent_pkg_chown(self):
        logger.info("[KarborProxy] Modifying an Owner Group.")
        self.ssh.ssh_cmds_to_all_proxys(
            self.agent_proxy, ShellCmd.update_chown)

    def _update_karborproxy_agent_pkg_auth(self):
        logger.info("[KarborProxy] Modifying Package Permissions")
        self.ssh.ssh_cmds_to_all_proxys(self.agent_proxy,
                                        ShellCmd.update_auth)

    def _agent_generated_sha256(self):
        if self.sha256_file_list:
            logger.info("[KarborProxy] "
                        "'Generate the SHA256 value of the agent package.'")
            self.ssh.ssh_cmds_to_all_proxys(
                self.agent_proxy, ShellCmd.generated_sha256)
