import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.step_base import DeployBaseSubJob

logger.init("CSBS")


class RegisterCMDB(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(RegisterCMDB, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.service_name = "CSBS-VBS"
        self.params = ParamUtil()
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.region_id = \
            self.params.get_param_value(self.pod_id, "public", "region0_id")
        self.cmdb_service_name = f"{self.service_name}_{self.region_id}"

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self._karbor_proxy_register_cmdb()
        except HCCIException as e:
            logger.error("Failed to register the scale info to CMDB, err_msg:{}.".format(str(e)))
            return Message(500, e)
        except Exception as e:
            logger.error("Failed executing RegisterCMDB, err_msg:{}.".format(str(e)))
            return Message(500, HCCIException("640001", str(e)))
        logger.info('Succeed executing RegisterCMDB.')
        return Message(200)

    def _karbor_proxy_register_cmdb(self):
        service_info_list = self.get_cloud_service_info()
        service_version = service_info_list[0].get("version")
        if not service_version:
            raise Exception("Failed to obtain the service version.")

        cmdb_service_name = f"{self.service_name}_{self.region_id}"
        service_scale = self.params.get_project_scales_by_project_id(self.project_id).get("scale")
        cloud_service_info = dict()
        for service_info in service_info_list:
            if not service_info.get("name") == cmdb_service_name:
                continue
            cloud_service_info["indexName"] = service_info.get("indexName")
            cloud_service_info["version"] = service_info.get("version")
            cloud_service_info["name"] = service_info.get("name")
            extend_info_list = []
            for extend_info in service_info.get("extendInfos"):
                if extend_info.get("key") == "karborproxy_region_service_scale":
                    continue
                extend_info_list.append(extend_info)
            extend_info_list.append(
                {
                    "key": "karborproxy_region_service_scale",
                    "value": service_scale,
                }
            )
            cloud_service_info["extendInfos"] = extend_info_list

        # 注册CMDB
        logger.info(f"Start to register cloud service info to CMDB, "
                    f"cloud service info is: {cloud_service_info}.")
        self.cmdb_util.set_cloud_service_info(
            region_code=self.region_id,
            cloud_service_info=cloud_service_info)
        logger.info(f"Succeed to register {self.service_name} cmdb.")

        cloud_service_info = self.get_cloud_service_info()
        for extend_info in cloud_service_info[0]["extendInfos"]:
            if extend_info["key"] == "karborproxy_region_service_scale" and extend_info["value"] != service_scale:
                raise Exception("Failed to register the service scale.")

    def get_cloud_service_info(self):
        logger.info('Obtain {} cloud service information '
                    'from the CMDB.'.format(self.service_name))
        cloud_service_info = \
            self.cmdb_util.get_cloud_service_info(
                self.region_id, self.service_name,
                name=self.cmdb_service_name)
        if not cloud_service_info:
            raise Exception("Failed to get {} service information "
                            "from CMDB.".format(self.service_name))
        logger.info("The service information obtained from the CMDB "
                    "is as follows:{}.".format(cloud_service_info))
        return cloud_service_info
