import utils.common.log as logger
from utils.business.arb_util import ArbApi
from utils.common import param_check
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.karbor import KarborUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.util import check_string_param

logger.init("CSBS")


class KarborArbitration(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(KarborArbitration, self).__init__(project_id, pod_id, regionid_list)
        self.destination = "/home/djmanager/"
        self.arb_api = ArbApi()
        self.karbor_util = KarborUtil(self.project_id, self.pod_id)
        self.param_tool = ParamTool(self.project_id, self.pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self._arbitration()
        except HCCIException as e:
            logger.error(f"Execute error:{str(e)}.")
            return Message(500, e)

    def _arbitration(self):
        karbor_node_list = self.karbor_util.get_karbor_node_list()[:2]
        self._upload_arb_cert_to_karbor_node(karbor_node_list)
        result = self._docking_arbiter(karbor_node_list[0])
        if not result:
            return Message(500, HCCIException(640066))
        return Message(200)

    def _upload_arb_cert_to_karbor_node(self, karbor_node_list):
        logger.info('Start to upload arbitration certificates '
                    'to the karbor node.')
        for node in karbor_node_list:
            logger.info('Upload arbitration ca.crt '
                        'from Arbitration_DC1_01_API '
                        f'to karbor :{node.ip}, dir:{self.destination}.')
            self.arb_api.get_arb_cers(self.pod_id, node.ip,
                                      node.user,
                                      node.user_pwd,
                                      self.destination)
        logger.info('Succeed uploading arbitration certificates.')

    def _docking_arbiter(self, karbor_node):
        karbor_client = SshClient()
        install_type = self.karbor_util.get_install_type()
        if install_type != "private_recover":
            raise Exception("The installation type of the Karbor node does "
                            "not support arbitration, the current installation"
                            f" type is {install_type}.")

        logger.info('Ssh to karbor node and docking unify arbiter.')
        arb_info = self.arb_api.get_arb_info(self.pod_id)
        arb_ip_list = arb_info.get("ips")
        if not arb_ip_list:
            raise Exception("Failed to obtain arb ips.")
        dc1_name = arb_info.get('dc_name').get('dc1_name')
        dc2_name = arb_info.get('dc_name').get('dc2_name')
        if not dc1_name or not dc2_name:
            raise Exception("Failed to obtain arb dc name.")
        arb_user_pwd = arb_info.get('user_info').get('password')
        if not arb_user_pwd:
            raise HCCIException(640003, "arb_business_password")

        # 参数校验
        ip_version = self.param_tool.get_ip_version()
        for _ip in arb_ip_list:
            if not param_check.check_param_ip(_ip, ip_version):
                raise Exception(f"Invalid ip address, the parma is {_ip}.")
        if not check_string_param(dc1_name) or not check_string_param(
                dc2_name):
            raise Exception("The obtained character string is invalid.")

        cmd = f"source /opt/huawei/dj/inst/utils.sh;configArb {','.join(arb_ip_list)} {dc1_name} {dc2_name}"
        logger.info(f'Cmd: {cmd}.')
        ssh_client = karbor_client.get_ssh_client(karbor_node)
        expect_str = "Enter password of user(arbiterClient):"
        result = karbor_client.ssh_send_command_expect(ssh_client,
                                                       cmd,
                                                       expect_str + ";# ;$ ")
        if karbor_client.failed_to_return(result, expect_str, ssh_client):
            logger.error(f'Failed executing:{cmd}, expect_str:{expect_str}, '
                         f'result:{result}.')
            return False

        expect_str = "Config arbitration service successful."
        result = karbor_client.ssh_send_command_expect(ssh_client,
                                                       arb_user_pwd,
                                                       expect_str + ";# ;$ ")
        if karbor_client.failed_to_return(result, expect_str, ssh_client):
            logger.error(f'Failed, expect_str:{expect_str}, result:{result}.')
            return False
        logger.info('Succeed to connect unified quorum server.')
        return True
