import time
from collections import defaultdict

import utils.common.log as logger

from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.scripts.upgrade.ab_test.tasks.test_util import get_checkpoint_item_detail
from plugins.CSBS.scripts.upgrade.ab_test.tasks.test_util import get_checkpoint_item_list
from plugins.CSBS.scripts.upgrade.ab_test.tasks.test_util import waiting_checkpoint_item_available

logger.init("CSBS-VBS")


class BackupTest(object):
    def __init__(self, context, http_client):
        self.context = context
        self.http_client = http_client
        self.console_host = self.context.basic_mo_test["console_host"]
        self.project_id = self.context.user_info['project_id']

    def csbs_backup(self):
        csbs_policy_info = self.context.csbs_policy_info
        csbs_backup_info = self._backup(csbs_policy_info['policy_id'],
                                        csbs_policy_info['resource_id_list'],
                                        constant.ECS_PRODUCT_TYPE)
        setattr(self.context, 'csbs_backup_info', csbs_backup_info)

    def vbs_backup(self):
        vbs_policy_info = self.context.vbs_policy_info
        vbs_backup_info = self._backup(vbs_policy_info['policy_id'],
                                       vbs_policy_info['resource_id_list'],
                                       constant.EVS_PRODUCT_TYPE)
        setattr(self.context, 'vbs_backup_info', vbs_backup_info)

    def _backup(self, policy_id, resource_id_list, product_type,
                auto_trigger=False, incremental=True):
        logger.info("Start performing backup.")

        if product_type == "ecs":
            provider_id = constant.CSBS_PROVIDER_ID
        elif product_type == "evs":
            provider_id = constant.VBS_PROVIDER_ID
        else:
            raise Exception("The product_type should be ecs or evs.")
        url = "https://{console_host}/cbs/rest/karbor/v1" \
              "/{project_id}/providers/{provider_id}" \
              "/checkpoints".format(console_host=self.console_host,
                                    project_id=self.project_id,
                                    provider_id=provider_id)
        body_dict = {
            "checkpoint": {
                "plan_id": policy_id,
                "parameters": {
                    "auto_trigger": auto_trigger,
                    "resources": resource_id_list,
                    "incremental": incremental
                }
            }
        }
        _, ret = self.http_client.post(url, body=body_dict)

        if ret["checkpoint"]["status"] == "protecting":
            logger.info("Expected result: Backup task issued successfully.")
        logger.info("Wait for the backup task to be completed.")
        checkpoint_id = ret["checkpoint"]["id"]

        # get checkpoint_item_id
        checkpoint_item_id = self._get_checkpoint_item_id(checkpoint_id,
                                                          product_type)
        logger.info("The checkpoint_item_id is {}.".format(checkpoint_item_id))
        waiting_checkpoint_item_available(self.http_client,
                                          self.console_host,
                                          self.project_id,
                                          checkpoint_item_id)
        logger.info("Backup finished.")
        backup_info = {
            "checkpoint_id": checkpoint_id,
            "checkpoint_item_id": checkpoint_item_id
        }
        return backup_info

    def _get_checkpoint_item_id(self, checkpoint_id, product_type):
        checkpoint_item_id = self._get_current_checkpoint_item_id(
            checkpoint_id,
            product_type)
        if product_type == "ecs":
            return checkpoint_item_id

        # evs need to update checkpoint item id
        is_end = False
        retry_times = 0
        # The copy id firstly got will be going to change later
        while not is_end:
            try:
                if retry_times <= 30:
                    logger.info("The {} time to determine whether the copy "
                                "id has changed.".format(retry_times + 1))
                    get_checkpoint_item_detail(self.http_client,
                                               self.console_host,
                                               self.project_id,
                                               checkpoint_item_id)
                    time.sleep(10)
                    retry_times += 1
                else:
                    is_end = True
            except Exception as e:
                logger.warn(str(e))
                is_end = True
        new_checkpoint_item_id = self._get_current_checkpoint_item_id(
            checkpoint_id,
            product_type)
        logger.info("The copy id has changed and a new id {} "
                    "is obtained.".format(new_checkpoint_item_id))
        return new_checkpoint_item_id

    def _get_current_checkpoint_item_id(self, checkpoint_id, product_type):
        _, checkpoint_info = get_checkpoint_item_list(self.http_client,
                                                      self.console_host,
                                                      self.project_id,
                                                      product_type)
        checkpoint_item_id = None
        for checkpoint_item in checkpoint_info["checkpoint_items"]:
            if checkpoint_item["checkpoint_id"] == checkpoint_id:
                checkpoint_item_id = checkpoint_item["id"]
                break
        if not checkpoint_item_id:
            raise Exception("Failed to get current checkpoint_item id, "
                            "please check.")
        return checkpoint_item_id

    def csbs_delete_copies(self):
        self._check_and_delete_copies(constant.ECS_PRODUCT_TYPE)

    def vbs_delete_copies(self):
        self._check_and_delete_copies(constant.EVS_PRODUCT_TYPE)

    def _check_and_delete_copies(self, product_type):
        logger.info("Check and delete copies.")

        _, checkpoint_item_list = get_checkpoint_item_list(self.http_client,
                                                           self.console_host,
                                                           self.project_id,
                                                           product_type)
        if not checkpoint_item_list.get('checkpoint_items'):
            logger.info("There are no copies needed to delete.")
        else:
            checkpoint_dict = defaultdict(list)
            for checkpoint_item_info in \
                    checkpoint_item_list["checkpoint_items"]:
                checkpoint_id = checkpoint_item_info['checkpoint_id']
                checkpoint_dict[checkpoint_id].append(
                    checkpoint_item_info['id'])
            logger.info("Current copy info: {}.".format(str(checkpoint_dict)))

            for checkpoint_id, checkpoint_item_ids in checkpoint_dict.items():
                self._delete_copies(checkpoint_id,
                                    checkpoint_item_ids,
                                    product_type)
            logger.info("Succeed to delete all copies.")

    def _delete_copies(self, checkpoint_id, checkpoint_item_ids, product_type):
        retry_times = 0
        is_end = False
        while not is_end:
            try:
                for checkpoint_item_id in checkpoint_item_ids:
                    _ = self._delete_copy(checkpoint_id, checkpoint_item_id, product_type)
            except Exception as err:
                logger.error(f"Deleting checkpoint_item_id: {checkpoint_item_ids}, "
                             f"retry times: {retry_times}, err: {err}.")
                if retry_times >= 10:
                    raise Exception("Failed deleting the copy, please check.") from err
                retry_times += 1
                time.sleep(10)
                continue
            is_end = True

    def _delete_copy(self, checkpoint_id, checkpoint_item_id, product_type):
        if product_type == "ecs":
            provider_id = constant.CSBS_PROVIDER_ID
        elif product_type == "evs":
            provider_id = constant.VBS_PROVIDER_ID
        else:
            raise Exception("The product type should be ecs or evs.")

        logger.info(f"Start deleting copy with id {checkpoint_item_id}.")
        try:
            _, body = get_checkpoint_item_detail(self.http_client, self.console_host,
                                                 self.project_id, checkpoint_item_id)
        except Exception as err:
            if "CSBS.3001" in str(err):
                logger.info("The backup copy does not exist, deleting successfully.")
                return True
            raise Exception(f"Failed to get the information of copy {checkpoint_item_id}, errMsg:{str(err)}.") from err
        logger.info("Succeed getting copy information.")
        checkpoint_status = body["checkpoint_item"]["status"]
        if checkpoint_status not in ["error", "available"]:
            raise Exception(f"The copy status is {checkpoint_status}, which can not to be deleted, please check.")

        url = f"https://{self.console_host}/cbs/rest/karbor/v1" \
              f"/{self.project_id}/providers/{provider_id}/checkpoints/{checkpoint_id}"
        self.http_client.delete(url)

        return self._check_delete_copy_result(checkpoint_id, product_type)

    def _check_delete_copy_result(self, checkpoint_id, product_type):
        logger.info("Waiting for the copy to be deleted successfully.")
        retry_times = 0
        delete_result = False
        while retry_times < 200:
            _, copy_list_info = get_checkpoint_item_list(self.http_client, self.console_host,
                                                         self.project_id, product_type)
            copy_checkpoint_id_list = []
            for copy in copy_list_info["checkpoint_items"]:
                copy_checkpoint_id_list.append(copy["checkpoint_id"])
            if checkpoint_id not in copy_checkpoint_id_list:
                delete_result = True
                logger.info("Succeed deleting the copy.")
                break
            time.sleep(10)
            retry_times += 1
            logger.info("Waited for {} seconds.".format(10 * retry_times))
        logger.info("Succeeded to delete the copy.")
        return delete_result
