import json
import time
from datetime import datetime, timedelta, timezone

import utils.common.log as logger

from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.scripts.upgrade.ab_test.tasks.tenant import TenantApi


POLICY_EXPIRED_TIME = 10


class FilesetTest(object):
    def __init__(self, context, http_client):
        self.context = context
        self.http_client = http_client
        self.console_host = self.context.basic_mo_test.get('console_host')
        self.project_id = self.context.user_info['project_id']
        self.region_id = self.context.engine_info['region_id']
        self.tenant_api = TenantApi(self.context, http_client)

    def get_agents_info(self):
        headers = {"X-MicroVersion": "1.2"}
        self.http_client.headers_update(headers)
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/hosts"
        return self.http_client.get(url)

    def fileset_apply_quota(self):
        self.apply_quota(product_name="文件集备份", quota_type="fileset_backup_capacity",
                         service_type="csbs")

    def fileset_create_policy(self):
        policy_name = f"{constant.FSBS_POLICY_NAME_PRE}_{time.strftime('%d%H%M%S')}"
        self.apply_policy(constant.FILESET_PROVIDER_ID, policy_name, resource_type="OS::Application::FileSet",
                          is_enable_full_backup=True)

    def fileset_query_tasklist(self):
        self.get_plan_logs(provider_id=constant.FILESET_PROVIDER_ID, resource_type="OS::Application::FileSet")

    def fileset_query_backup_copy(self):
        self.get_backup_copies("OS::Application::FileSet")

    def fileset_delete_policy(self):
        self.delete_policy(constant.FILESET_PROVIDER_ID, resource_type="OS::Application::FileSet")

    def fileset_release_quota(self):
        self.release_quota(product_name="文件集备份", quota_type="fileset_backup_capacity", service_type="csbs")

    def apply_quota(self, product_name, quota_type, service_type='csbs'):
        service_type_list = ['csbs', 'vbs']
        if service_type not in service_type_list:
            raise Exception("The service_type should be "
                            f"in {service_type_list}.")
        if quota_type not in constant.QUOTA_TYPE_DICT:
            raise Exception("The quota_type should be "
                            f"in {constant.QUOTA_TYPE_DICT}.")

        logger.info("Start to get quota information.")
        quota_info = self.get_quota_info()
        logger.info("Succeed to get quota information, "
                    f"quota info:{quota_info}.")

        is_need_add_quota = False
        cur_backup_quota = 0
        for quota in quota_info["quotas"]["resources"]:
            if quota["type"] != quota_type:
                continue
            cur_backup_quota = quota["quota"]
            if cur_backup_quota < constant.MINIMUM_QUOTA:
                logger.info(
                    f"The {quota_type} is {cur_backup_quota}M, "
                    "less than the established capacity.")
                is_need_add_quota = True
            else:
                logger.info(f"The {quota_type} is {cur_backup_quota}M, "
                            "which can be used normally.")
        if is_need_add_quota:
            logger.info(f"Start to apply {quota_type} for {service_type} "
                        "backup space.")
            quota_num = constant.MINIMUM_QUOTA - cur_backup_quota
            self._add_quota(service_type, product_name, quota_type, quota_num)
            logger.info(f"Succeed to apply {quota_type} quota.")

    def release_quota(self, quota_type, product_name, service_type='csbs'):
        logger.info("Start to get quota information.")
        quotas_info = self.get_quota_info()
        logger.info("Get quotas info successfully, "
                    f"quotas_info is:{str(quotas_info)}.")
        for quota in quotas_info["quotas"]["resources"]:
            if quota_type == quota["type"]:
                quota_num = quota["quota"]
                if quota_num != 0:
                    logger.info(f"Start to release quota, quota type is {quota_type}, "
                                f"quota quantity  is {quota_num}.")
                    quota_params = self._assemble_quota_params(quota_type, -quota_num)
                    self._modify_quota(service_type, product_name, quota_params)
                    logger.info(f"Succeed release quota, quota type is {quota_type}.")

    def get_quota_info(self):
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/quotas"
        _, body = self.http_client.get(url)
        return body

    def _add_quota(self, service_type, product_name, quota_type, quota_num):
        logger.info("Start to increase quota space.")
        params = self._assemble_quota_params(quota_type, quota_num)
        quota_info = self._modify_quota(service_type, product_name, params)
        logger.info("Succeed in increasing backup space.")
        return quota_info

    def get_product_id_from_name(self, product_name, service_type="csbs"):
        """根据product名字获取product_id

        :param :
        :return:
        """
        res = self.tenant_api.get_project_infos_by_service_type(service_type)
        for product in res:
            if product_name in product.get("name") and product.get("online_status") == "online" \
                    and not product.get("approval"):
                product_id = product.get('product_id')
                logger.info(f"Get product id successfully, product name={product_name}, id={product_id}.")
                return product_id
        error_msg = f"Failed to obtain the {service_type} product id, " \
                    "check whether the product is online or needs to " \
                    "be approved. If the product is offline, " \
                    "log in to ManageOne Operation Portal as an " \
                    "operation administrator and retry this step " \
                    "after the product is online. If the product " \
                    "needs to be approved, log in to ManageOne " \
                    "Operation Portal as an operation administrator, " \
                    "complete all existing approval processes for the " \
                    "product, cancel the product approval process, " \
                    "and retry this step. Otherwise, contact " \
                    "technical support."
        logger.error(error_msg)
        raise Exception(error_msg)

    def _modify_quota(self, service_type, product_name, quota_params):
        product_id = self.get_product_id_from_name(product_name)
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/products" \
              f"/{product_id}/action"

        body_dict = {
            "apply": {
                "service_type": service_type,
                "region_id": self.region_id,
                "params": quota_params
            }
        }
        return self.http_client.post(url, json=body_dict)

    @staticmethod
    def _assemble_quota_params(capacity_type, var_quota_num):
        if capacity_type not in constant.QUOTA_TYPE_DICT:
            raise Exception(f"The quota_type should be in {constant.QUOTA_TYPE_DICT}.")
        display_num = var_quota_num // 1024
        params = {
            "count": 1,
            "param_items": [
                {
                    "type": capacity_type,
                    "quota": var_quota_num
                }
            ],
            "display": {
                "en_US": [
                    {
                        "label": constant.QUOTA_TYPE_DICT[capacity_type]['en_label'],
                        "type": "string",
                        "value": f"{display_num}GB"
                    }
                ],
                "zh_CN": [
                    {
                        "label": constant.QUOTA_TYPE_DICT[capacity_type]['cn_label'],
                        "type": "string",
                        "value": f"{display_num}GB"
                    }
                ]
            }
        }
        return json.dumps(params)

    def get_backup_copies(self, resource_type):
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/checkpoint_items" \
              f"?resource_type={resource_type}"
        return self.http_client.get(url)

    def get_plan_logs(self, provider_id, resource_type, operation_type=None):
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/operation_logs" \
              f"?resource_type={resource_type}&provider_id={provider_id}"
        if operation_type:
            if not isinstance(operation_type, (list, tuple)):
                raise Exception("The value of 'operation_type' must be a list or tuple.")
            if not set(operation_type) <= set(constant.OP_LOG_TYPE):
                raise Exception(f"The value of 'operation_type' must be in {constant.OP_LOG_TYPE}, "
                                f"the 'operation_type' is {operation_type}.")
            for _type in operation_type:
                url += f"&operation_type={_type}"
        return self.http_client.get(url)

    def get_policies(self, provider_id, resource_type=None):
        # 云服务器和云硬盘备份策略不能传入resource_type
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/policies?" \
              f"provider_id={provider_id}"
        if resource_type:
            url += f"&resource_type={resource_type}"
        _, body = self.http_client.get(url)
        return body

    def _get_policy_detail_by_policy_id(self, policy_id):
        url = f"{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/policies/{policy_id}"
        return self.http_client.get(url=url, verify=False)

    def apply_policy(self, provider_id, policy_name, resource_type,
                     resource_list=None,
                     is_enable_backup=False, is_enable_copy=False,
                     is_enable_full_backup=False):
        datetime.utcnow()
        expired_time = datetime.now(tz=timezone.utc) + timedelta(POLICY_EXPIRED_TIME)
        expired_time_str = expired_time.strftime("%Y-%m-%dT%H:%M:%S")
        if not resource_list:
            resource_list = []
        if not isinstance(resource_list, list):
            raise Exception("The value of 'resource_list' must be a list.")

        logger.info("Start to create a backup plan "
                    "that sets up a scheduling policy.")
        return self._post_create_policy_order(
            provider_id, policy_name,
            resource_type, resource_list,
            expired_time=expired_time_str,
            is_enabled_backup=is_enable_backup,
            is_enabled_copy=is_enable_copy,
            is_enabled_full_backup=is_enable_full_backup)

    def delete_policy(self, provider_id, resource_type=None):
        logger.info("Start to obtain policy info, "
                    f"provider_id: {provider_id}, "
                    f"resource_type: {resource_type}.")
        policy_info_list = self.get_policies(provider_id, resource_type).get("policies", None)
        logger.info(f"Succeed to obtain policy info, policy infos: {policy_info_list}.")
        policy_id_list = [policy_info["id"] for policy_info in policy_info_list]
        for policy_id in policy_id_list:
            self.delete_policy_by_policy_id(policy_id)

    def delete_policy_by_policy_id(self, policy_id):
        logger.info(f"Start to delete policy, policy id: {policy_id}.")
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/policies/{policy_id}"
        self.http_client.delete(url)
        logger.info(f"Succeed to delete policy, policy id: {policy_id}.")

    def _post_create_policy_order(self, provider_id, policy_name,
                                  resource_type, resources,
                                  expired_time="", is_enabled_backup=False,
                                  is_enabled_copy=False,
                                  is_enabled_full_backup=False,
                                  encryption=False):
        if not is_enabled_backup and not is_enabled_full_backup and not is_enabled_copy:
            raise Exception("When creating a policy, specify the operation type of at least one policy. "
                            "The operation types of the policy are as follows:"
                            "full_backup, backup, copy.")
        url = f"https://{self.console_host}/cbs/rest/karbor/v1/{self.project_id}/policies"
        scheduling_name_backup = "scheduling_name_backup"
        scheduling_name_copy = "scheduling_name_copy"
        scheduling_name_full_backup = "scheduling_name_full_backup"

        scheduled_operations = []
        if is_enabled_full_backup:
            scheduling_full_backup = self._assemble_scheduled_operation(
                scheduling_name=scheduling_name_full_backup,
                is_enabled=is_enabled_full_backup,
                operation_type='full_backup')
            scheduled_operations.append(scheduling_full_backup)
        if is_enabled_backup:
            scheduling_backup = self._assemble_scheduled_operation(
                scheduling_name=scheduling_name_backup,
                is_enabled=is_enabled_backup,
                operation_type='backup')
            scheduled_operations.append(scheduling_backup)
        if is_enabled_copy:
            scheduling_copy = self._assemble_scheduled_operation(
                scheduling_name=scheduling_name_copy,
                is_enabled=is_enabled_copy,
                operation_type='copy')
            scheduled_operations.append(scheduling_copy)
        body_dict = {
            "policy": {
                "name": policy_name,
                "provider_id": provider_id,
                "scheduled_operations": scheduled_operations,
                "resources": resources,
                "parameters": {
                    "common": {
                        "az": self.context.az_id,
                        "expired_at": expired_time
                    },
                    "advanced_params": {
                        "encryption": encryption
                    }
                },
                "resource_type": resource_type
            }
        }
        return self.http_client.post(url, json=body_dict)

    @staticmethod
    def _assemble_scheduled_operation(scheduling_name, is_enabled,
                                      operation_type):
        max_backups = 20
        pattern_freq = "WEEKLY"
        pattern_by_day = "MO"
        pattern_by_hour = "10"
        pattern_by_minute = "20"

        trigger_pattern_str = "BEGIN:VCALENDAR\r\nBEGIN:VEVENT\r\n" \
                              "RRULE:FREQ={freq};BYDAY={by_day};" \
                              "BYHOUR={by_hour};BYMINUTE={by_minute}\r\n" \
                              "END:VEVENT\r\n" \
                              "END:VCALENDAR\r\n"
        trigger_pattern = trigger_pattern_str.format(freq=pattern_freq,
                                                     by_day=pattern_by_day,
                                                     by_hour=pattern_by_hour,
                                                     by_minute=pattern_by_minute)

        result_dict = {
            "name": scheduling_name,
            "description": "",
            "enabled": is_enabled,
            "operation_definition": {
                "max_backups": max_backups
            },
            "trigger": {
                "properties": {
                    "pattern": trigger_pattern
                }
            },
            "operation_type": operation_type
        }
        return result_dict
