import json
import time

import utils.common.log as logger

from plugins.CSBS.scripts.upgrade.ab_test.tasks.tenant import TenantApi
from plugins.AcceptanceTest_Upgrade.basic_fs_test.common.basic_login_serviceom \
    import LoginServiceOm
from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.common.upgrade.karbor import KarborOperation

logger.init("CSBS-VBS")


class ResourceTest(object):
    def __init__(self, context, http_client):
        self.context = context
        self.http_client = http_client
        self.tenant_api = TenantApi(self.context, http_client)
        self.console_host = self.context.basic_mo_test.get('console_host')
        self.project_id = self.context.user_info['project_id']
        self.image_name = self.context.basic_fs_test.get('vm_template_kvm')
        self.image_name_x86 = \
            self.context.basic_fs_test.get('vm_template_kvm_x86')
        self.image_name_arm = \
            self.context.basic_fs_test.get('vm_template_kvm_arm')
        self.flavor_name = self.context.basic_fs_test.get('kvm_flavor_name1')
        self.volume_type_list = \
            self._get_volume_type_list_from_az(self.context.az_id)
        self.project_evs_name = "弹性云硬盘"

        self.engine_id = self.context.engine_id
        self.region_id = self.context.engine_info['region_id']

    def turn_on_price_rate(self):
        """Turn on the price_rate switch on the karbor node to make rate_params

        be added into the request params when Karbor places an order.
        """
        self._set_price_rate_switch()

    def turn_off_price_rate(self):
        """Turn off the price_rate switch on the karbor node to make

        rate_params not be added into the request params when Karbor places an
        order, avoiding acceptance test failure on the condition of resource
        charging being on.
        """
        self._set_price_rate_switch(target="false")

    def _set_price_rate_switch(self, target="true"):
        KarborOperation(self.engine_id).set_price_rate_switch(target)

    def _set_vm_info(self, vol_type=None):
        if self.context.az_info:
            cpu_arch = self.context.az_info.get('cpuArch')
        else:
            raise Exception('Fail to set vm info,Check whether AZ '
                            'of the KVM type exists.')
        return dict(name=constant.VM_NAME, az_id=self.context.az_id,
                    vol_type=vol_type, cpu_arch=cpu_arch)

    def _get_vm_info(self, vm_name):
        logger.info("Start to get vm info,the vm_name:{}.".format(vm_name))
        url = "https://{console_url}/ecm/rest/v1.0/servers?limit=10" \
              "&offset=1&not-tags=__type_baremetal" \
              "&detail=3".format(console_url=self.console_host)
        _, body = self.http_client.get(url=url, verify=False)
        for server_info in body["servers"]:
            if server_info["name"] == vm_name:
                logger.info("Got vm info successfully, vm name is {}, "
                            "id is {}.".format(vm_name, server_info["id"]))
                return server_info
        err_msg = f"Can not find the vm, vm name is [{vm_name}]."
        logger.error(err_msg)
        return {}

    def _get_volume_type_list_from_az(self, az_id):
        simple_vol_type_list = []
        url = 'https://{}/ecm/rest/v1/resource-tags/volume_type?' \
              'availability_zone-any={}&not-hybrid_filter=false' \
            .format(self.console_host, az_id)
        _, body = self.http_client.get(url=url, verify=False)
        for vol_type_info in body['resources']:
            vol_type_name_list = vol_type_info['tags']['name']
            if vol_type_name_list:
                vol_type_name_list.reverse()
                simple_vol_type_list.extend(vol_type_name_list)
        if not simple_vol_type_list:
            raise Exception('Fail to get volume type from az, '
                            'detail info:[{}].'.format(str(body)))
        return simple_vol_type_list

    def _get_image_info_by_name(self, image_name):
        logger.info(
            "Start to get image info by image name: {}.".format(image_name))
        for _v in ["v1", "v2"]:
            url = "https://{}/ecm/rest/{}/ims/combine/images?" \
                  "__support_kvm=true&__imagetype=gold&status=active&" \
                  "sort_key=name&sort_dir=asc".format(self.console_host, _v)
            try:
                _, body = self.http_client.get(url=url, verify=False)
            except Exception as e:
                logger.error(e)
                continue
            for image in body["images"]:
                logger.info("Check image, "
                            "image name is [{}].".format(image["name"]))
                if image_name == image["name"]:
                    logger.info("Find the image, id={}.".format(image["id"]))
                    return image
        raise Exception("Failed to get image info, "
                        "image name is {}.".format(image_name))

    def _get_flavor_info_by_name(self, flavor_name):
        logger.info("Start to get flavor info by flavor name, "
                    "name is {}.".format(flavor_name))
        url = "https://{}/ecm/rest/v1/{}/cloudservers/flavors".format(
            self.console_host, self.project_id)
        _, body = self.http_client.get(url=url, verify=False)
        for flavor in body["flavors"]:
            logger.info("Check flavor, "
                        "flavor name is [{}].".format(flavor["name"]))
            if flavor_name == flavor["name"]:
                logger.info("Find the flavor, id={}.".format(flavor["id"]))
                return flavor
        raise Exception("Failed to get flavor info, "
                        "flavor name is {}.".format(flavor_name))

    def _get_image_name(self, structure_type):
        if self.image_name:
            image_name = self.image_name
        elif structure_type == 'x86':
            image_name = self.image_name_x86
        elif structure_type == 'arm':
            image_name = self.image_name_arm
        else:
            raise Exception("Failed to get image_name.")
        return image_name

    def _get_flavor_name(self, az_id):
        # login service om,create flavor and external network
        obj = LoginServiceOm(self.context.engine_id)
        flavor1_name = obj.flavor1_name + "_" + az_id
        flavor2_name = obj.flavor2_name + "_" + az_id
        flavor1 = obj.query_flavor(flavor1_name)
        if flavor1:
            flavor_name = flavor1_name
        else:
            flavor2 = obj.query_flavor(flavor2_name)
            if flavor2:
                flavor_name = flavor2_name
            else:
                flavor_name = self.flavor_name + "_" + az_id
        return flavor_name

    def _build_nics_data(self, vpc_id):
        subnet_id = self.tenant_api.get_subnet_info_by_name(self.tenant_api.subnet_name, vpc_id)["id"]
        if not subnet_id:
            raise Exception(f"Failed to get subnet info, name is {self.tenant_api.subnet_name}.")
        return [{"subnet_id": subnet_id,
                 "ip_address": "",
                 "ip_address_v6": "",
                 "nictype": "",
                 "physical_network": "",
                 "extra_dhcp_opts": [],
                 "binding:profile": {"disable_security_groups": False,
                                     "availability_zone": ""}
                 }]

    def _get_create_vm_data(self, az_id, vm_name, vol_type, structure_type):
        vpc_info = self.tenant_api.get_vpc_info_by_name(self.tenant_api.vpc_name)
        if not vpc_info:
            raise Exception(f"Failed to get vpc info by name, the name is {self.tenant_api.vpc_name}.")
        vpc_id = vpc_info["id"]

        keypair_info = self.tenant_api.get_keypair_info_by_name(self.tenant_api.keypair_name)
        if not keypair_info:
            raise Exception(f"Failed to get keypair info, name is {self.tenant_api.keypair_name}.")

        image_name = self._get_image_name(structure_type)
        flavor_name = self._get_flavor_name(az_id)
        flavor_id = self._get_flavor_info_by_name(flavor_name)["id"]
        image_id = self._get_image_info_by_name(image_name)["id"]
        security_group_id = self.tenant_api.get_security_group_id()
        product_id = self.tenant_api.get_product_id(service_type=constant.ECS_PRODUCT_TYPE)

        nics = self._build_nics_data(vpc_id)
        params = {"tenantId": self.project_id, "availability_zone": az_id,
                  "name": vm_name, "imageRef": image_id, "flavorRef": flavor_id,
                  "root_volume": {"volumetype": vol_type, "size": 20,
                                  "extendparam": {"resourceSpecCode": "",
                                                  "resourceType": ""}
                                  },
                  "data_volumes": [],
                  "vpcid": vpc_id, "nics": nics,
                  "security_groups": [{"id": security_group_id}],
                  "personality": [],
                  "count": 1,
                  "extendparam": {"chargingMode": 0,
                                  "regionID": self.region_id},
                  "metadata": {"op_svc_userid": "",
                               "__instance_vwatchdog": False,
                               "_ha_policy_type": "remote_rebuild"
                               },
                  "tags": ["__single_storage"],
                  "display": {},
                  "extra": {"devices": [{"device_type": "cdrom"}]}}
        secret_params = {"metadata": {}, "key_name": keypair_info['name']}
        send_data = {"subscriptions": [{"operate_type": "apply",
                                        "project_id": self.project_id,
                                        "product_id": product_id,
                                        "region_id": self.region_id,
                                        "service_type": "ecs", "tenancy": "0",
                                        "params": json.dumps(params),
                                        "time_zone": "GMT+08:00",
                                        "secret_params": json.dumps(secret_params)}]}
        return send_data

    def _operate_cloud_server(self, opt_type, vm_name):
        opt_url = f"https://{self.console_host}/ecm/rest/v1/{self.project_id}/cloudservers/action"
        logger.info(f"{opt_type} cloud server is start, vm name is {vm_name}, url is {opt_url}.")
        vm_state = self._get_vm_info(vm_name)
        vm_id = vm_state['id']
        if opt_type == constant.OPT_TYPE_OFF:
            if vm_state['status'] == "SHUTOFF":
                logger.info("The server status is already SHUTOFF, "
                            "no need stop again.")
                return True
            opt_body = {"os-stop": {
                "type": "HARD",
                "servers": [{"id": vm_id}]
            }
            }
        elif opt_type == constant.OPT_TYPE_ON:
            if vm_state['status'] == "ACTIVE":
                logger.info("The server status is already ACTIVE, no need to start again.")
                return True
            opt_body = {"os-start": {"servers": [{"id": vm_id}]}}
        else:
            raise Exception(f"Failed to operate cloud server, operation type is [{opt_type}].")
        self.http_client.post(url=opt_url, data=json.dumps(opt_body))
        logger.info(f"Send {opt_type} cloud server cmd successfully.")
        return self._check_operate_server_result(vm_name, opt_type)

    def _check_operate_server_result(self, vm_name, opt_type):
        logger.info("Start to check server status after 10s.")
        while True:
            vm_info = self._get_vm_info(vm_name)
            if opt_type == constant.OPT_TYPE_OFF:
                if vm_info['status'] == "SHUTOFF":
                    logger.info(f"{opt_type} cloud server successfully.")
                    return True
                if vm_info['OS-EXT-STS:task_state'] == "powering-off":
                    logger.info("The cloud server is powering-off, retry check after 10s.")
                    continue
                raise Exception(f"Failed to {opt_type} cloud server, the server status is {vm_info['status']}, "
                                f"task state is {vm_info['OS-EXT-STS:task_state']}.")
            if opt_type == constant.OPT_TYPE_ON:
                if vm_info['status'] == "ACTIVE":
                    logger.info(f"{opt_type} cloud server successfully.")
                    return True
                if vm_info['OS-EXT-STS:task_state'] == "powering-on":
                    logger.info("The cloud server is powering-on, retry check after 10s.")
                    continue
                raise Exception(f"Failed to {opt_type} cloud server, server status is {vm_info['status']}, "
                                f"task state is {vm_info['OS-EXT-STS:task_state']}.")
            time.sleep(10)

    def _create_vm(self, vm_list):
        created_vm_list = []
        for vm_info in vm_list:
            _vm_info = self._get_vm_info(vm_info["name"])
            if _vm_info:
                logger.info("The vm already exists, no need to create.")
                if _vm_info["status"].upper() == "SHUTOFF":
                    self._operate_cloud_server(constant.OPT_TYPE_ON,
                                               vm_info["name"])
                return True
            logger.info("Start to create vm, the vm name is {}, "
                        "volume type is {}".format(vm_info["name"],
                                                   vm_info["vol_type"]))
            body_data = self._get_create_vm_data(vm_info["az_id"],
                                                 vm_info["name"],
                                                 vm_info["vol_type"],
                                                 vm_info["cpu_arch"])
            ecs_url = "https://{}/ecm/rest/subscription/v3.0/" \
                      "subscriptions".format(self.console_host)
            _, body = self.http_client.post(url=ecs_url,
                                            data=json.dumps(body_data),
                                            verify=False)
            sub_id = body['purchases'][0]['subscription_id']
            logger.info("Send command successfully, "
                        "order id={}.".format(sub_id))
            created_vm_list.append(vm_info["name"])
            if self.tenant_api.monitor_order_status(sub_id):
                break
        # Monitors the situation where an order is successfully created,
        # but the VM status is abnormal.
        for vm_name in created_vm_list:
            logger.info("Start to check created vm's status.")
            _vm_info = self._get_vm_info(vm_name)
            vm_status = _vm_info["status"].upper()
            logger.info("Current vm name:{}, vm status: {}.".format(vm_name,
                                                                    vm_status))
            if vm_status != "ACTIVE":
                raise Exception('The order status is successful, '
                                'but vm status is {}.'.format(vm_status))
        return True

    def _delete_vm(self, vm_info):
        logger.info("Start to delete vm, "
                    "the vm info is [{}].".format(str(vm_info)))
        params = {"count": 1,
                  "display": [],
                  "ids": [{"id": vm_info["id"]}],
                  "delete_publicip": True,
                  "delete_volume": True,
                  "delete_snapshot": True}
        send_data = {"subscriptions": [
            {"operate_type": "delete",
             "project_id": self.project_id,
             "region_id": self.region_id,
             "service_type": "ecs",
             "secret_params": "",
             "params": json.dumps(params)}]}
        url = "https://{}/ecm/rest/subscription/v3.0/" \
              "subscriptions".format(self.console_host)
        _, body = self.http_client.post(url=url,
                                        data=json.dumps(send_data),
                                        verify=False)
        sub_id = body['purchases'][0]['subscription_id']
        logger.info("Send command successfully, "
                    "order id: {}.".format(sub_id))
        if not self.tenant_api.monitor_order_status(sub_id):
            raise Exception("Delete vm failed, login to ManageOne using "
                            "the test account to view details.")
        # The order is sent successfully, but the VM is not deleted.
        logger.info("Check vm which named {}.".format(vm_info["name"]))
        _vm_info = self._get_vm_info(vm_info["name"])
        if not _vm_info:
            logger.info('The vm:{} is not exist.'.format(vm_info['name']))
        else:
            raise Exception("The order status is 'successed', "
                            "but the cloud server is not deleted.")

    def create_vm(self):
        logger.info('Start to create the test VM.')
        self.tenant_api.create_vpc()
        self.tenant_api.create_keypair(self.tenant_api.keypair_name)

        vm_info = self._get_vm_info(constant.VM_NAME)
        if vm_info:
            if vm_info["status"].upper() != 'ACTIVE':
                logger.info('The VM already exists, but status is '
                            'not ACTIVE,begin to delete it.')
                self._delete_vm(vm_info)
            else:
                logger.info('The VM already exists, status is ACTIVE,'
                            ' no need to create.')
                return True
        vm_info_list = []
        for vol_type_name in self.volume_type_list:
            vm1_info = self._set_vm_info(vol_type=vol_type_name)
            logger.info('Current VM info:{}.'.format(vm1_info))
            vm_info_list.append(vm1_info)
        try:
            self._create_vm(vm_info_list)
        except Exception as e:
            raise Exception('Failed to create VM, errMsg:{}.'.format(str(e))) from e
        logger.info('Succeed to create the test VM.')
        return True

    def delete_vm(self):
        logger.info('Start to delete the test VM.')
        vm_info = self._get_vm_info(constant.VM_NAME)
        if vm_info:
            try:
                self._delete_vm(vm_info)
            except Exception as e:
                err_msg = f"Failed to delete test VM, errMsg:{str(e)}."
                logger.error(err_msg)
                raise Exception(err_msg) from e
        else:
            logger.info('The vm is not exist.')
        logger.info('Succeed to delete the test VM.')
        self.tenant_api.delete_keypair(self.tenant_api.keypair_name)
        self.tenant_api.delete_vpc()
        return True

    def _cloud_service_subscription_evs(self, innerparams, operate_type,
                                        service_type="evs", tenancy="0"):
        url = "https://{}/ecm/rest/subscription/v3.0/" \
              "subscriptions".format(self.console_host)
        params_data = {
            "subscriptions": [{
                "project_id": self.project_id,
                "region_id": self.region_id,
                "product_id": "",
                "comments": "",
                "operate_type": operate_type,
                "service_type": service_type,
                "time_zone": "GMT+08:00",
                "params": json.dumps(innerparams).replace("\"null\"", "null")
            }]
        }
        project_id = self.tenant_api.get_product_id(service_type=constant.EVS_PRODUCT_TYPE)
        params_data["subscriptions"][0]["product_id"] = project_id
        params_data["subscriptions"][0]["tenancy"] = tenancy
        params = json.dumps(params_data)
        _, body = self.http_client.post(url=url, data=params)
        logger.info("Subscription type:{}.".format(operate_type))
        logger.debug(params)
        return body

    def _check_order_status(self, order_id, expect_status, time_out=1000):
        status_no_list = ['partialSuccessed', 'failed', 'closed', 'timeout']
        sleep_time = 10
        count = time_out // sleep_time
        logger.info("Cycle to check order's status.")
        cur_status = ""
        for _ in range(count):
            url = "https://{host}/motenantconsolehomewebsite/goku/rest/" \
                  "order/v3.0/orders/{order_id}".format(host=self.console_host,
                                                        order_id=order_id)
            _, body = self.http_client.get(url)
            logger.info("Search order details succeed.")
            cur_status = body["status"]
            logger.info('Current order status:{}.'.format(cur_status))
            if cur_status == expect_status:
                break
            elif cur_status in status_no_list:
                break
            time.sleep(sleep_time)
        if cur_status != expect_status:
            logger.warn('Order status expected:{}, '
                        'actual status:{}.'.format(expect_status, cur_status))
        return cur_status

    def _get_order_resource(self, order_id):
        logger.info("Start to get a list of order resources.")
        url = "https://{host}/motenantconsolehomewebsite/goku/rest/order/" \
              "v3.0/orders/{order_id}/resources".format(host=self.console_host,
                                                        order_id=order_id)
        _, body = self.http_client.get(url)
        logger.info("Get successfully")
        list_len = len(body)
        resource_ids_list = []
        for i in range(list_len):
            resource_ids_list.append(body[i]["resource_id"])
        return resource_ids_list

    def _get_volumes_by_volname(self, vol_name):
        url = "https://{host}/ecm/rest/v2/detail/{project_id}/volumes/" \
              "detail?limit=100&offset=0&" \
              "name={vol_name}".format(host=self.console_host,
                                       project_id=self.project_id,
                                       vol_name=vol_name)
        _, volumes_info = self.http_client.get(url)
        return volumes_info

    def _get_volume_info_by_volume_id(self, volume_id):
        url = "https://{host}/ecm/rest/v2/get/{project_id}/" \
              "volumes/{vol_id}".format(host=self.console_host,
                                        project_id=self.project_id,
                                        vol_id=volume_id)
        _, volume_info = self.http_client.get(url)
        return volume_info

    def _wait_volume_until_status(self, volume_id, status="available"):
        count = 0
        while True:
            vol_info = self._get_volume_info_by_volume_id(volume_id)
            volume_status = vol_info["status"]
            if volume_status == status:
                break
            time.sleep(10)
            count = count + 1
            if count > 20:
                raise Exception("Time out to get volume's status, "
                                "please check.")
            if volume_status == "error" or volume_status == "error_deleting":
                raise Exception("The status of the volume is error,"
                                'volume_id:{}.'.format(volume_id))
        logger.info("The volume's state is {}，"
                    "volume_id is {}.".format(status, volume_id))

    def _delete_volume_by_id(self, volume_id):
        logger.info('Start to delete volume,volume_id:{}.'.format(volume_id))
        url = "https://{}/ecm/rest/subscription/v3.0/" \
              "subscriptions".format(self.console_host)
        params = {"action": "delete_volume",
                  "ids": [{
                      "id": volume_id,
                      "service_type": "evs",
                      "name": constant.EVS_NAME
                  }],
                  "region_id": self.region_id,
                  "display": {}
                  }
        data = {
            "subscriptions": [
                {
                    "project_id": self.project_id,
                    "region_id": self.region_id,
                    "operate_type": "delete",
                    "service_type": "evs",
                    "params": json.dumps(params)
                }
            ]
        }
        _, body = self.http_client.post(url=url, data=json.dumps(data))
        logger.info('Succeed to send the order of delete volume,'
                    'the volume_id:{}.'.format(volume_id))
        order_id = body["purchases"][0]["subscription_id"]
        logger.info("Wait for the order's status displayed entirely.")
        self._check_order_status(order_id, "successed")
        logger.info("Get the list of order's resource.")
        resource_ids_list = self._get_order_resource(order_id)
        volume_id = resource_ids_list[0]
        try:
            self._wait_volume_until_status(volume_id, status="deleting")
        except Exception as e:
            logger.warn("Check status error, error msg:{}.".format(str(e)))
            if "itemNotFound" not in str(e):
                raise Exception(str(e)) from e
        logger.info("The EVS disk is deleted successfully, "
                    "volume id is:{}.".format(volume_id))

    def _create_evs(self):
        logger.info('Start to create the test volume.')
        order_status = 'failed'
        order_id = None
        for volume_type in self.volume_type_list:
            logger.info("Build creating cloud volume parameter body, "
                        "current volume's type:{}.".format(volume_type))
            # build evs order params
            params = {"action": "create_volume",
                      "count": 1,
                      "region_id": self.region_id,
                      "volumes": [{
                          "backup_id": "null", "count": 1,
                          "size": int(constant.EVS_SIZE),
                          "availability_zone": self.context.az_id,
                          "name": constant.EVS_NAME,
                          "description": "", "snapshot_id": "null",
                          "multiattach": "false", "volume_type": volume_type,
                          "imageRef": "null",
                          "metadata": {"hw:passthrough": "false"},
                          "OS-SCH-HNT:scheduler_hints": {},
                          "tags": [{"key": "", "value": ""},
                                   {"key": "", "value": ""}]
                      }]
                      }
            logger.info("Call to create a cloud volume by order interface.")
            result = self._cloud_service_subscription_evs(params, "apply")

            logger.info("Order was created successfully.")
            order_id = result["purchases"][0]["subscription_id"]
            logger.info(
                "Waiting for the order status to be displayed entirely.")
            order_status = self._check_order_status(order_id, "successed")
            if order_status == 'successed':
                logger.info('The order status is succeed.')
                break
            else:
                logger.error('The order status is {}, use next volume type '
                             'to create volume.'.format(order_status))
                continue
        if order_status != 'successed':
            raise Exception('Failed to create volume with all volume types.')
        logger.info("Get a list of order resource's ids.")
        resource_ids_list = self._get_order_resource(order_id)
        volume_id = resource_ids_list[0]
        self._wait_volume_until_status(volume_id=volume_id)
        logger.info("The volume has been created successfully.")

    def create_evs(self):
        test_volume_info = self._get_volumes_by_volname(constant.EVS_NAME)
        if test_volume_info["volumes"] and \
                test_volume_info["volumes"][0]["status"] == "available":
            logger.info("The EVS disk already exists and "
                        "does not need to be created.")
            return True
        else:
            self._create_evs()
        return True

    def delete_evs(self):
        test_volume_info = self._get_volumes_by_volname(constant.EVS_NAME)
        for test_volume in test_volume_info["volumes"]:
            self._delete_volume_by_id(test_volume["id"])
        logger.info("Succeed to delete the volume.")
