import re

import requests
import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.message import Message
from utils.common.check_result import CheckResult

from plugins.CSBS.common.ebackup_util import EBackupUtil
from plugins.CSBS.common.util import check_ip
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("CSBS-VBS")


class PreCheckWorkflow(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super().__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.ebackup_util = EBackupUtil(project_id, pod_id, regionid_list)
        self.param_util = ParamUtil()

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self.precheck_workflow()
        except Exception as exception:
            logger.error(f"Execute error:{exception}.")
            return Message(500, f"Execute error:{exception}.")

    def precheck_workflow(self):
        ebackup_workflow_user = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS',
                                                                          'karbor_eBackup_admin')
        if not ebackup_workflow_user:
            raise Exception("Get eBackup workflow user admin failed.")
        ebackup_workflow_user_pwd = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS',
                                                                              'karbor_eBackup_admin_pwd')
        if not ebackup_workflow_user_pwd:
            raise Exception("Get eBackup workflow user password failed.")

        workflow_ip_list = self.ebackup_util.get_workflow_ip_list()
        workflow_node_ip = None
        for workflow_ip in workflow_ip_list:
            workflow_login_url = f"https://{workflow_ip}:8088"
            logger.info(f"workflow_login_url is:{workflow_login_url}")
            try:
                rsp = requests.post(workflow_login_url, headers={'Accept': 'application/json'}, verify=False, timeout=5)
            except Exception as exception:
                logger.info(f"Request {workflow_login_url} failed:{exception}.")
                continue
            redirect_history_list = rsp.history
            if not redirect_history_list:
                continue
            redirect_url = redirect_history_list[len(redirect_history_list) - 1].headers['Location']
            workflow_node_ip = re.findall(r'[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+', redirect_url)[0]
            if workflow_node_ip:
                break
        if not check_ip(workflow_node_ip):
            logger.error("Unable to find the workflow_node_ip in workflow_ip_list that can log in workflow.")
            raise Exception("Unable to find the workflow_node_ip in workflow_ip_list that can log in workflow.")
        logger.info(f"workflow_node_ip is:{workflow_node_ip}.")

        headers = self.login(workflow_node_ip, ebackup_workflow_user, ebackup_workflow_user_pwd)
        if not headers:
            raise Exception("Login workflow check failed.")
        self.check_permit_for_admin(headers, workflow_node_ip)
        self.logout(workflow_node_ip, headers)
        logger.info('Login workflow check success.')
        result = CheckResult(status="success")
        return Message(200, check_results=[result])

    @staticmethod
    def check_permit_for_admin(headers, float_ip):
        logger.info("begin to check permit of admin account.")
        check_url = f'https://{float_ip}:8088/rest/dev/vbackup_server'
        rsp = requests.get(check_url, headers=headers, verify=False, timeout=5)
        if rsp.status_code != 200:
            logger.error(f"Request {check_url} return not 200.")
        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            raise Exception("Errcode is empty.")
        if errcode == 1610664355:
            raise Exception("Account admin is abnormal.")
        elif errcode != 0:
            description = rsp.json()["error"]["description"]
            raise Exception(f"Login {float_ip} failed,description:{description}.")
        else:
            logger.info("Check admin account succ.")

    def login(self, float_ip, user, pwd):
        logger.info("Begin to login ebackup.")
        login_url = f'https://{float_ip}:8088/rest/dev/login'
        headers = {'Accept': 'application/json'}
        data = f'{{"scope":0,"username":"{user}","password":"{pwd}"}}'
        rsp = None
        try:
            rsp = requests.post(login_url, headers=headers, data=data, verify=False, timeout=5)
        except Exception as exception:
            logger.error("Failed to login eBackup(%s) with admin, reason: %s." % (float_ip, str(exception)))
        if not rsp:
            return {}
        if rsp.status_code != 200:
            logger.error(f"Request {login_url} return not 200.")
        errcode = rsp.json()["error"]["code"]
        if errcode == 0:
            logger.info("Login ebackup success.")
            token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            session_content = cookie.split(';')[0]
            headers = {
                'Accept': 'application/json;version=2.2;charset=UTF-8',
                'iBaseToken': token,
                'Cookie': f'language=en;{session_content};DEVICE_ID=dev;'
                          f'sessionIdleTime=60000; MACHINEROLE=0;CSRF_IBASE_TOKEN={token}'
            }
            return headers
        else:
            description = rsp.json()["error"]["description"]
            ip_dec = 'The IP address is %s ;' % float_ip
            description = ip_dec + description
            logger.error("Login ebackup failed,description:" + description)
            return {}

    @staticmethod
    def logout(float_ip, headers):
        url = f"https://{float_ip}:8088/rest/dev/sessions"
        requests.delete(url, headers=headers, verify=False, timeout=5)
        logger.info("Logout success.")
