import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.constant import CSBS_SERVICE_NAME
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob


logger.init("CSBS-VBS")


class RegisterCMDB(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(RegisterCMDB, self).__init__(project_id, pod_id, regionid_list)
        self.service_name = CSBS_SERVICE_NAME
        self.cipher_type = "CipherType"
        self.cmdb_util = ManageOneCmdbUtil(project_id)
        self.karbor_operation = KarborOperation(self.project_id)
        self.param_tool = ParamsTools(project_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self._update_csbs_to_cmdb()
        except Exception as e:
            logger.error("Failed executing register CMDB, "
                         "errmsg: {}.".format(e))
            return Message(500, HCCIException(645022))
        self._delete_console_cmdb()
        logger.info("The backup service information has been register succeed on CMDB.")
        return Message(200)

    def _update_csbs_to_cmdb(self):
        # 从CMDB获取注册信息
        cmdb_service_name = f"{self.service_name}_{self.region_id}"
        logger.info(f"Start to obtain {self.service_name} service "
                    f"info from CMDB.")
        service_info_list = self.cmdb_util.get_cloud_service_info(
            region_code=self.region_id,
            index_name=self.service_name,
            name=cmdb_service_name)
        if not service_info_list:
            raise Exception(f"Failed to obtain {self.service_name} info.")
        logger.info(f"The cloud service information obtained "
                    f"from the CMDB is: {service_info_list}.")

        # 组织注册数据
        karbor_version = self.param_tool.get_karbor_version_from_conf()
        cloud_service_info = dict()
        cloud_service_info["version"] = karbor_version.target_version
        cloud_service_info["originalVersion"] = karbor_version.original_version
        cloud_service_info["name"] = cmdb_service_name
        for service_info in service_info_list:
            if not service_info.get("name") == cmdb_service_name:
                continue
            cloud_service_info["indexName"] = service_info.get("indexName")
            cloud_service_info["name"] = service_info.get("name")
            extend_info_list = []
            for extend_info in service_info.get("extendInfos"):
                if extend_info.get("key") == self.cipher_type:
                    continue
                extend_info_list.append(extend_info)
            karbor_cipher = self.karbor_operation.get_cipher_type()
            extend_info_list.append(
                {
                    "key": self.cipher_type,
                    "value": karbor_cipher
                }
            )
            cloud_service_info["extendInfos"] = extend_info_list

        if not cloud_service_info:
            raise Exception("Failed to construct CMDB data.")

        # 注册CMDB
        logger.info(f"Start to register cloud service info to CMDB, "
                    f"cloud service info is: {cloud_service_info}.")
        self.cmdb_util.set_cloud_service_info(
            region_code=self.region_id,
            cloud_service_info=cloud_service_info)
        logger.info(f"Succeed to register {self.service_name} cmdb.")

    def _delete_console_cmdb(self):
        # 从CMDB获取注册信息
        console_index = "CSBS-VBS-Console"
        logger.info(f"Start to obtain {console_index} service info from CMDB.")
        service_info_list = self.cmdb_util.get_cloud_service_info(
            region_code=self.region_id,
            index_name=console_index)
        if not service_info_list:
            logger.info("The console CMDB is deleted.")
            return
        logger.info(f"The cloud service information obtained "
                    f"from the CMDB is: {service_info_list}.")

        console_version = ""
        for service_info in service_info_list:
            if service_info.get("name") != f"{console_index}_{self.region_id}":
                continue
            console_version = service_info.get("version")
        if not console_version:
            raise Exception(f"Failed to obtain console version.")

        logger.info(f"Start to delete {console_index} cmdb.")
        self.cmdb_util.remove_cloud_service_info(
            self.region_id, console_index, console_version)
        logger.info(f"Succeed to delete {console_index} cmdb.")
