import json
import time

import requests
import utils.common.log as logger
from utils.common.message import Message

from plugins.CSBS.common.iam_util import IamUtil
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade import constant
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("CSBS")


DJMANAGER_DESCRIPTION = {
    "zh-cn": "考虑到对系统安全的影响，在登录节点时，不能使用root用户直接登录，"
             "需要由djmanager用户登录后通过执行命令 \"sudo su - root\"切换为root用户",
    "en-us": "To ensure system security, you cannot log in to the node as "
             "user root. You need to log in to the node as user djmanager "
             "and run the \"sudo su - root\" command to switch to user root"}
ROOT_DESCRIPTION = {
    "zh-cn": "使用该帐户登录karbor服务器进行运维操作，查看节点配置，"
             "查看日志，巡检和信息收集等。",
    "en-us": "Used to log in to an karbor server to perform O&M operations, "
             "view node configurations and logs, perform inspection, "
             "and collect information."}
DJMANAGER_RISK_MESSAGE = {"zh-cn": "djmanager", "en-us": "djmanager"}
ROOT_RISK_MESSAGE = {"zh-cn": "root", "en-us": "root"}


class RegisterCsbsAcountUnifyPasswd(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(RegisterCsbsAcountUnifyPasswd, self).__init__(project_id, pod_id, regionid_list)
        self.service_name = "CSBS-VBS"
        self.param_tool = ParamsTools(self.project_id)
        self.karbor_user_info = self.param_tool.get_karbor_user_info()
        self.iam_util = IamUtil(self.project_id, self.pod_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Start to build the data of register the unified password of the csbs account.")
        account_data = self._build_csbs_account_unify_password_data()
        logger.info("Data is successfully organized and then start to register unified password to oc.")
        # 更新注册账号
        try:
            result = self._send_unify_pwd_req(account_data)
        except Exception as err:
            logger.error(f"Failed to register the IAM account with the unified password platform, err_msg:{str(err)}")
            return Message(500, f"注册IAM账号至统一密码平台失败，错误信息：{str(err)}",
                           f"Failed to register the IAM account with the "
                           f"unified password platform, err_msg:{str(err)}.")
        if not result:
            logger.error("Failed to register the IAM account with the unified password platform.")
            return Message(500, "Failed to register the IAM account with the unified password platform.")
        logger.info("Succeed to register the csbs account with the unified password platform.")
        return Message(200)

    def _build_account(self, username, password, node):
        passwd_change_max_between = 90
        expire_time_params = self._get_last_passwd_change_and_passwd_expires(node, username)
        last_passwd_change = expire_time_params.get("last_passwd_change")
        passwd_complexity = dict()
        if username == "root":
            passwd_expires = -1
            account_description = json.dumps(ROOT_DESCRIPTION)
            risk_message = json.dumps(ROOT_RISK_MESSAGE)
            passwd_complexity = {"difok": 3, "minlen": 8}
        else:
            passwd_expires = expire_time_params.get("passwd_expires")
            account_description = json.dumps(DJMANAGER_DESCRIPTION)
            risk_message = json.dumps(DJMANAGER_RISK_MESSAGE)
            passwd_complexity["passwdChangeMaxBetweenTime"] = passwd_change_max_between
        if passwd_expires == 'never':
            passwd_expires = -1
        account = {
            "accountName": username,
            "modifyType": 1,
            "region": self.region_id,
            "ip": node.ip,
            "accountType": 1,
            "usedScene": "karbor",
            "accountDescription": account_description,
            "passwd": password,
            "oldPasswd": password,
            "lastPasswdChange": last_passwd_change,
            "passwdExpires": passwd_expires,
            "riskMessage": risk_message,
            "passwdComplexity": passwd_complexity,
            "urlModify": "/v1.5/unify_passwd/modify",
            "urlValid": "/v1.5/unify_passwd/check",
            "urlSyncStatus": "/v1.5/unify_passwd/tasks",
            "operationType": 0
        }
        return account

    def _get_last_passwd_change_and_passwd_expires(self, node, username):
        try:
            return self.__get_result_dict(node, username)
        except Exception as e:
            logger.error("Failed to get password expired time, err_msg: {}.".format(str(e)))
            raise Exception("Failed to get password expired time") from e

    def __get_result_dict(self, node, username):
        karbor_client = SshClient()
        ssh_client = karbor_client.get_ssh_client(node)
        chage_cmd = "chage -l {} ".format(username)
        awk_cmd = "awk -F ': ' {'print $2'}"
        cmds = {
            "last_passwd_change": '|'.join([chage_cmd, "grep 'Last password change'", awk_cmd]),
            "passwd_expires": '|'.join([chage_cmd, "grep 'Password expires'", awk_cmd])
        }
        result_dict = dict()
        for key, cmd in cmds.items():
            ret = karbor_client.ssh_exec_command_return(ssh_client, cmd)
            if not karbor_client.is_ssh_cmd_executed(ret):
                logger.error("Failed to execute cmd: {}.".format(cmd))
            time_expire_str = ret[0]
            if time_expire_str == "never":
                result_dict[key] = -1
            else:
                time_expire = time.mktime(time.strptime(time_expire_str, '%b %d, %Y'))
                result_dict[key] = int(round(time_expire * 1000))
        karbor_client.ssh_close(ssh_client)
        return result_dict

    def _build_csbs_account_unify_password_data(self):
        node_list = self.param_tool.get_karbor_node_list()
        account_list = list()

        for node in node_list:
            djmanager_account = self._build_account(
                self.karbor_user_info.user_name, self.karbor_user_info.user_pwd, node)
            root_account = self._build_account(
                self.karbor_user_info.root_name, self.karbor_user_info.root_pwd, node)
            account_list.append(djmanager_account)
            account_list.append(root_account)

        register_info = {
            "componentName": self.service_name,
            "subComponents": [{"subComponentName": self.service_name,
                               "createdAccountList": account_list}]
        }
        return register_info

    def _send_unify_pwd_req(self, account_data):
        iam_token = self._get_iam_token()
        oc_url = self.param_tool.get_oc_url(self.region_id)
        register_url = f"{oc_url}{constant.REGISTER_ACCOUNT_API}"
        header = {
            "Content-Type": "application/json",
            "X-Auth-Token": iam_token
        }
        register_retry_times = 0
        logger.info(f"Register unify password, register url:{register_url}.")
        resp = None
        status_code = 600
        retry_time_limit = 5
        while True:
            try:
                resp = requests.request("POST", register_url, data=json.dumps(account_data), headers=header,
                                        verify=False)
            except Exception as err:
                logger.warn(f"Failed to register password to MO, err_msg:{err}, "
                            "start to register unify password by float ip.")
                oc_float_url = self.param_tool.get_oc_url(self.region_id, is_oc_float_ip=True)
                register_url = f"{oc_float_url}{constant.REGISTER_ACCOUNT_API}"
                logger.info(f"Use oc float ip, Register url is {register_url}")
                try:
                    resp = requests.request("POST", register_url, data=json.dumps(account_data), headers=header,
                                            verify=False)
                except Exception as e:
                    logger.error(f"Failed to register password to MO unify password platform, err_msg:{e}.")
            if resp:
                status_code = resp.status_code
                logger.info(f"Register unify password result code:{status_code}.")
            if status_code <= 300 and resp.content:
                rsp_content = json.loads(resp.content)
                if rsp_content.get("code") == "00000000":
                    logger.info("Succeed to register password to MO unify password platform.")
                    return True

            logger.error("Failed to register password to MO unify password platform, Waiting for retry.")
            register_retry_times += 1
            if register_retry_times > retry_time_limit:
                logger.error(f"Maximum number of registrations reached, retry times:{retry_time_limit}, "
                             "so do not try again")
                return False
            time.sleep(30)

    def _get_iam_token(self):
        """Get token from IAM"""
        op_service_account = self.iam_util.get_iam_account_info()
        try:
            return self.iam_util.get_iam_token(op_service_account)
        except Exception as e:
            logger.error(f"Failed to use the current account for authentication, "
                         f"account name:{op_service_account.account_name}, err_msg:{str(e)}")
            raise Exception(f"Failed to use the current account for authentication, "
                            f"account name:{op_service_account.account_name}, err_msg:{str(e)}") from str(e)
