import json
import time

import requests

import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.message import Message

from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob


logger.init("CSBS-VBS")

REQUEST_PORT = "26335"
IAM_DOMAIN = "iam-cache-proxy"
AUTH_API = "/v3/auth/tokens"
REGISTER_ACCOUNT_API = "/rest/mounpwdservice/v1/account"
RETRY_TIME_LIMIT = 5


class RegisterIAMUnifyPasswd(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(RegisterIAMUnifyPasswd, self).__init__(project_id, pod_id,
                                                     regionid_list)
        self.service_name = "CSBS-VBS"
        self.param_util = ParamUtil()
        self.param_tool = ParamsTools(self.project_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Start to build the data of register the unified password of the IAM account.")
        account_data = self._build_unify_password_info()
        logger.info("Data is successfully organized and then start to register unified password to oc.")
        try:
            result = self._send_unify_pwd_req(account_data)
        except Exception as err:
            logger.error(f"Execute error:{err}")
            return Message(500, f"注册IAM账号至统一密码平台失败，"
                                f"错误信息：{str(err)}",
                           f"Failed to register the IAM account with the "
                           f"unified password platform, err_msg:{str(err)}.")
        if not result:
            logger.error("Failed to register the IAM account with "
                         "the unified password platform.")
            return Message(500, "Failed to register the IAM account with "
                                "the unified password platform.")
        logger.info("Succeed to register the IAM account with "
                    "the unified password platform.")
        return Message(200)

    def _send_unify_pwd_req(self, data):
        iam_token = self._get_iam_token()
        global_domain_name = self._get_global_domain()
        oc_url = f"oc.{self.region_id}.{global_domain_name}"
        register_url = f"https://{oc_url}:{REQUEST_PORT}{REGISTER_ACCOUNT_API}"
        header = {
            "Content-Type": "application/json",
            "X-Auth-Token": iam_token
        }
        register_retry_times = 0
        logger.info(f"Register unify password, register url:{register_url}.")
        resp = None
        status_code = 600
        while True:
            try:
                resp = requests.request("POST", register_url, data=json.dumps(data), headers=header, verify=False)
            except Exception as err:
                logger.warn("Failed to register password to MO, "
                            f"err_msg:{err}, "
                            "start to register unify password by float ip.")
                oc_float_ip = self.param_tool.get_oc_float_ip(self.pod_id)
                register_url = f"https://{oc_float_ip}:{REQUEST_PORT}" \
                               f"{REGISTER_ACCOUNT_API}"
                try:
                    resp = requests.request("POST", register_url, data=json.dumps(data),
                                            headers=header, verify=False)
                except Exception as e:
                    logger.error("Failed to register password to MO unify "
                                 f"password platform, err_msg:{e}.")
            if resp:
                status_code = resp.status_code
                logger.info(
                    f"Register unify password result code:{status_code}.")
            if status_code <= 300 and resp.content:
                rsp_content = json.loads(resp.content)
                if rsp_content.get("code") == "00000000":
                    logger.info("Succeed to register password to MO unify "
                                f"password platform.")
                    return True

            logger.error("Failed to register password to MO unify password"
                         f" platform, Waiting for retry.")
            register_retry_times += 1
            if register_retry_times > RETRY_TIME_LIMIT:
                logger.error("Maximum number of registrations reached, "
                             f"retry times:{RETRY_TIME_LIMIT}, "
                             f"so do not try again")
                return False
            time.sleep(30)

    def _get_iam_token(self):
        """Get token from IAM

        """
        iam_user_name, iam_user_pwd = self._get_iam_op_service_account_info()
        global_domain_name = self._get_global_domain()
        iam_url = f"https://{IAM_DOMAIN}.{self.region_id}." \
                  f"{global_domain_name}:{REQUEST_PORT}{AUTH_API}"
        iam_data = {
            "auth": {
                "identity": {
                    "methods": ["password"],
                    "password": {
                        "user": {
                            "domain": {"name": "op_service"},
                            "name": iam_user_name,
                            "password": iam_user_pwd}}},
                "scope": {
                    "domain": {"name": "op_service"}}}}
        header = {'Content-Type': 'application/json'}
        logger.info("Start to get token by iam account.")
        try:
            resp = requests.request("POST", iam_url, headers=header,
                                    data=json.dumps(iam_data), verify=False)
        except Exception as err:
            logger.error(f"Failed to get token, err_msg: {err}.")
            oc_float_ip = self.param_tool.get_oc_float_ip(self.pod_id)
            iam_url = f"https://{oc_float_ip}:{REQUEST_PORT}{AUTH_API}"
            resp = requests.request("POST", iam_url, headers=header,
                                    data=json.dumps(iam_data), verify=False)
        code = resp.status_code
        if code in [200, 201]:
            return resp.headers.get('X-Subject-Token')
        else:
            raise Exception(f"Status code of the request for obtaining "
                            f"the token:{code}.")

    def _build_unify_password_info(self):
        """Construct information that needs to be registered to mo.

        :return:
        """
        account_data_list = self._build_iam_account_unify_password_data()
        register_info = {
            "componentName": "CSBS-VBS",
            "subComponents": [{"subComponentName": "CSBS-VBS",
                               "createdAccountList": account_data_list}]
        }
        return register_info

    def _build_iam_account_unify_password_data(self):
        def _build_data(username, pwd, _ip=None):
            return {
                "accountName": username,
                "passwd": pwd,
                "oldPasswd": pwd,
                "ip": _ip,
                "modifyType": 3,
                "region": self.region_id,
                "accountType": 5,
                "accountDescription":
                    "{\"zh-cn\": \"IAM鉴权帐号\","
                    "\"en-us\": \"IAM authentication account\"}",
                "passwdExpires": -1,
                "riskMessage":
                    "{\"zh-cn\": \"非必要场景不建议修改此密码\","
                    "\"en-us\": \"You are advised not to modify the "
                    "password when it is unnecessary.\"}",
                "passwdComplexity": {"difok": 3, "minlen": 8},
                "urlValid": "/v1.5/unify_passwd/check",
                "operationType": 0
            }

        karbor_float_ip = self.param_tool.get_karbor_float_ip()
        op_service_account_name, op_service_account_pwd = \
            self._get_iam_op_service_account_info()
        op_service_account_data = _build_data(op_service_account_name,
                                              op_service_account_pwd,
                                              karbor_float_ip)
        op_scv_account_name, op_scv_account_pwd = \
            self._get_iam_op_scv_account_info()
        op_scv_account_data = _build_data(op_scv_account_name,
                                          op_scv_account_pwd,
                                          karbor_float_ip)
        return [op_service_account_data, op_scv_account_data]

    def _get_iam_op_scv_account_info(self):
        op_svc_account = self.params_store.db_params_dict.get("iam_op_svc_csbs_account")
        if not op_svc_account:
            op_svc_account = "op_svc_csbs"
        op_svc_account_pwd = self.params_store.db_params_dict.get("iam_op_svc_csbs_pwd")
        if not op_svc_account_pwd:
            raise Exception(f"Failed to obtain the IAM {op_svc_account} pwd.")
        return op_svc_account, op_svc_account_pwd

    def _get_iam_op_service_account_info(self):
        account_name = self.params_store.db_params_dict.get("iam_karbor_admin")
        if not account_name:
            account_name = "karbor"
        account_pwd = self.params_store.db_params_dict.get("iam_karbor_admin_pwd")
        if not account_pwd:
            raise Exception(f"Failed to obtain the IAM {account_name} pwd.")
        return account_name, account_pwd

    def _get_global_domain(self):
        mo_public_params = self.param_util.get_service_cloud_param(
            self.project_id, "ManageOne_public_params", self.region_id)
        global_domain_name = mo_public_params.get(
            "ManageOne_global_domain_name")
        if not global_domain_name:
            global_domain_name = self.param_util.get_value_from_cloud_param(
                self.project_id, "ManageOne",
                "ManageOne_global_domain_name", self.region_id)
        return global_domain_name
