import time

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.dmk_client import DMKClient
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("CSBS-VBS")


class UpgradeKarbor(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(UpgradeKarbor, self).__init__(project_id, pod_id, regionid_list)
        self.karbor_operation = KarborOperation(self.project_id)
        self.dmk_client = DMKClient(self.params_store)
        self.ssh_client = SshClient()

    def execute(self, project_id, pod_id, regionid_list=None):
        logger.info("Login dmk and call DMK to upgrade Karbor.")
        action = "[Upgrade] CSBS-VBS Karbor"
        try:
            result = self.dmk_client.dmk_deploy_karbor(action, self.upgrade_path.karbor_pkg_version)
        except Exception as e:
            logger.error(f"Failed to upgrade Karbor, reason:{str(e)}.")
            return Message(500, HCCIException(645011))
        if not result:
            logger.error("Failed to upgrade Karbor.")
            return Message(500, HCCIException(645011))
        logger.info("Upgrade Karbor successfully.")
        # stop karbor operation service
        self.karbor_operation.change_operation_service(tar_status="stop")
        self.karbor_operation.set_max_sync_time_interval_allowed(60)
        # 只在升级到830版本时对karbor3节点的sys.ini进行修改
        if len(self.karbor_operation.karbor_node_list) == 3 and not self.project_scene.is_csha_scene:
            self._update_karbor3_sys_ini()
        return Message(200)

    def _update_karbor3_sys_ini(self):
        tar_ips = f"{self.karbor_operation.karbor_node_list[0].ip},{self.karbor_operation.karbor_node_list[1].ip}"
        logger.info(f"Start to update karbor3 sys.ini, and conf ips are: {tar_ips}.")
        sys_file = "/opt/huawei/dj/cfg/sys.ini"
        k3_node = self.karbor_operation.karbor_node_list[-1]
        k3_client = self.ssh_client.get_ssh_client(k3_node)
        cmds = [
            f"sed -i '/gaussdb_nodes/d' {sys_file}",
            f"sed -i '/omm-ha_nodes/d' {sys_file}",
            f"sed -i '/cps-monitor_nodes/a gaussdb_nodes = {tar_ips}' {sys_file}",
            f"sed -i '/cps-monitor_nodes/a omm-ha_nodes = {tar_ips}' {sys_file}"
        ]
        for cmd in cmds:
            result = self.ssh_client.ssh_exec_command_return(k3_client, cmd, timeout=30)
            logger.info(f"The exec result: {result}.")
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.warning(f"Failed to execute cmd: {cmd}.")
                time.sleep(5)
                self.ssh_client.ssh_exec_command_return(k3_client, cmd, timeout=30)
                if not self.ssh_client.is_ssh_cmd_executed(result):
                    raise Exception(f"Failed to execute [{cmd}] on {k3_node.ip} node.")
        if k3_client:
            self.ssh_client.ssh_close(k3_client)
        logger.info("Succeeded to update karbor3 sys.ini.")
