# -coding:utf-8-
import configparser
import logging
import os
import re
import stat
import sys
import time
from logging.handlers import WatchedFileHandler

import netifaces
from safety_lib import safe_command

SERVICE_LIST = {
    "base": {},
    "digital_certificate": {},
    "cps-monitor": {},
    "omm-ha": {"hostPorts": "25555,26666,61806", "nodes": "1,2"},
    "gaussdb": {"hostPorts": "5432,12211", "nodes": "1,2"},
    "zookeeper": {"hostPorts": "2888,3888,9888"},
    "cms": {"hostPorts": "28098"},
    "scagent": {"hostPorts": "28688"},
    "rabbitmq": {"hostPorts": "5671,4369,4370,4371"},
    "haproxy": {"hostPorts": "8799,6001,26001"},
    "cli-client": {},
    "inspect": {},
    "backup": {},
    "alarm": {"hostPorts": "28099"},
    "karbor": {"hostPorts": "28799"},
    "resource_manager": {"hostPorts": "28899"},
}


class ServiceManager(object):
    def __init__(self):
        self.root_path = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
        self.log = None

    def _get_log_path(self, name):
        try:
            log_config = configparser.RawConfigParser()
            log_config.read(f"{self.root_path}/cfg/rsyslog_csbs.conf")
            if log_config.has_option("LOG_FILE", name):
                return log_config.get("LOG_FILE", name)
            return "/var/log/huawei/dj/install.log"
        except Exception:
            return "/var/log/huawei/dj/install.log"

    def _get_logger(self, name):
        log_file = self._get_log_path(name)
        log = logging.getLogger(name)
        log_fmt = "%(asctime)s.%(msecs)03d %(levelname)s %(name)s [pid:%(process)d] " \
                  "[%(filename)s:%(lineno)d %(funcName)s] %(message)s"
        log_hdlr = WatchedFileHandler(filename=log_file)
        log_hdlr.setFormatter(logging.Formatter(log_fmt, datefmt="%Y-%m-%d %H:%M:%S"))
        log.setLevel(logging.INFO)
        log.addHandler(log_hdlr)
        return log

    def _print_and_judge_progress(self, progress, is_success, msg):
        cur_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(time.time()))
        if is_success:
            length = 100 - len(msg) - len(cur_time) - 5
        else:
            length = 100 - len(msg) - len(cur_time) - 6
        result = "..".center(length, ".")
        level = "INFO" if is_success else "ERROR"
        out_msg = f"{cur_time} [{level}] {msg} {result} {progress:3}%"
        print(out_msg)
        if self.log:
            self.log.info(msg) if is_success else self.log.error(msg)

    def _check_result(self, result, msg):
        if not result:
            return
        self._print_and_judge_progress(100, False, msg)
        exit(1)

    def _is_file_exist(self, file_path):
        if os.path.exists(file_path):
            return True
        if self.log:
            self.log.warning("The file %s is not exist" % file_path)
        return False

    @staticmethod
    def _get_progress(service, step):
        service_list = list(SERVICE_LIST.keys())
        num = len(service_list)
        number = service_list.index(service) + 1
        if step in ["uninstall"]:
            number = num - number
        if step in ["install", "uninstall"]:
            return int(number * 90 / num) if num else 0 + 9
        return 100


class ServiceInstall(ServiceManager):
    def __init__(self, config_file):
        super(ServiceInstall, self).__init__()
        self.conf = config_file
        self.log = self._get_logger("csbs_install")

    def _get_from_conf(self, section_name, key):
        conf = configparser.RawConfigParser()
        conf.read(self.conf)
        if conf.has_option(section_name, key):
            return conf.get(section_name, key)

    def _put_to_conf(self, section_name, key, value):
        conf_file = "/opt/huawei/dj/cfg/sys.ini"
        conf = configparser.RawConfigParser()
        conf.read(conf_file if os.path.isfile(conf_file) else self.conf)
        conf.set(section_name, key, value)
        cmd_ret = safe_command.run_command(["mkdir", "-p", os.path.dirname(conf_file)])
        self._check_result(cmd_ret.exist_code, "Create config dir failed.")
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        with os.fdopen(os.open(conf_file, flags, modes), 'w') as _conf_file:
            conf.write(_conf_file)

    def _conversion_cfg(self):
        conf = configparser.RawConfigParser()
        conf.read(self.conf)
        if conf.getint("FEATURE", "configuration_conversion_flag", fallback=0) == 0:
            return
        gaussdb_admin_pwd = conf.get("SYSTEM", "gaussdb_admin_pwd", fallback=None)
        if "gaussdb_password" not in conf.options("FEATURE"):
            conf.set("FEATURE", "gaussdb_password", gaussdb_admin_pwd)
        conf.remove_option("SYSTEM", "gaussdb_admin_pwd")
        conf.remove_option("FEATURE", "configuration_conversion_flag")
        node_list = conf.get("SYSTEM", "manage_ip_list").split(",")
        name_list = conf.get("SYSTEM", "hostname_list").split(",")
        for node_index in range(len(node_list)):
            node = node_list[node_index]
            if node not in conf.sections():
                conf.add_section(node)
            conf.set(node, "hostname", name_list[node_index])

        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        with os.fdopen(os.open(self.conf, flags, modes), 'w') as _conf_file:
            conf.write(_conf_file)

    def _get_node_info(self):
        conf = configparser.RawConfigParser()
        conf.read(self.conf)
        ip_version = conf.get("SYSTEM", "ip_version")
        net = netifaces.AF_INET6 if ip_version == "ipv6" else netifaces.AF_INET
        interface = netifaces.gateways()['default'][net][1]
        self.node_ip = netifaces.ifaddresses(interface)[net][0]['addr']
        section = conf.sections()
        section.remove("SYSTEM")
        section.remove("FEATURE")
        self.node_list = section
        self.node_index = self.node_list.index(self.node_ip) + 1
        self.node = len(self.node_list)
        self.profile = conf.get("SYSTEM", "profile")

    def _get_deploy_nodes(self, service):
        if self.node == 1:
            return ["1"]
        deploy_nodes = SERVICE_LIST.get(service).get("nodes")
        if not deploy_nodes:
            return ["1", "2"] if self.profile == "private_recover" else ["1", "2", "3"]
        return deploy_nodes.split(',')

    def _whether_install(self, service):
        service_node = self._get_from_conf("SYSTEM", f"{service}_nodes")
        if service_node and self.node_ip not in service_node.split(","):
            return False
        if self.node == 1:
            return True
        deploy_nodes_list = self._get_deploy_nodes(service)
        self._save_deploy_ips(service, deploy_nodes_list)
        if str(self.node_index) not in deploy_nodes_list:
            return False
        return True

    def _save_deploy_ips(self, service, deploy_nodes_list):
        deploy_ip_list = list()
        for node in deploy_nodes_list:
            deploy_ip_list.append(self.node_list[int(node) - 1])
        self._put_to_conf('SYSTEM', f"{service}_nodes", ','.join(deploy_ip_list))

    def _check_service_float_ip(self, service):
        if service not in ["gaussdb", "haproxy"]:
            self.log.info(f"Service {service} not ha module")
            return True
        deploy_nodes_list = self._get_deploy_nodes(service)
        if not deploy_nodes_list or len(deploy_nodes_list) == 1:
            self.log.info(f"{service} not deployNodes, assign float ip failed.")
            return True
        float_ip = self._get_from_conf("SYSTEM", f"{service}_float_ip")
        if not float_ip or float_ip == "None":
            self.log.error(f"{service}_float_ip not config")
            return False
        self.log.info(f"{service} assign float ip successfully")
        return True

    def _check_service(self, service):
        self.log.info(f"===begin to check {service}===")
        cmd_ret = safe_command.run_command(['netstat', '-tulpn'])
        self._check_result(cmd_ret.exist_code, "Check port failed")
        if self.node != 1 and not self._check_service_float_ip(service):
            self._check_result(1, f"Check float ip of {service} failed.")
        if cmd_ret.std_out and not self._check_service_port(service, cmd_ret.std_out):
            self._check_result(1, f"Check port of {service} failed")

    @staticmethod
    def _check_service_port(service, port_inuse):
        ports_str = SERVICE_LIST.get(service).get("hostPorts")
        if not ports_str:
            return True
        comp_port_list = ports_str
        if isinstance(comp_port_list, str):
            comp_port_list = comp_port_list.split(",")
        for port in comp_port_list:
            find_string = f".*:{port} "
            if re.search(find_string, port_inuse):
                return False
        return True

    def _merge_service(self, service):
        self.log.info(f"===begin to merge {service}===")
        merge = os.path.join(f"{self.root_path}/services", service, "merge.sh")
        if not self._is_file_exist(merge):
            self.log.info(f"The merge of {service} not exsit.")
            return 0
        cmd_ret = safe_command.run_command(['sh', merge], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Merge {service} failed")
        self.log.info(f"Merge {service} successfully.")
        return 0

    def _post_start_service(self, service):
        self.log.info(f"===begin to post_start {service}===")
        post_s = os.path.join(f"{self.root_path}/services", service, "post_start.sh")
        if not self._is_file_exist(post_s):
            self.log.info(f"The Post_start of {service} not exsit.")
            return 0
        cmd_ret = safe_command.run_command(['sh', post_s], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Post_start {service} failed")
        self.log.info(f"Post_start {service} successfully.")
        return 0

    def _install_service(self, service):
        self.log.info(f"===begin to install {service}===")
        install = os.path.join(f"{self.root_path}/services", service, "install.sh")
        if not self._is_file_exist(install):
            self._check_result(1, f"The install file of {service} not exsit.")
        cmd_ret = safe_command.run_command(['sh', install], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Install {service} failed")
        self.log.info(f"Install {service} successfully.")
        return 0

    def _config_service(self, service):
        self.log.info(f"===begin to config {service}===")
        config = os.path.join(f"{self.root_path}/services", service, "config.sh")
        if not self._is_file_exist(config):
            self.log.info(f"The config of {service} not exsit")
            return 0
        cmd_ret = safe_command.run_command(['sh', config], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Config {service} failed.")
        self.log.info(f"Config {service} successfully. ")
        return 0

    def _start_service(self, service):
        self.log.info(f"===begin to start {service}===")
        start = os.path.join(f"{self.root_path}/services", service, "start.sh")
        if not self._is_file_exist(start):
            self.log.info(f"The start of {service} not exsit.")
            return 0
        cmd_ret = safe_command.run_command(['sh', start], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Start {service} failed")
        self.log.info(f"Start {service} successfully.")
        return 0

    def _deploy_service(self, service):
        if not self._whether_install(service):
            self.log.info(f"{service} not need install on this node")
            return
        self._check_service(service)
        self._merge_service(service)
        self._install_service(service)
        self._config_service(service)
        self._start_service(service)
        self._post_start_service(service)
        progress = self._get_progress(service, "install")
        self._print_and_judge_progress(progress, True, f"Successfully Install {service}")

    def _pre_install(self):
        self._conversion_cfg()
        check_conf = f"{self.root_path}/bin/pre_check/check_sys_ini.sh"
        cmd_ret = safe_command.run_command(['sh', check_conf, self.conf], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Check sys ini failed.")
        check_os = f"{self.root_path}/bin/pre_check/pre_check.sh"
        cmd_ret = safe_command.run_command(['sh', check_os], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Check os configuration failed.")
        self._print_and_judge_progress(8, True, f"Successfully Check configuration")
        self._get_node_info()

    def _post_install(self):
        security_harden = f"{self.root_path}/install_scripts/security_harden.sh"
        cmd_ret = safe_command.run_command(['bash', security_harden], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Security hardening failed.")

    def _check_software_exist(self):
        path = "/opt/huawei/dj/cfg/sys.ini"
        ret = 0 if not os.path.exists(path) else 1
        self._check_result(ret, "Karbor software has already installed")
        self.log.info("Karbor software has not installed, check OK.")

    def deploy(self):
        self._check_software_exist()
        service_list = list(SERVICE_LIST.keys())
        self.log.info(f"The deploy service list: {service_list}")
        self._print_and_judge_progress(0, True, "Successfully Begin Install Karbor.")
        self._pre_install()
        for service in service_list:
            self._deploy_service(service)
        self._post_install()
        self._print_and_judge_progress(100, True, "Successfully Finish Install Karbor.")


class ServiceUninstall(ServiceManager):
    def __init__(self):
        super(ServiceUninstall, self).__init__()

    def _get_uninstall_dir(self):
        if os.path.isdir(f"{self.root_path}/services"):
            return f"{self.root_path}/services"
        return self.root_path

    def _uninstall_service(self, service):
        uninstall_dir = self._get_uninstall_dir()
        uninstall = os.path.join(uninstall_dir, service, "uninstall.sh")
        if not self._is_file_exist(uninstall):
            return 0
        cmd_ret = safe_command.run_command(['sh', uninstall], timeout=1800)
        self._check_result(cmd_ret.exist_code, f"Uninstall {service} failed")
        return 0

    def _remove_service(self, service):
        self._uninstall_service(service)
        progress = self._get_progress(service, "uninstall")
        self._print_and_judge_progress(progress, True, f"Successfully Uninstall {service}")

    def remove(self):
        is_unistall = input('Are you sure to uninstall Karbor (y/n):')
        if is_unistall != 'y':
            sys.exit(0)
        print("\n")
        service_list = list(SERVICE_LIST.keys())
        self._print_and_judge_progress(0, True, "Successfully Begin Uninstall Karbor.")
        while service_list:
            self._remove_service(service_list.pop())
        self._print_and_judge_progress(100, True, "Successfully Finish Uninstall Karbor.")


if __name__ == '__main__':
    if len(sys.argv) == 3 and sys.argv[1] == "install" and os.path.isfile(sys.argv[2]):
        srv_man = ServiceInstall(sys.argv[2])
        srv_man.deploy()
    elif len(sys.argv) == 2 and sys.argv[1] == "uninstall":
        srv_man = ServiceUninstall()
        srv_man.remove()
    else:
        print("First param must be in ['install', 'uninstall'], When install Second param must be sys_ini")
        sys.exit(1)
    sys.exit(0)
